<?php // https://divibooster.com/customize-divi-gallery-module-pagination-with-custom-text-and-icons/

namespace DiviBooster\GalleryBooster\GridPaginationNav;

// === Constants ===
const DIVI4_MODULE_SLUG = 'et_pb_gallery';
const DIVI5_MODULE_SLUG = 'divi/gallery';

// Divi 4 field slugs (snake_case)
const D4_DISPLAY_MODE = 'dbdb_pagination_nav_display';
const D4_PREV_TEXT    = 'dbdb_pagination_prev_text';
const D4_NEXT_TEXT    = 'dbdb_pagination_next_text';
const D4_PREV_ICON    = 'dbdb_pagination_prev_icon';
const D4_NEXT_ICON    = 'dbdb_pagination_next_icon';

// Divi 5 attribute slugs (camelCase)
const D5_DISPLAY_MODE = 'diviboosterPaginationNavDisplay';
const D5_PREV_TEXT    = 'diviboosterPaginationPrevText';
const D5_NEXT_TEXT    = 'diviboosterPaginationNextText';
const D5_PREV_ICON    = 'diviboosterPaginationPrevIcon';
const D5_NEXT_ICON    = 'diviboosterPaginationNextIcon';

// === Utilities ===
function is_grid_pagination_enabled_d4($props) {
    // Returns true if grid layout and pagination enabled
    return isset($props['fullwidth']) && $props['fullwidth'] === 'off' && isset($props['show_pagination']) && $props['show_pagination'] === 'on';
}
function is_grid_pagination_enabled_d5($attrs) {
    // Returns true if grid layout and pagination enabled
    $fullwidth = $attrs['module']['advanced']['fullwidth']['desktop']['value'] ?? 'off';
    $show_pagination = $attrs['pagination']['advanced']['showPagination']['desktop']['value'] ?? 'on';
    return ($fullwidth === 'off') && ($show_pagination === 'on');
}
function extract_order_class($html) {
    // Extracts the gallery module order class from HTML
    if (!is_string($html)) return '';
    if (preg_match('/class="([^"]+)"/i', $html, $m)) {
        foreach (preg_split('/\s+/', $m[1], -1, PREG_SPLIT_NO_EMPTY) as $cls) {
            if (strpos($cls, 'et_pb_gallery_') === 0) return $cls;
        }
    }
    return '';
}
function css_unicode_to_entity($css_unicode) {
    // Converts '\e046' or '\E046' to '&#xe046;'
    if (!is_string($css_unicode) || $css_unicode === '') return '';
    if (preg_match('/\\([0-9a-fA-F]{3,6})/', $css_unicode, $m)) {
        return '&#x' . strtolower($m[1]) . ';';
    }
    // If it's already an entity like '&#xe046;' or '&#xE046;', normalize case
    if (preg_match('/&#x([0-9a-fA-F]+);/', $css_unicode, $m)) {
        return '&#x' . strtolower($m[1]) . ';';
    }
    return '';
}
function normalize_icon_from_d4($value) {
    // Converts Divi 4 icon value to array with unicode and fontFamily
    if (!is_string($value) || $value === '') return false;
    $parts = explode('||', $value);
    $entity = trim($parts[0]);
    if (!$entity) return false;
    $font_family = 'ETmodules';
    return [
        'unicode'    => $entity,
        'fontFamily' => $font_family,
    ];
}
function normalize_icon_from_d5($attrs_value) {
    // Converts Divi 5 icon value to array with unicode and fontFamily
    if (!is_array($attrs_value)) return false;
    $unicode = $attrs_value['innerContent']['desktop']['value']['unicode'] ?? '';
    if (!$unicode) return false;
    // Normalize entity casing
    if (preg_match('/&#x([0-9a-fA-F]+);/', $unicode, $m)) {
        $unicode = '&#x' . strtolower($m[1]) . ';';
    }
    return [
        'unicode'    => $unicode,
        'fontFamily' => 'ETmodules',
    ];
}
function build_config_from_d4($props) {
    $mode = isset($props[D4_DISPLAY_MODE]) ? trim((string)$props[D4_DISPLAY_MODE]) : 'default';
    $prev_text = isset($props[D4_PREV_TEXT]) ? (string)$props[D4_PREV_TEXT] : '';
    $next_text = isset($props[D4_NEXT_TEXT]) ? (string)$props[D4_NEXT_TEXT] : '';
    $prev_icon = isset($props[D4_PREV_ICON]) ? normalize_icon_from_d4($props[D4_PREV_ICON]) : false;
    $next_icon = isset($props[D4_NEXT_ICON]) ? normalize_icon_from_d4($props[D4_NEXT_ICON]) : false;
    return [
        'mode'     => in_array($mode, ['default','text','icon','icon_text'], true) ? $mode : 'default',
        'prevText' => $prev_text,
        'nextText' => $next_text,
        'prevIcon' => $prev_icon,
        'nextIcon' => $next_icon,
    ];
}
function build_config_from_d5($attrs) {
    $mode = $attrs[D5_DISPLAY_MODE]['desktop']['value'] ?? ($attrs[D5_DISPLAY_MODE]['value'] ?? 'default');
    $prev_text = $attrs[D5_PREV_TEXT]['innerContent']['desktop']['value'] ?? ($attrs[D5_PREV_TEXT]['value'] ?? '');
    $next_text = $attrs[D5_NEXT_TEXT]['innerContent']['desktop']['value'] ?? ($attrs[D5_NEXT_TEXT]['value'] ?? '');
    $prev_icon = normalize_icon_from_d5($attrs[D5_PREV_ICON] ?? []);
    $next_icon = normalize_icon_from_d5($attrs[D5_NEXT_ICON] ?? []);
    return [
        'mode'     => in_array($mode, ['default','text','icon','icon_text'], true) ? $mode : 'default',
        'prevText' => is_string($prev_text) ? $prev_text : '',
        'nextText' => is_string($next_text) ? $next_text : '',
        'prevIcon' => $prev_icon,
        'nextIcon' => $next_icon,
    ];
}
function inject_config_into_pagination_container($html, $config, $order_class) {
    if (!is_string($html) || $html === '') return $html;
    $mode = $config['mode'] ?? 'default';
    if ($mode === 'default') return $html;
    $prevIcon   = $config['prevIcon'] ?? false;
    $nextIcon   = $config['nextIcon'] ?? false;
    $prevText = $config['prevText'] ?? '';
    $nextText = $config['nextText'] ?? '';
    // Write custom text and display mode to data attributes on the gallery container
    if ($prevText !== '') {
        $html = preg_replace('/(<div[^>]*class="[^"]*et_pb_gallery[^"]*"[^>]*)/i', '$1 data-prev-text="' . esc_attr($prevText) . '"', $html, 1);
    }
    if ($nextText !== '') {
        $html = preg_replace('/(<div[^>]*class="[^"]*et_pb_gallery[^"]*"[^>]*)/i', '$1 data-next-text="' . esc_attr($nextText) . '"', $html, 1);
    }
    // Always set data-display-mode
    $html = preg_replace('/(<div[^>]*class="[^"]*et_pb_gallery[^"]*"[^>]*)/i', '$1 data-display-mode="' . esc_attr($mode) . '"', $html, 1);
    $prefix     = $order_class ? '.' . $order_class . ' ' : '';

    // Converts HTML entity (e.g. &#xe046;) to CSS unicode (e.g. \e046)
    $entity_to_css_unicode = function($entity) {
        if (preg_match('/&#x([0-9a-fA-F]+);/', $entity, $m)) return '\\' . strtolower($m[1]);
        if (preg_match('/\\([0-9a-fA-F]{3,6})/', $entity, $m)) return '\\' . strtolower($m[1]);
        if (strlen($entity) === 1) return $entity;
        return '';
    };

    $css = $prefix . '.et_pb_gallery_pagination li.prev, ' . $prefix . '.et_pb_gallery_pagination li.next { color: #999; }';

    // Helper for building pseudo selectors
    $pseudo = function($selector, $pseudo, $props) use ($prefix) {
        $css = $prefix . $selector . $pseudo . ' { ';
        foreach ($props as $k => $v) $css .= $k . ': ' . $v . '; ';
        return $css . '}';
    };

    // Only handle custom icons; skip custom prev/next text and related styles
    if ($mode === 'icon' || $mode === 'icon_text') {
        if ($prevIcon && !empty($prevIcon['unicode'])) {
            $iconContent = $entity_to_css_unicode($prevIcon['unicode']);
            $fontFamily  = $prevIcon['fontFamily'] ?? 'ETmodules';
            $css .= $pseudo('.et_pb_gallery_pagination a.page-prev', '::before', [
                'content'     => '"' . $iconContent . '"',
                'font-family' => addslashes($fontFamily),
            ]);
            if ($mode === 'icon_text') {
                $css .= $prefix . '.et_pb_gallery_pagination a.page-prev { position: relative; }';
                $css .= $prefix . '.et_pb_gallery_pagination a.page-prev::before { visibility: visible; position: static; color: inherit !important; vertical-align: baseline; margin-right: 0.2em; }';
            }
        }
        if ($nextIcon && !empty($nextIcon['unicode'])) {
            $iconContent = $entity_to_css_unicode($nextIcon['unicode']);
            $fontFamily  = $nextIcon['fontFamily'] ?? 'ETmodules';
            if ($mode === 'icon_text') {
                $css .= $prefix . '.et_pb_gallery_pagination a.page-next { position: relative; }';
                $css .= $prefix . '.et_pb_gallery_pagination a.page-next::after { visibility: visible; position: static; color: inherit !important; vertical-align: baseline; margin-left: 0.2em; }';
            }
            $css .= $pseudo('.et_pb_gallery_pagination a.page-next', '::after', [
                'content'     => '"' . $iconContent . '"',
                'font-family' => addslashes($fontFamily),
            ]);
        }
    }

    if ($css) {
        add_action('wp_footer', function() use ($css) {
            echo '<style>' . $css . '</style>';
        });
    }
    return $html;
}

// === Divi 4: Add Fields ===
add_filter('et_pb_all_fields_unprocessed_' . DIVI4_MODULE_SLUG, __NAMESPACE__ . '\d4_add_fields');
function d4_add_fields($fields) {
    if (!is_array($fields)) return $fields;
    // Display mode selector
    $fields[D4_DISPLAY_MODE] = array(
        'label'           => esc_html__('Pagination Prev/Next Display', 'divi-booster'),
        'type'            => 'select',
        'options'         => array(
            'default'   => esc_html__('Default', 'divi-booster'),
            'text'      => esc_html__('Text', 'divi-booster'),
            'icon'      => esc_html__('Icon', 'divi-booster'),
            'icon_text' => esc_html__('Icon + Text', 'divi-booster'),
        ),
        'default'         => 'default',
        'option_category' => 'configuration',
        'tab_slug'        => 'general',
        'toggle_slug'     => 'elements',
        'description'     => esc_html__('Choose how to display the Previous/Next pagination links for the gallery grid.', 'divi-booster'),
        'show_if'         => array(
            'fullwidth'       => 'off',
            'show_pagination' => 'on',
        ),
    );
    // Prev / Next Text
    $fields[D4_PREV_TEXT] = array(
        'label'           => esc_html__('Previous Link Text', 'divi-booster'),
        'type'            => 'text',
        'option_category' => 'basic_option',
        'tab_slug'        => 'general',
        'toggle_slug'     => 'elements',
        'description'     => esc_html__('Custom text for the Previous pagination link. Leave empty to use the default localized label.', 'divi-booster'),
        'show_if'         => array(
            'fullwidth'       => 'off',
            'show_pagination' => 'on',
            D4_DISPLAY_MODE => array('text','icon_text')
        ),
    );
    $fields[D4_NEXT_TEXT] = array(
        'label'           => esc_html__('Next Link Text', 'divi-booster'),
        'type'            => 'text',
        'option_category' => 'basic_option',
        'tab_slug'        => 'general',
        'toggle_slug'     => 'elements',
        'description'     => esc_html__('Custom text for the Next pagination link. Leave empty to use the default localized label.', 'divi-booster'),
            'show_if'         => array(
                'fullwidth'       => 'off',
                'show_pagination' => 'on',
                D4_DISPLAY_MODE => array('text','icon_text')
            ),
    );
    // Prev / Next Icons
    $fields[D4_PREV_ICON] = array(
        'label'           => esc_html__('Previous Link Icon', 'divi-booster'),
        'type'            => 'select_icon',
        'class'           => array('et-pb-font-icon'),
        'option_category' => 'basic_option',
        'tab_slug'        => 'general',
        'toggle_slug'     => 'elements',
        'description'     => esc_html__('Choose an icon to display for the Previous pagination link.', 'divi-booster'),
            'show_if'         => array(
                'fullwidth'       => 'off',
                'show_pagination' => 'on',
                D4_DISPLAY_MODE => array('icon','icon_text')
            ),
        'mobile_options'  => false,
        'hover'           => 'tabs',
        'sticky'          => true,
    );
    $fields[D4_NEXT_ICON] = array(
        'label'           => esc_html__('Next Link Icon', 'divi-booster'),
        'type'            => 'select_icon',
        'class'           => array('et-pb-font-icon'),
        'option_category' => 'basic_option',
        'tab_slug'        => 'general',
        'toggle_slug'     => 'elements',
        'description'     => esc_html__('Choose an icon to display for the Next pagination link.', 'divi-booster'),
            'show_if'         => array(
                'fullwidth'       => 'off',
                'show_pagination' => 'on',
                D4_DISPLAY_MODE => array('icon','icon_text')
            ),
        'mobile_options'  => false,
        'hover'           => 'tabs',
        'sticky'          => true,
    );
    return $fields;
}

// === Divi 5: Attribute Registration (PHP) ===
add_filter('divi_module_library_register_module_attrs', __NAMESPACE__ . '\\d5_register_attrs', 10, 2);
function d5_register_attrs($attrs, $args) {
    if (($args['name'] ?? '') !== DIVI5_MODULE_SLUG) return $attrs;
    $attrs[D5_DISPLAY_MODE] = get_d5_display_attr_definition();
    $attrs[D5_PREV_TEXT]    = get_d5_text_attr_definition(D5_PREV_TEXT, __('Previous Link Text', 'divi-booster'));
    $attrs[D5_NEXT_TEXT]    = get_d5_text_attr_definition(D5_NEXT_TEXT, __('Next Link Text', 'divi-booster'));
    $attrs[D5_PREV_ICON]    = get_d5_icon_attr_definition(D5_PREV_ICON, __('Previous Link Icon', 'divi-booster'));
    $attrs[D5_NEXT_ICON]    = get_d5_icon_attr_definition(D5_NEXT_ICON, __('Next Link Icon', 'divi-booster'));
    return $attrs;
}


function get_d5_display_attr_definition() {
    return [
        'type' => 'object',
        'settings' => [
            'innerContent' => [
                'groupType' => 'group-items',
                'items' => [
                    'displayMode' => [
                        'groupSlug'   => 'contentElements',
                        'attrName'    => D5_DISPLAY_MODE,
                        'label'       => __('Pagination Prev/Next Display', 'divi-booster'),
                        'description' => __('Choose how to display the Previous/Next pagination links for the gallery grid.', 'divi-booster'),
                        'features'    => [ 'hover' => false, 'sticky' => false, 'responsive' => false, 'preset' => 'elements' ],
                        'render'      => true,
                        'priority'    => 60,
                        'component'   => [ 'type' => 'field', 'name' => 'divi/select', 'props' => [ 'options' => [
                            'default'   => [ 'label' => __('Default', 'divi-booster') ],
                            'text'      => [ 'label' => __('Text', 'divi-booster') ],
                            'icon'      => [ 'label' => __('Icon', 'divi-booster') ],
                            'icon_text' => [ 'label' => __('Icon + Text', 'divi-booster') ],
                        ]]],
                        'defaultAttr' => [ 'desktop' => [ 'value' => 'default' ] ],
                        // 'visible' assigned in VB inline JS
                    ],
                ],
            ],
        ],
    ];
}
function get_d5_text_attr_definition($slug, $label) {
    return [
        'type' => 'object',
        'settings' => [
            'innerContent' => [
                'groupType' => 'group-items',
                'items' => [
                    $slug => [
                        'groupSlug'   => 'contentElements',
                        'attrName'    => $slug . '.innerContent',
                        'label'       => __($label, 'divi-booster'),
                        'description' => __('Leave empty to use the default localized label.', 'divi-booster'),
                        'features'    => [ 'hover' => false, 'sticky' => false, 'responsive' => false, 'preset' => 'elements' ],
                        'render'      => true,
                        'priority'    => 61,
                        'component'   => [ 'type' => 'field', 'name' => 'divi/text' ],
                        // 'visible' assigned in VB inline JS
                    ],
                ],
            ],
        ],
    ];
}
function get_d5_icon_attr_definition($slug, $label) {
    return [
        'type' => 'object',
        'settings' => [
            'innerContent' => [
                'groupType' => 'group-items',
                'items' => [
                    $slug => [
                        'groupSlug'   => 'contentElements',
                        'attrName'    => $slug . '.innerContent',
                        'label'       => __($label, 'divi-booster'),
                        'description' => __('Choose an icon to display for the pagination link.', 'divi-booster'),
                        'features'    => [ 'hover' => false ],
                        'render'      => true,
                        'priority'    => 62,
                        'component'   => [ 'type' => 'field', 'name' => 'divi/icon-picker' ],
                        // 'visible' assigned in VB inline JS
                    ],
                ],
            ],
        ],
    ];
}

// === Divi 5: VB Attribute Registration & Conversion Mapping (JS) ===
add_action('divi_visual_builder_assets_before_enqueue_scripts', __NAMESPACE__ . '\\enqueue_d5_vb_script');
function enqueue_d5_vb_script() {
    $handle = sanitize_title('divibooster-d5-gallery-grid-pagination-nav');
    wp_register_script($handle, '', ['lodash', 'divi-vendor-wp-hooks'], null, true);
    wp_enqueue_script($handle);
    wp_add_inline_script($handle, get_d5_vb_inline_js());
}
function get_d5_vb_inline_js() {
    $d4slug = wp_json_encode(DIVI4_MODULE_SLUG);
    $defs = [
        D5_DISPLAY_MODE => get_d5_display_attr_definition(),
        D5_PREV_TEXT    => get_d5_text_attr_definition(D5_PREV_TEXT, __('Previous Link Text', 'divi-booster')),
        D5_NEXT_TEXT    => get_d5_text_attr_definition(D5_NEXT_TEXT, __('Next Link Text', 'divi-booster')),
        D5_PREV_ICON    => get_d5_icon_attr_definition(D5_PREV_ICON, __('Previous Link Icon', 'divi-booster')),
        D5_NEXT_ICON    => get_d5_icon_attr_definition(D5_NEXT_ICON, __('Next Link Icon', 'divi-booster')),
    ];
    $defs_json = [];
    foreach ($defs as $k => $v) { $defs_json[$k] = $v; }
    $defs_json = wp_json_encode($defs_json);

    $display_key = wp_json_encode(D5_DISPLAY_MODE);
    $prev_text_key = wp_json_encode(D5_PREV_TEXT);
    $next_text_key = wp_json_encode(D5_NEXT_TEXT);
    $prev_icon_key = wp_json_encode(D5_PREV_ICON);
    $next_icon_key = wp_json_encode(D5_NEXT_ICON);

    $d5_module_name = wp_json_encode(DIVI5_MODULE_SLUG);

    return <<<JS
(function(){
  const { addFilter } = window.vendor?.wp?.hooks || {};
  if (!addFilter) return;

// Visibility helpers
const visibleInGridWithPagination = ({ attrs }) => {
    const fullwidth = attrs?.module?.advanced?.fullwidth?.desktop?.value ?? "off";
    const showPagination = attrs?.pagination?.advanced?.showPagination?.desktop?.value ?? "on";
    return fullwidth === "off" && showPagination === "on";
};
const getDisplayMode = attrs => {
    return attrs?.diviboosterPaginationNavDisplay?.desktop?.value
        ?? attrs?.diviboosterPaginationNavDisplay?.value
        ?? "default";
};
const visiblePrevText = ({ attrs }) =>
    visibleInGridWithPagination({ attrs }) &&
    ["text", "icon_text"].includes(getDisplayMode(attrs));
const visibleNextText = ({ attrs }) =>
    visibleInGridWithPagination({ attrs }) &&
    ["text", "icon_text"].includes(getDisplayMode(attrs));
const visiblePrevIcon = ({ attrs }) =>
    visibleInGridWithPagination({ attrs }) &&
    ["icon", "icon_text"].includes(getDisplayMode(attrs));
const visibleNextIcon = ({ attrs }) =>
    visibleInGridWithPagination({ attrs }) &&
    ["icon", "icon_text"].includes(getDisplayMode(attrs));

// Register attributes
addFilter('divi.moduleLibrary.moduleAttributes.divi.gallery', 'divi', (attributes, metadata) => {
    const defs = {$defs_json};

    // Attach JS visible callbacks
    try {
        if (defs[{$display_key}]?.settings?.innerContent?.items?.displayMode) {
            defs[{$display_key}].settings.innerContent.items.displayMode.visible = visibleInGridWithPagination;
        }
        if (defs[{$prev_text_key}]?.settings?.innerContent?.items?.[{$prev_text_key}]) {
            defs[{$prev_text_key}].settings.innerContent.items[{$prev_text_key}].visible = visiblePrevText;
        }
        if (defs[{$next_text_key}]?.settings?.innerContent?.items?.[{$next_text_key}]) {
            defs[{$next_text_key}].settings.innerContent.items[{$next_text_key}].visible = visibleNextText;
        }
        if (defs[{$prev_icon_key}]?.settings?.innerContent?.items?.[{$prev_icon_key}]) {
            defs[{$prev_icon_key}].settings.innerContent.items[{$prev_icon_key}].visible = visiblePrevIcon;
        }
        if (defs[{$next_icon_key}]?.settings?.innerContent?.items?.[{$next_icon_key}]) {
            defs[{$next_icon_key}].settings.innerContent.items[{$next_icon_key}].visible = visibleNextIcon;
        }
    } catch (e) { /* no-op */ }

    attributes[{$display_key}] = defs[{$display_key}];
    attributes[{$prev_text_key}] = defs[{$prev_text_key}];
    attributes[{$next_text_key}] = defs[{$next_text_key}];
    attributes[{$prev_icon_key}] = defs[{$prev_icon_key}];
    attributes[{$next_icon_key}] = defs[{$next_icon_key}];
    return attributes;
});
  // Conversion outline (D4 -> D5)
  addFilter('divi.moduleLibrary.conversion.moduleConversionOutline', 'divi', (outline, name) => {
    if (name !== {$d4slug}) return outline;
    outline.module = outline.module || {};
    outline.module[''.concat('{$d4slug}')];
    outline.module['dbdb_pagination_nav_display'] = {$display_key} + '.*';
    outline.module['dbdb_pagination_prev_text'] = {$prev_text_key} + '.innerContent.*';
    outline.module['dbdb_pagination_next_text'] = {$next_text_key} + '.innerContent.*';
    outline.module['dbdb_pagination_prev_icon'] = {$prev_icon_key} + '.innerContent.*';
    outline.module['dbdb_pagination_next_icon'] = {$next_icon_key} + '.innerContent.*';
    outline.valueExpansionFunctionMap = outline.valueExpansionFunctionMap || {};
    const { convertFontIcon } = window?.divi?.conversion || {};
    outline.valueExpansionFunctionMap['dbdb_pagination_prev_icon'] = convertFontIcon;
    outline.valueExpansionFunctionMap['dbdb_pagination_next_icon'] = convertFontIcon;
    return outline;
  });
})();
JS;
}

// === Divi 5: PHP conversion outline registration ===
add_filter('divi.moduleLibrary.conversion.moduleConversionOutline', __NAMESPACE__ . '\\d5_register_conversion_outline', 10, 2);
function d5_register_conversion_outline($conversion_outline, $module_name) {
    if ($module_name !== DIVI5_MODULE_SLUG) return $conversion_outline;
    if (!isset($conversion_outline['module']) || !is_array($conversion_outline['module'])) {
        $conversion_outline['module'] = array();
    }
    $conversion_outline['module'][D4_DISPLAY_MODE] = D5_DISPLAY_MODE . '.*';
    $conversion_outline['module'][D4_PREV_TEXT]    = D5_PREV_TEXT . '.innerContent.*';
    $conversion_outline['module'][D4_NEXT_TEXT]    = D5_NEXT_TEXT . '.innerContent.*';
    $conversion_outline['module'][D4_PREV_ICON]    = D5_PREV_ICON . '.innerContent.*';
    $conversion_outline['module'][D4_NEXT_ICON]    = D5_NEXT_ICON . '.innerContent.*';
    if (!isset($conversion_outline['valueExpansionFunctionMap']) || !is_array($conversion_outline['valueExpansionFunctionMap'])) {
        $conversion_outline['valueExpansionFunctionMap'] = array();
    }
    $conversion_outline['valueExpansionFunctionMap'][D4_PREV_ICON] = 'convertFontIcon';
    $conversion_outline['valueExpansionFunctionMap'][D4_NEXT_ICON] = 'convertFontIcon';
    return $conversion_outline;
}


// === Divi 4 Output Filter ===
add_filter('et_module_shortcode_output', __NAMESPACE__ . '\\d4_output_filter', 10, 3);
function d4_output_filter($output, $render_slug, $module) {
    if (!is_string($output) || $render_slug !== DIVI4_MODULE_SLUG || !isset($module->props) || !is_array($module->props)) {
        return $output;
    }
    $props = $module->props;
    if (!is_grid_pagination_enabled_d4($props)) return $output;

    $config = build_config_from_d4($props);
    if ($config['mode'] === 'default') return $output; // no changes required

    $order_class = extract_order_class($output);
    $output = inject_config_into_pagination_container($output, $config, $order_class);

    return $output;
}

// === Divi 5 Block Output Filter ===
add_filter('render_block_' . DIVI5_MODULE_SLUG, __NAMESPACE__ . '\\d5_output_filter', 10, 3);
function d5_output_filter($block_content, $parsed_block, $block) {
    $attrs = $parsed_block['attrs'] ?? array();
    if (!is_grid_pagination_enabled_d5($attrs)) return $block_content;

    $config = build_config_from_d5($attrs);
    if ($config['mode'] === 'default') return $block_content;

    $order_class = '';
    if (is_string($block_content) && preg_match('/class="([^"]+)"/i', $block_content, $m)) {
        $classes = preg_split('/\s+/', $m[1], -1, PREG_SPLIT_NO_EMPTY);
        foreach ($classes as $cls) { if (strpos($cls, 'et_pb_gallery_') === 0) { $order_class = $cls; break; } }
    }
    $block_content = inject_config_into_pagination_container($block_content, $config, $order_class);

    return $block_content;
}

// === Notes: VB Live Style hook (optional) – no-op since we render via JS ===
add_filter('divi_module_wrapper_render', function($module_wrapper, $args){
    // No server-side styles to add; JS handles runtime rendering per-instance.
    return $module_wrapper;
}, 10, 2);

add_action('wp_footer', function() {
    ?>
    <style>
    /* Hide li.prev-dots only when it is the second child and display mode is icon, text, or icon_text */
    .et_pb_gallery[data-display-mode="icon"] .et_pb_gallery_pagination li.prev-dots:nth-child(2),
    .et_pb_gallery[data-display-mode="text"] .et_pb_gallery_pagination li.prev-dots:nth-child(2),
    .et_pb_gallery[data-display-mode="icon_text"] .et_pb_gallery_pagination li.prev-dots:nth-child(2) {
        display: none !important;
    }
    /* Hide prev/next links until JS updates them */
    .et_pb_gallery.et_pb_gallery_grid .et_pb_gallery_pagination {
        visibility: hidden;
    }
    </style>
    <script>
    (function(){
        document.querySelectorAll('.et_pb_gallery.et_pb_gallery_grid').forEach(function(gallery){
            var pagination = gallery.querySelector('.et_pb_gallery_pagination');
            if (!pagination) return;
            var displayMode = gallery.getAttribute('data-display-mode') || 'default';
            var updateLinks = function(){
                setTimeout(function() {
                    var prev = pagination.querySelector('a.page-prev');
                    var next = pagination.querySelector('a.page-next');
                    if (displayMode === 'icon') {
                        if (prev) prev.textContent = '';
                        if (next) next.textContent = '';
                    } else {
                        if (prev) prev.textContent = gallery.getAttribute('data-prev-text') || 'Prev';
                        if (next) next.textContent = gallery.getAttribute('data-next-text') || 'Next';
                    }
                    // Show links after update
                    pagination.style.visibility = 'visible';
                }, 1000);
            };
            updateLinks();
            var observer = new MutationObserver(function(mutations){
                updateLinks();
                observer.disconnect();
            });
            observer.observe(pagination, { childList: true });
        });
    })();
    </script>
    <?php
});

// Created at 1755748131.
