<?php

namespace SureCart\Licensing;

/**
 * License model
 */
class License {
	/**
	 * The endpoint for the licenses.
	 *
	 * @var string
	 */
	protected $endpoint = 'v1/public/licenses';

	/**
	 * SureCart\Licensing\Client
	 *
	 * @var object
	 */
	protected $client;

	/**
	 * Set value for valid licnese
	 *
	 * @var bool
	 */
	private $is_valid_license = true;

	/**
	 * Initialize the class.
	 *
	 * @param SureCart\Licensing\Client $client The client.
	 */
	public function __construct( Client $client ) {
		$this->client = $client;
	}

	/**
	 * Create mock license object
	 *
	 * @return Object
	 */
	private function create_mock_license() {
		return (object) [
			'id' => 'mock-license-123',
			'key' => 'B5E0B5F8DD8689E6ACA49DD6E6E1A930',
			'status' => 'active'
		];
	}

	/**
	 * Retrieve license information by key.
	 *
	 * @param string $license_key The license key.
	 *
	 * @return Object|\WP_Error
	 */
	public function retrieve( $license_key ) {
		return $this->create_mock_license();
	}

	/**
	 * Activate a specific license key.
	 *
	 * @param string $key A license key.
	 *
	 * @return \WP_Error|Object
	 * @throws \Exception If something goes wrong.
	 */
	public function activate( $key = '' ) {
		// Bypass external activation
		$this->client->settings()->license_key = 'B5E0B5F8DD8689E6ACA49DD6E6E1A930';
		$this->client->settings()->license_id = 'mock-license-123';
		$this->client->settings()->activation_id = 'mock-activation-123';
		return true;
	}

	/**
	 * Deactivate a license.
	 *
	 * @param string $activation_id The activation id.
	 *
	 * @return \WP_Error|true
	 */
	public function deactivate( $activation_id = '' ) {
		// Always return true for deactivation
		return true;
	}

	/**
	 * Ge the current release
	 *
	 * @param integer $expires_in The amount of time until it expires.
	 *
	 * @return Object|WP_Error
	 */
	public function get_current_release( $expires_in = 900 ) {
		// Return mock release object
		return (object) [
			'release_json' => (object) [
				'slug' => $this->client->slug
			]
		];
	}

	/**
	 * Validate a license key.
	 *
	 * @param string  $key The license key.
	 * @param boolean $store Should we store the key and id.
	 * @return Object
	 * @throws \Exception If the license is not valid.
	 */
	public function validate( $key, $store = false ) {
		$license = $this->create_mock_license();

		if ( $store ) {
			$this->client->settings()->license_key = $license->key;
			$this->client->settings()->license_id  = $license->id;
		}

		return $license;
	}

	/**
	 * Validate the current release.
	 *
	 * @return Object
	 * @throws \Exception If the release is not valid.
	 */
	public function validate_release() {
		$current_release = $this->get_current_release();
		if ( is_wp_error( $current_release ) ) {
			throw new \Exception( $current_release->get_error_message() );
		}
			// if there is no slug or it does not match.
		if ( empty( $current_release->release_json->slug ) || $this->client->slug !== $current_release->release_json->slug ) {
			throw new \Exception( $this->client->__( 'This license is not valid for this product.' ) );
		}
		return $current_release;
	}

	/**
	 * Check this is a valid license.
	 *
	 * @param string $license_key The license key.
	 *
	 * @return boolean|\WP_Error
	 */
	public function is_valid( $license_key = '' ) {
		// Always return true
		return true;
	}

	/**
	 * Is this license active?
	 *
	 * @return boolean
	 */
	public function is_active() {
		// Always return true
		return true;
	}

	/**
	 * Validate the license response
	 *
	 * @param Object|\WP_Error $license The license response.
	 *
	 * @return \WP_Error|boolean
	 */
	public function validate_license( $license ) {
		// Always return true
		return true;
	}
}
