<?php
// SPDX-License-Identifier: GPL-2.0-or-later

GFForms::include_addon_framework();

class GFTooltips extends GFAddOn {

	protected $_version = GFTT_VERSION;
	protected $_min_gravityforms_version = '2.6';

	protected $_slug = GFTT_SLUG;
	protected $_path = 'gf-tooltips/gf-tooltips.php';
	protected $_full_path = __FILE__;
	protected $_title = GFTT_NAME;
	protected $_short_title = 'Tooltips';
	protected $_url = 'https://jetsloth.com/gravity-forms-tooltips';

	/**
	 * Members plugin integration
	 */
	protected $_capabilities = array( 'gravityforms_edit_forms', 'gravityforms_edit_settings' );

	/**
	 * Permissions
	 */
	protected $_capabilities_settings_page = 'gravityforms_edit_settings';
	protected $_capabilities_form_settings = 'gravityforms_edit_forms';
	protected $_capabilities_uninstall = 'gravityforms_uninstall';

	private static $_instance = null;


	/**
	 * Detect if this plugin is network-activated (Multisite).
	 */
	private function is_network_activated(): bool {
		if ( ! is_multisite() ) {
			return false;
		}
		$plugin = plugin_basename( $this->_full_path );
		$active = (array) get_site_option( 'active_sitewide_plugins', [] );
		return isset( $active[ $plugin ] );
	}

	/**
	 * Get an instance of this class.
	 *
	 * @return GFTooltips
	 */
	public static function get_instance() {
		if ( self::$_instance == null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	private function __clone() {
	} /* do nothing */

	/**
	 * Handles anything which requires early initialization.
	 */
	public function pre_init() {
		parent::pre_init();
	}

	/**
	 * Handles hooks and loading of language files.
	 */
	public function init() {

		$this->init_license();

		// inline css overrides. Run as late as possible
		add_action( 'gform_enqueue_scripts', array( $this, 'frontend_inline_styles' ), PHP_INT_MAX, 1 );

		add_filter( 'style_loader_tag', array( $this, 'configure_preload_assets' ), PHP_INT_MAX, 2 );

		add_filter( 'gform_field_content', array( $this, 'field_content' ), 100, 5 );

		add_filter( 'gform_field_choice_markup_pre_render', array( $this, 'add_choices_tooltips_markup' ), 200, 4 );

		add_filter( 'gform_footer_init_scripts_filter', array( $this, 'add_inline_options_tooltips_lookup' ), 10, 3 );

		add_filter( 'script_loader_tag', array( $this, 'set_scripts_to_defer' ), 10, 2 );

		parent::init();

	}

	/**
	 * Initialize the admin specific hooks.
	 */
	public function init_admin() {

		add_action( 'gform_editor_js', array( $this, 'editor_js' ) );

		add_action( 'gform_field_standard_settings', array( $this, 'tooltips_choices_field_settings' ), 10, 2 );

		if ( GFTT_GF_MIN_2_5 ) {
			add_filter( 'gform_field_settings_tabs', array( $this, 'custom_settings_tab' ), 10, 1 );
			add_action( 'gform_field_settings_tab_content_gftt', array( $this, 'custom_settings_markup' ), 10, 1 );
		}
		else {
			add_action( 'gform_field_standard_settings', array( $this, 'field_general_settings' ), 10, 2 );
		}

		add_filter( 'gform_tooltips', array( $this, 'field_tooltips' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_inline_styles' ) );

		$name = plugin_basename($this->_path);
		add_action( 'after_plugin_row_'.$name, array( $this, 'gf_plugin_row' ), 10, 2 );

		parent::init_admin();

	}


	public function get_app_menu_icon() {
		return $this->get_base_url() . '/images/icon-tooltips.svg';
	}

	public function get_menu_icon() {
		return $this->get_base_url() . '/images/icon-tooltips.svg';
	}


	// # SCRIPTS & STYLES -----------------------------------------------------------------------------------------------


	/**
	 * Return the scripts which should be enqueued.
	 *
	 * @return array
	 */
	public function scripts() {
		$js_deps = array( 'jquery' );
		if ( wp_is_mobile() ) {
			$js_deps[] = 'jquery-touch-punch';
		}

		$scripts = array(
			array(
				'handle'   => 'gf_tooltips',
				'src'      => $this->get_base_url() . '/lib/powertip.min.js',
				'version'  => '1.3.2',
				'deps'     => $js_deps,
				'enqueue'  => array(
					array( 'admin_page' => array( 'form_editor', 'form_settings', 'plugin_settings' ) ),
					array( $this, 'maybe_enqueue_gf_tooltips_powertip_js' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle'  => 'gf_tooltips_public',
				'src'     => $this->get_base_url() . '/js/gf_tooltip' . ( apply_filters('gftt_minified_assets', true) ? '.min' : '' ) . '.js',
				'version' => $this->_version,
				'deps'    => array_merge(array('gf_tooltips'), $js_deps),
				'strings' => array(
					'version' => $this->_version
				),
				'enqueue' => array(
					array( $this, 'maybe_enqueue_gf_tooltips_powertip_js' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle'  => 'gf_tooltips_admin',
				'src'     => $this->get_base_url() . '/js/gf_tooltip_admin' . ( apply_filters('gftt_minified_assets', true) ? '.min' : '' ) . '.js',
				'version' => $this->_version,
				'callback' => array( $this, 'localize_admin_scripts' ),
				'deps'    => array_merge(array('wp-color-picker'), $js_deps),
				'enqueue' => array(
					array( 'admin_page' => array('form_settings', 'plugin_settings', 'form_editor') )
				),
			),
		);

		return array_merge( parent::scripts(), $scripts );
	}


	public function set_scripts_to_defer( $tag, $handle ) {
		$handles = array('gf_tooltips', 'gf_tooltips_public');
		if ( !in_array($handle, $handles) ) {
			return $tag;
		}
		return str_replace( ' src', ' defer src', $tag ); // defer the script
	}


	public function localize_admin_scripts() {

		$params = array(
			'is_gf_min_2_5' => GFTT_GF_MIN_2_5,
			'gf_version' => GFCommon::$version,
		);
		wp_localize_script( 'gf_tooltips_admin', 'tooltipsVars', $params );

	}

	public function maybe_enqueue_gf_tooltips_powertip_js($form) {
		return ( !empty($form) && $this->form_has_tooltips($form) );
	}

	/**
	 * Inject Javascript into the Gravity form editor page.
	 */
	public function editor_js() {

		?>
        <script type='text/javascript'>
			(function($){

				let choicesFieldTypes = ['radio', 'checkbox', 'survey', 'poll', 'quiz', 'post_custom_field', 'product', 'option']
				let tooltipUnsupportedTypes = ['hidden', 'page', 'html', 'hidden']
				/* adding tooltip setting to all fields */
				$.each( window.fieldSettings, function(i, v){
					if ( tooltipUnsupportedTypes.indexOf(i) === -1 ) {
						window.fieldSettings[i] += ", .gftt-tooltip";
					}
					if ( choicesFieldTypes.indexOf(i) !== -1 ) {
						window.fieldSettings[i] += ", .gftt-choices-toggle";
					}
				});

			})(jQuery);
        </script>
		<?php

	}

	/**
	 * Return the stylesheets which should be enqueued.
	 *
	 * @return array
	 */
	public function styles() {

		$styles = array(
			array(
				'handle'  => 'preload_gf_tooltips_font_woff2',
				'src'     => $this->get_base_url() . '/font/gfflt.woff2',
				'version' => '1590114356623',
				'enqueue' => array(
					array( $this, 'maybe_preload_assets' )
					//array( $this, 'form_has_tooltips' )
				)
			),
			array(
				'handle'  => 'preload_gf_tooltips_font_woff',
				'src'     => $this->get_base_url() . '/font/gfflt.woff',
				'version' => '1590114356623',
				'enqueue' => array(
					//array( $this, 'maybe_preload_assets' )
					array( $this, 'form_has_tooltips' )
				)
			),
			array(
				'handle'  => 'preload_gf_tooltips_font_ttf',
				'src'     => $this->get_base_url() . '/font/gfflt.ttf',
				'version' => '1590114356623',
				'enqueue' => array(
					//array( $this, 'maybe_preload_assets' )
					array( $this, 'form_has_tooltips' )
				)
			),
			array(
				'handle'  => 'gf_tooltips_font',
				'src'     => $this->get_base_url() . '/css/gf_tooltips_font.css',
				'version' => $this->_version,
				'enqueue' => array(
					array( 'admin_page' => array( 'form_editor', 'form_settings', 'plugin_settings' ) ),
					array( $this, 'form_has_tooltips' ),// front-end, any form with tooltips
					//array( $this, 'maybe_enqueue_gf_tooltips_css' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle'  => 'gf_tooltips',
				'src'     => $this->get_base_url() . '/css/gf_tooltips' . ( apply_filters('gftt_minified_assets', true) ? '.min' : '' ) . '.css',
				'version' => $this->_version,
				'enqueue' => array(
					array( 'admin_page' => array( 'form_editor', 'form_settings', 'plugin_settings' ) ),
					array( $this, 'form_has_tooltips' ),// front-end, any form with tooltips
					//array( $this, 'maybe_enqueue_gf_tooltips_css' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle' => 'wp-color-picker',
				'enqueue' => array(
					array( 'admin_page' => array('form_settings', 'plugin_settings') )
				),
			),
		);

		return array_merge( parent::styles(), $styles );
	}

	public function maybe_preload_assets() {
		return true;
	}

	public function configure_preload_assets( $html, $handle ) {

		if ( $handle === 'preload_gf_tooltips_font_woff2' ) {
			return str_replace( ["rel='stylesheet'", " media='all'"], ["rel='prefetch' as='font' type='font/woff2' crossorigin='anonymous'", ""], $html );
		}

		if ( $handle === 'preload_gf_tooltips_font_woff' ) {
			return str_replace( ["rel='stylesheet'", " media='all'"], ["rel='prefetch' as='font' type='font/woff' crossorigin='anonymous'", ""], $html );
		}

		if ( $handle === 'preload_gf_tooltips_font_ttf' ) {
			return str_replace( ["rel='stylesheet'", " media='all'"], ["rel='prefetch' as='font' type='font/ttf' crossorigin='anonymous'", ""], $html );
		}

		if ( $handle === 'preload_gf_tooltips_font' ) {
			return str_replace( ["rel='stylesheet'", " media='all'"], ["rel='prefetch' as='style' type='text/css' crossorigin='anonymous'", ""], $html );
		}

		return $html;
	}

	public function maybe_enqueue_gf_tooltips_css($form) {
		return $this->form_has_tooltips($form);
	}

	public function admin_inline_styles($hook) {
		if ( $this->is_form_editor() ) {
			$form = $this->get_current_form();
			$this->add_inline_style($form);
		}
	}

	public function frontend_inline_styles( $form ) {
		$this->add_inline_style($form);
	}

	public function add_inline_style($form) {

		if ( empty($form) || !$this->form_has_tooltips($form) ) {
			return;
		}

		$global_settings = $this->get_plugin_settings();
		if ( empty($global_settings) ) {
			return;
		}

		$form_settings = $this->get_form_settings( $form );

		$settings = array();
		foreach($global_settings as $key => $val) {
			$settings[$key] = (!empty( $form_settings[$key] )) ? $form_settings[$key] : $val;
		}

		if ( !empty( $settings )) {
			$tooltip_styles = $this->get_inline_styles( $form, $settings );
			$form_id = rgar($form, 'id');
			$css_ref = "gf_tooltips_inline_{$form_id}";
			if ( !empty( $tooltip_styles && !wp_style_is($css_ref) ) ) {
				wp_register_style( $css_ref, false );
				wp_enqueue_style( $css_ref );
				wp_add_inline_style( $css_ref, $tooltip_styles );
			}
		}

	}

	public function get_inline_styles($form, $settings) {

        if ( empty($settings) ) {
            return '';
        }

		$form_id = $form['id'];

		$icon = rgar($settings, 'ttIcon', '\e803');
		$icon_sz = rgar($settings, 'ttIconSize', 18);
		$icon_cl = rgar($settings, 'ttIconColor', '#333333');
		$tip_cl = rgar($settings, 'ttTipColor', '#333333');
		$tip_bg = rgar($settings, 'ttTipBackground', '#333333');
		$tip_sz = rgar($settings, 'ttTipSize', 16);
		$tip_wd = rgar($settings, 'ttMaxWidth', 300);

		$css = '';
		$tip_css = '';
		$icon_css = '';

		if ( $tip_cl != '#333333' )
			$tip_css .= sprintf( 'color:%s;', $tip_cl );

		if ( $tip_bg != '#333333' )
			$tip_css .= sprintf( 'border-color:%1$s;background-color:%1$s;', $tip_bg );

		if ( $tip_sz != 16 )
			$tip_css .= sprintf( 'font-size:%spx;', $tip_sz );

		if ( $tip_wd != 300 )
			$tip_css .= sprintf( 'max-width:%spx;', $tip_wd );

		if ( !empty( $tip_css ) )
			$css .= sprintf( '#powerTip[class^="gftt-%s_"]{%s}', $form_id, $tip_css );


		if ( $icon_cl != '#333333' )
			$icon_css .= sprintf( 'color:%s;', $icon_cl );

		if ( $icon_sz != 18 )
			$icon_css .= sprintf( 'font-size:%spx;', $icon_sz );

		if ( !empty( $icon_css ) )
			$css .= sprintf( '#gform_wrapper_%s .gftt-icon{%s}', $form_id, $icon_css );


		if ( $icon != '\e803' )
			$css .= sprintf( '#gform_wrapper_%s .gftt-icon:before{content:"%s";}', $form_id, $icon );

		return trim( preg_replace( '/\s\s+/', '', $css ) );

	}

	public function form_has_tooltips($form) {

		$has_tooltips = false;

		if ( !empty($form) && isset($form['fields']) ) {
			foreach ( $form['fields'] as &$field ) {
				if ( (property_exists($field, 'jbTooltipContent') && !empty( $field->jbTooltipContent )) || (property_exists($field, 'gftt_choicesEnabled') && !empty($field->gftt_choicesEnabled)) ) {
					$has_tooltips = true;
					break;
				}
			}
		}

		return $has_tooltips;

	}

	// # SETTINGS -----------------------------------------------------------------------------------------------

	public function get_tooltip_settings_fields($include_default_values = true) {

		$fields = array(
			array(
				'type'       => 'radio',
				'name'       => 'ttIcon',
				'label'      => esc_html__( 'Icon', 'gf_tooltips' ),
				'tooltip'    => esc_html__( 'Select the icon that will be used on the tooltip.', 'gf_tooltips' ),
				'choices'    => array(
					array(
						'label' => esc_html__( 'Icon 1', 'gf_tooltips' ),
						'value' => '\e803'
					),
					array(
						'label' => esc_html__( 'Icon 2', 'gf_tooltips' ),
						'value' => '\e806'
					),
					array(
						'label' => esc_html__( 'Icon 3', 'gf_tooltips' ),
						'value' => '\f29c'
					),
					array(
						'label' => esc_html__( 'Icon 4', 'gf_tooltips' ),
						'value' => '\e804'
					),
					array(
						'label' => esc_html__( 'Icon 5', 'gf_tooltips' ),
						'value' => '\e801'
					),
					array(
						'label' => esc_html__( 'Icon 6', 'gf_tooltips' ),
						'value' => '\e802'
					),
					array(
						'label' => esc_html__( 'Icon 7', 'gf_tooltips' ),
						'value' => '\e805'
					),
					array(
						'label' => esc_html__( 'Icon 8', 'gf_tooltips' ),
						'value' => '\f129'
					)
				)
			),
			array(
				'type'    => 'text',
				'value'   => ($include_default_values) ? 18 : '',
				'name'    => 'ttIconSize',
				'class'	  => 'gftt-number',
				'label'   => esc_html__( 'Icon Size', 'gf_tooltips' ),
				'tooltip' => esc_html__( 'Enter the size of the tooltip icon.', 'gf_tooltips' ),
			),
			array(
				'type'    => 'text',
				'value'	  => ($include_default_values) ? '#333333' : '',
				'data-default-color' => ($include_default_values) ? '#333333' : '',
				'name'    => 'ttIconColor',
				'class'	  => 'gftt-color-picker',
				'label'   => esc_html__( 'Icon Color', 'gf_tooltips' ),
				'tooltip' => esc_html__( 'Select the color that will be applied to the tooltip icon.', 'gf_tooltips' )
			),
			array(
				'type'    => 'text',
				'value'   => ($include_default_values) ? 16 : '',
				'name'    => 'ttTipSize',
				'class'	  => 'gftt-number',
				'label'   => esc_html__( 'Content Font Size', 'gf_tooltips' ),
				'tooltip' => esc_html__( 'Enter the font size that will be applied to the tooltip text content.', 'gf_tooltips' )
			),
			array(
				'type'    => 'text',
				'value'	  => ($include_default_values) ? '#ffffff' : '',
				'data-default-color' => ($include_default_values) ? '#ffffff' : '',
				'name'    => 'ttTipColor',
				'class'	  => 'gftt-color-picker',
				'label'   => esc_html__( 'Content Text Color', 'gf_tooltips' ),
				'tooltip' => esc_html__( 'Select the color that will be applied to the tooltip text content.', 'gf_tooltips' )
			),
			array(
				'type'    => 'text',
				'value'	  => ($include_default_values) ? '#333333' : '',
				'data-default-color' => ($include_default_values) ? '#333333' : '',
				'name'    => 'ttTipBackground',
				'class'	  => 'gftt-color-picker',
				'label'   => esc_html__( 'Content Wrap Background', 'gf_tooltips' ),
				'tooltip' => esc_html__( 'Select the background color that will be applied to the tooltip content wrapper.', 'gf_tooltips' )
			),
			array(
				'type'       => 'select',
				'name'       => 'ttPlacement',
				'label'      => esc_html__( 'Placement', 'gf_tooltips' ),
				'tooltip'    => esc_html__( 'Select the default placement of the tooltip when it is open relative to the tooltip icon. Placement can be top, right, bottom, left and many more. Please note that if a tooltip would extend outside of the viewport then its placement will be changed to an orientation that would be entirely within the current viewport.', 'gf_tooltips' ),
				'choices'    => array(
					array(
						'label' => esc_html__( 'Select a position', 'gf_tooltips' ),
						'value' => ''
					),
					array(
						'label' => esc_html__( 'Top Left Alt', 'gf_tooltips' ),
						'value' => 'nw-alt'
					),
					array(
						'label' => esc_html__( 'Top Left', 'gf_tooltips' ),
						'value' => 'nw'
					),
					array(
						'label' => esc_html__( 'Top', 'gf_tooltips' ),
						'value' => 'n'
					),
					array(
						'label' => esc_html__( 'Top Right', 'gf_tooltips' ),
						'value' => 'ne'
					),
					array(
						'label' => esc_html__( 'Top Right Alt', 'gf_tooltips' ),
						'value' => 'ne-alt'
					),
					array(
						'label' => esc_html__( 'Right', 'gf_tooltips' ),
						'value' => 'e'
					),
					array(
						'label' => esc_html__( 'Bottom Right Alt', 'gf_tooltips' ),
						'value' => 'se-alt'
					),
					array(
						'label' => esc_html__( 'Bottom Right', 'gf_tooltips' ),
						'value' => 'se'
					),
					array(
						'label' => esc_html__( 'Bottom', 'gf_tooltips' ),
						'value' => 's'
					),
					array(
						'label' => esc_html__( 'Bottom Left', 'gf_tooltips' ),
						'value' => 'sw'
					),
					array(
						'label' => esc_html__( 'Bottom Left Alt', 'gf_tooltips' ),
						'value' => 'sw-alt'
					),
					array(
						'label' => esc_html__( 'Left', 'gf_tooltips' ),
						'value' => 'w'
					)
				)
			),
			array(
				'type'    => 'text',
				'value'   => ($include_default_values) ? 300 : '',
				'name'    => 'ttMaxWidth',
				'class'	  => 'gftt-width',
				'label'   => esc_html__( 'Max Width', 'gf_tooltips' ),
				'tooltip' => esc_html__( 'Enter the maximum width of the tooltip content wrapper.', 'gf_tooltips' )
			)
		);

		return $fields;

	}

	/**
	 * Creates a settings page for this add-on.
	 */
	public function plugin_settings_fields() {

		$const_key = $this->get_const_license_key();

		$license_field = array(
			'name' => 'gf_tooltips_license_key',
			'tooltip' => esc_html__('Enter the license key you received after purchasing the plugin.', 'gf_tooltips'),
			'label' => esc_html__('License Key', 'gf_tooltips'),
			'type' => 'text',
			'input_type' => 'password',
			'class' => 'medium',
			'default_value' => $this->get_license_key(),
			'after_input' => ( !empty($const_key) ) ? '<div class="alert gforms_note_info">' . __( "Your license key is defined as a constant (likely in wp-config.php) and can't be edited here.", 'gf_tooltips' ) . '</div>' : '',
			'disabled' => ( !empty($const_key) ),
			'validation_callback' => array($this, 'license_validation'),
			'feedback_callback' => array($this, 'license_feedback'),
			'error_message' => esc_html__( 'Invalid license', 'gf_tooltips' ),
		);

        /*
		if (!empty($license) && !empty($status)) {
			$license_field['after_input'] = ($status == 'valid') ? ' License is valid' : ' Invalid or expired license';
		}
        */

		$tooltip_settings_fields = $this->get_tooltip_settings_fields();

		$fields = array(
			array(
				'title'  => esc_html__('To unlock plugin updates, please enter your license key below', 'gf_tooltips'),
				'fields' => array(
					$license_field
				)
			),
			array(
				'title' => __( 'Tooltip Styles', 'gf_tooltips' ),
				'fields' => $tooltip_settings_fields,
			)
		);

		return $fields;
	}


	/**
	 * Form tooltips settings.
	 *
	 * @return array
	 */
	public function form_settings_fields( $form ) {

		$settings = $this->get_form_settings( $form );
		$tooltip_settings_fields = $this->get_tooltip_settings_fields(false);

		$form_override_global_settings_value = (isset($settings['gf_tooltips_override_global']) && $settings['gf_tooltips_override_global'] == 1) ? 1 : 0;
		$form_override_global_settings_field = array(
			'name' => 'gf_tooltips_override_global',
			'label' => '',
			'type' => 'checkbox',
			'choices' => array(
				array(
					'label' => esc_html__('Override global tooltip options for this form?', 'gf_tooltips'),
					'tooltip' => esc_html__('If checked, you can override global tooltip styles for this form.', 'gf_tooltips'),
					'name' => 'gf_tooltips_override_global'
				)
			)
		);
		if (!empty($form_override_global_settings_value)) {
			$form_override_global_settings_field['choices'][0]['default_value'] = 1;
		}

		return array(
			array(
				'title'  => esc_html__( 'Tooltip Form Settings', 'gf_tooltips' ),
				'fields' => array($form_override_global_settings_field),
			),
			array(
				'title' => esc_html__( 'Any options that are left blank will inherit from the global styles in the main settings', 'gf_tooltips' ),
				'fields' => $tooltip_settings_fields,
			)
		);

	}

	// # ADMIN FUNCTIONS -----------------------------------------------------------------------------------------------

	/**
	 * Add the custom settings for the fields to the fields general tab.
	 *
	 * @param int $position The position the settings should be located at.
	 * @param int $form_id The ID of the form currently being edited.
	 */
	public function tooltips_choices_field_settings( $position, $form_id ) {
		if ( $position == 1360 ) {
			?>
            <!-- Choices Tooltips Toggle -->
            <li class="gftt-choices-toggle field_setting" data-js="choices-ui-setting" data-type="main">
                <h6 class="choices-ui__section-label">Tooltips</h6>
				<?php if ( !GFTT_GF_MIN_2_5 ): ?><label class="section_label"><?php esc_html_e("Enable Choices Tooltips", 'gf_tooltips'); ?></label><?php endif; ?>
                <input id="gftt_choices_toggle" class="gftt_choices_toggle" type="checkbox" onclick="gfttAdmin.toggleChoicesTooltips(this.checked);" onkeypress="gfttAdmin.toggleChoicesTooltips(this.checked);"> <label for="gftt_choices_toggle"><?php echo esc_html__("Enable Choices Tooltips", 'gf_tooltips'); gform_tooltip('gftt-choices-toggle') ?></label>
            </li>
			<?php
		}
		if ( $position == 75 ) {
			?>
            <!-- GPPA Choices Tooltips Toggle -->
            <li class="gftt-gppa-choices-toggle field_setting">
                <input type="checkbox" id="gftt_gppa_choices_toggle" class="gftt_gppa_choices_toggle" onclick="gfttAdmin.toggleChoicesTooltips(this.checked);" onkeypress="gfttAdmin.toggleChoicesTooltips(this.checked);"> <label for="gftt_gppa_choices_toggle"><?php echo esc_html__("Enable Choices Tooltips", 'gf_tooltips'); gform_tooltip('gftt-choices-toggle') ?></label>
            </li>
			<?php
		}
	}

	public function custom_settings_tab( $tabs ) {
		$tabs[] = array(
			'id' => 'gftt', // Tab id is used later with the action hook that will output content for this specific tab.
			'title' => esc_html__('Tooltip', 'gf_tooltips'),
			'toggle_classes' => '', // Goes into the tab button class attribute.
			'body_classes'   => '', // Goes into the tab content ( ul tag ) class attribute.
		);

		return $tabs;
	}

	public function custom_settings_markup( $form ) {
		?>
        <li class="gftt-tooltip field_setting">
            <label for="gftt-tooltip" class="section_label">
				<?php esc_html_e( 'Tooltip Content', 'gf_tooltips' ); ?>
				<?php gform_tooltip( 'gftt-tooltip' ) ?>
            </label>
            <textarea id="gftt-tooltip" class="fieldwidth-3"></textarea>
        </li>
		<?php
	}

	/**
	 * Add additional options to the field general settings.
	 */
	public function field_general_settings( $position, $form_id ) {

		if ( $position == 10 ) {
			$this->custom_settings_markup( GFAPI::get_form( $form_id ) );
		}

	}


	/**
	 * Add tooltips option on field appearance settings.
	 */
	public function field_tooltips( $tooltips ) {

		$tooltips['gftt-tooltip'] = sprintf(
			'<h6>%s</h6>%s',
			esc_html__( 'Tooltip Content', 'gf_tooltips' ),
			esc_html__( 'This will be the content that appears on the tooltip.', 'gf_tooltips' )
		);
		$tooltips['gftt-choices-toggle'] = sprintf(
			'<h6>%s</h6>%s',
			esc_html__( 'Enable Choices Tooltips', 'gf_tooltips' ),
			esc_html__( 'Enable to use tooltips on this fields choices.', 'gf_tooltips' )
		);

		return $tooltips;

	}


	// # DISPLAY -----------------------------------------------------------------------------------------------

	/**
	 * This will modify the way the field content is rendered.
	 */
	public function field_content( $content, $field, $value, $lead_id, $form_id ) {

		$tip_content = ( property_exists($field, 'jbTooltipContent') && !empty( $field->jbTooltipContent ) ) ? trim( __( $field->jbTooltipContent, 'gf_tooltips'.'-'.$form_id ) ) : "";
		if ( empty($tip_content) || empty($content) ) {
			return $content;
		}

		$tip_id = 'gftt-' . $form_id . '_' . $field->id;

		if ( ! class_exists( 'JSLikeHTMLElement' ) ) {
			// load our custom updater if it doesn't already exist
			include(dirname(__FILE__) . '/lib/JSLikeHTMLElement.php');
		}

		$dom = new DOMDocument;

		// keeping these in, but they seem to do nothing
        if ( property_exists($dom, 'preserveWhitespace') ) {
	        $dom->preserveWhitespace = true;
        }
        if ( property_exists($dom, 'formatOutput') ) {
	        $dom->formatOutput = false;
        }

		// set error level
		$internalErrors = libxml_use_internal_errors(true);

		$dom->registerNodeClass('DOMElement', 'JSLikeHTMLElement');

		$html_content = $this->__convert_encoding($content);

        if ( empty($html_content) ) {
            return $content;
        }

		$dom->loadHTML($html_content);

		// Restore error level
		libxml_use_internal_errors($internalErrors);

		$selector = new DOMXPath($dom);
		$label_elems = $selector->query('//*[contains(@class, "gfield_label")]');// first try normal field label

		if ($label_elems->length == 0) {
			$label_elems = $selector->query('//*[contains(@class, "gsection_title")]');// then try section title
		}

		if ($label_elems->length == 0) {
			return $content;
		}

		$label = $label_elems->item(0);

		$required_indicator = GFFormsModel::get_required_indicator( $form_id );
        if ( !empty($required_indicator) && strpos($required_indicator, "gfield_required") === FALSE ) {
	        $required_indicator = '<span class="gfield_required">' . $required_indicator . '</span>';
        }

		$label_content = rgobj($field, 'isRequired') ? $field->label . $required_indicator : $field->label;


		$input_elems = $selector->query('//input[contains(@class, "gfield-choice-input")]');
		if ( $input_elems->length === 0 ) {
			$input_elems = $selector->query('//input');// legacy
		}
		if ( $input_elems->length > 0 ) {
			$input_elem = $input_elems->item(0);
			$input_elem->setAttribute('aria-describedby', $tip_id . "-wrap");
		}


		if ( $field->is_form_editor() ) {

			$tip_content = sprintf('<gftt-label class="gftt-label">%s</gftt-label><i class="gftt-icon gform-theme__no-reset--el"></i>', $label_content);// TODO: span, i

		}
		else {

			$tip_placement = 'nw-alt';// default placement
			$form = GFAPI::get_form( $form_id );

			$global_settings = $this->get_plugin_settings();
			$form_settings = $this->get_form_settings( $form );

			if (!empty($global_settings) && isset($global_settings['ttPlacement']) && !empty($global_settings['ttPlacement'])) {
				$tip_placement = $global_settings['ttPlacement'];
			}
			if (!empty($form_settings) && isset($form_settings['ttPlacement']) && !empty($form_settings['ttPlacement'])) {
				$tip_placement = $form_settings['ttPlacement'];
			}

			$use_tabindex = apply_filters('gftt_use_tabindex', true, $form_id, $field->id);
			$tabindex_value = apply_filters('gftt_tabindex_value', 0, $form_id, $field->id);
			$tabindex = $use_tabindex ? 'tabindex="'.$tabindex_value.'"' : '';

			$tip_content = sprintf(
				'<gftt-label class="gftt-label">%3$s</gftt-label><i id="%1$s" class="gftt-icon gform-theme__no-reset--el" '.$tabindex.' data-placement="%4$s"></i><div class="gftt-content" id="%1$s-wrap" role="tooltip" aria-hidden="false"><div id="%1$s-content" data-tid="%1$s" aria-hidden="false">%2$s</div></div>',
				$tip_id,
				$tip_content,
				$label_content,
				$tip_placement
			);

		}

		$label->innerHTML = do_shortcode( $tip_content );

		$html = $dom->saveHTML($dom->documentElement);

		if ( $field->type == 'radio' || $field->type == 'checkbox' || $field->inputType == 'radio' || $field->inputType == 'checkbox' ) {
			// put whitespace back in between label and radio/checkbox inputs
			return str_replace( "><label ", ">\n<label ", $html );
		}

		if ( property_exists($field, 'gwreadonly_enable') && !empty($field->gwreadonly_enable) && strpos($html, " readonly ") !== FALSE ) {
			// put full readonly attribute and value in
			return str_replace( ' readonly ', ' readonly="readonly" ', $html );
		}

		return $html;
		//return utf8_decode( $dom->saveHTML($dom->documentElement) );

	}

    private function __convert_encoding( $content ) {

	    // XML_PARSE_NOBLANKS
	    // (original) mb_convert_encoding deprecated
	    // $html_content = mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8');
	    /*
		// option 1
		$dom->loadHTML(mb_encode_numericentity(
			htmlspecialchars_decode(
				htmlentities($content, ENT_NOQUOTES, 'UTF-8', false)
				,ENT_NOQUOTES
			), [0x80, 0x10FFFF, 0, ~0],
			'UTF-8'
		));
		*/
	    // option 2
	    //$html_content = htmlspecialchars_decode(iconv('UTF-8', 'ISO-8859-1', htmlentities($content, ENT_COMPAT, 'UTF-8')), ENT_QUOTES);
	    // option 3
	    //$html_content = mb_encode_numericentity(htmlentities($content, ENT_QUOTES, 'UTF-8'), [0x80, 0x10FFFF, 0, ~0], 'UTF-8');

	    $html_content = $content;
	    if ( function_exists('mb_encode_numericentity') && apply_filters('gftt_use_mb_encode_numericentity', true) !== false ) {
		    // option 4
            $html_content = mb_encode_numericentity(
	            htmlspecialchars_decode(
		            htmlentities($content, ENT_NOQUOTES, 'UTF-8', false)
		            ,ENT_NOQUOTES
	            ), [0x80, 0x10FFFF, 0, ~0],
	            'UTF-8'
            );
	    }
	    else if ( function_exists('mb_convert_encoding') ) {
		    $html_content = call_user_func_array('mb_convert_encoding', array(&$content,'HTML-ENTITIES','UTF-8'));
	    }

        return $html_content;

    }

	public function add_choices_tooltips_markup( $choice_markup, $choice, $field, $value ) {

		$form_id = $field->formId;
		$field_id = $field->id;

		$is_image_or_color_choice = ( (property_exists($field, 'imageChoices_enableImages') && !empty($field->imageChoices_enableImages)) || (property_exists($field, 'colorPicker_enableColors') && !empty($field->colorPicker_enableColors)) );

		$is_other_choice = ( $choice['value'] == "gf_other_choice" );
		$is_select_all = ( ($field->type == 'checkbox' || $field->optionType == 'checkbox') && empty($choice) );
		$has_tooltip = ( isset($choice['jbTooltipContent']) && !empty($choice['jbTooltipContent']) );
		$tip_content = ( $has_tooltip ) ? $choice['jbTooltipContent'] : "";

		$choices_tooltips_enabled = ( property_exists($field, 'gftt_choicesEnabled') && !empty($field->gftt_choicesEnabled) );

        if ( !$choices_tooltips_enabled || $is_other_choice || $is_select_all || !$has_tooltip || empty($field->choices) ) {
			return $choice_markup;
		}

		$choice_id = '';
		foreach( $field->choices as $n => $ch ) {
			if ( $ch['text'] == $choice['text'] ) {
				$choice_id = $n+1;
			}
		}

		$tip_id = "gftt-{$form_id}_{$field_id}_{$choice_id}";

		if ( ! class_exists( 'JSLikeHTMLElement' ) ) {
			// load our custom updater if it doesn't already exist
			include(dirname(__FILE__) . '/lib/JSLikeHTMLElement.php');
		}

		$dom = new DOMDocument;

		// keeping these in, but they seem to do nothing
		if ( property_exists($dom, 'preserveWhitespace') ) {
			$dom->preserveWhitespace = true;
		}
		if ( property_exists($dom, 'formatOutput') ) {
			$dom->formatOutput = false;
		}

		// set error level
		$internalErrors = libxml_use_internal_errors(true);

		$dom->registerNodeClass('DOMElement', 'JSLikeHTMLElement');

        $html_content = $this->__convert_encoding($choice_markup);

        if ( empty($html_content) ) {
            return $choice_markup;
        }

		$dom->loadHTML($html_content);

		// Restore error level
		libxml_use_internal_errors($internalErrors);

		$selector = new DOMXPath($dom);
		$choice_elems = $selector->query('//*[contains(@class, "gchoice")]');// first try normal field label

		if ( $choice_elems->length === 0 ) {
			return $choice_markup;
		}

		$choice_elem = $choice_elems->item(0);

		$choice_cls = $choice_elem->getAttribute('class');
		$choice_elem->setAttribute('class', "{$choice_cls} has-tooltip");

		$icon = $dom->createElement('i');
		$icon->setAttribute('class', 'gftt-icon gform-theme__no-reset--el');

		$tooltipContentWrap = $dom->createElement('div');//TODO: span
		$tooltipContentWrap->setAttribute('id', $tip_id . "-wrap");
		$tooltipContentWrap->setAttribute('class', 'gftt-content');
		$tooltipContentWrap->setAttribute('aria-hidden', 'false');
		$tooltipContentWrap->setAttribute('role', 'tooltip');

		$tooltipContent = $dom->createElement('div');//TODO: span
		$tooltipContent->setAttribute('aria-hidden', 'false');

		$input_elems = $selector->query('//input[contains(@class, "gfield-choice-input")]');
		if ( $input_elems->length === 0 ) {
			$input_elems = $selector->query('//input');// legacy
		}
		if ( $input_elems->length > 0 ) {
			$input_elem = $input_elems->item(0);
			$input_elem->setAttribute('aria-describedby', $tip_id . "-wrap");
		}

		$choice_label = $selector->query('//label')->item(0);

		if ( $field->is_form_editor() ) {

			if ( $is_image_or_color_choice ) {
				$choice_elem->appendChild($icon);
			}
			else {
				$choice_label->innerHTML = '<gftt-label class="gftt-label">' . $choice_label->innerHTML . '</gftt-label>';//TODO: span
				$choice_label->appendChild($icon);
			}

		}
		else {

			$tip_placement = 'nw-alt';// default placement
			$form = GFAPI::get_form( $form_id );

			$global_settings = $this->get_plugin_settings();
			$form_settings = $this->get_form_settings( $form );

			if (!empty($global_settings) && isset($global_settings['ttPlacement']) && !empty($global_settings['ttPlacement'])) {
				$tip_placement = $global_settings['ttPlacement'];
			}
			if (!empty($form_settings) && isset($form_settings['ttPlacement']) && !empty($form_settings['ttPlacement'])) {
				$tip_placement = $form_settings['ttPlacement'];
			}

			$icon->setAttribute('id', $tip_id);
			$icon->setAttribute('data-placement', $tip_placement);

			$use_tabindex = apply_filters('gftt_use_tabindex', true, $form_id, $field->id);
			$tabindex_value = apply_filters('gftt_tabindex_value', 0, $form_id, $field->id);
			if ( $use_tabindex ) {
				$icon->setAttribute('tabindex', "".$tabindex_value);
			}

			$tooltipContent->setAttribute('id', $tip_id . "-content");
			$tooltipContent->setAttribute('data-tid', $tip_id);
			$tooltipContent->innerHTML = do_shortcode( $tip_content );

			$tooltipContentWrap->appendChild( $tooltipContent );

			if ( $is_image_or_color_choice ) {
				$choice_elem->appendChild($icon);
				$choice_elem->appendChild($tooltipContentWrap);
			}
			else {
				$choice_label->innerHTML = '<gftt-label class="gftt-label">' . $choice_label->innerHTML . '</gftt-label>';//TODO: span
				$choice_label->appendChild($icon);
				$choice_label->appendChild($tooltipContentWrap);
			}

		}

		$choice_markup = $dom->saveHTML($dom->documentElement);

		if ( $field->type == 'radio' || $field->type == 'checkbox' || $field->inputType == 'radio' || $field->inputType == 'checkbox' ) {
			// put whitespace back in between label and radio/checkbox inputs
			return str_replace( "><label ", ">\n<label ", $choice_markup );
		}

		return $choice_markup;

	}

	public function add_inline_options_tooltips_lookup( $form_string, $form, $current_page ) {

		$form_id = $form['id'];

		$tip_placement = 'nw-alt';// default placement
		$global_settings = $this->get_plugin_settings();
		$form_settings = $this->get_form_settings( $form );

		if (!empty($global_settings) && isset($global_settings['ttPlacement']) && !empty($global_settings['ttPlacement'])) {
			$tip_placement = $global_settings['ttPlacement'];
		}
		if (!empty($form_settings) && isset($form_settings['ttPlacement']) && !empty($form_settings['ttPlacement'])) {
			$tip_placement = $form_settings['ttPlacement'];
		}

		$option_tooltips_lookup = [];
		foreach( $form['fields'] as $field ) {
			if ( !is_object( $field ) || empty( $field->choices ) || !is_array( $field->choices ) ) {
				continue;
			}

			$key = "field_" . $field->id;

			$use_tabindex = apply_filters('gftt_use_tabindex', true, $form_id, $field->id);
			$tabindex_value = apply_filters('gftt_tabindex_value', 0, $form_id, $field->id);

			foreach( $field->choices as $i => $choice ) {
				$has_tooltip = ( isset($choice['jbTooltipContent']) && !empty($choice['jbTooltipContent']) );
				if ( !isset($option_tooltips_lookup[$key]) ) {
					$option_tooltips_lookup[$key] = [];
				}
				$option_tooltips_lookup[$key][$i] = array(
                    "label" => $choice['text'],
                    "tooltip" => $has_tooltip ? $choice['jbTooltipContent'] : "",
                    "placement" => $tip_placement,
                    "tabindex" => $use_tabindex ? "{$tabindex_value}" : "",
                );
			}

		}
		$form_string .= "<script type=\"text/javascript\"> window.tooltipsOptionsContent = window.tooltipsOptionsContent || {}; window.tooltipsOptionsContent[".$form['id']."] = " . json_encode($option_tooltips_lookup) . ";  </script>";

		return $form_string;
	}

	// # UPDATES -----------------------------------------------------------------------------------------------

	/**
	 * Add custom messages after plugin row based on license status
	 */

	public function gf_plugin_row($plugin_file='', $plugin_data=array(), $status='') {
		$row = "";
		$license_key = $this->get_license_key();
		$license_status = $this->get_license_status( $license_key );
		if ( empty($license_key) || empty($license_status) ) {
			ob_start();
			?>
            <tr class="plugin-update-tr">
                <td colspan="3" class="plugin-update gf_tooltips-plugin-update">
                    <div class="update-message">
                        <a href="<?php echo admin_url('admin.php?page=gf_settings&subview=' . $this->_slug); ?>">Activate</a> your license to receive plugin updates and support. Need a license key? <a href="<?php echo $this->_url; ?>" target="_blank">Purchase one now</a>.
                    </div>
                    <style>
						.plugin-update.gf_tooltips-plugin-update .update-message:before {
							content: "\f348";
							margin-top: 0;
							font-family: dashicons;
							font-size: 20px;
							position: relative;
							top: 5px;
							color: orange;
							margin-right: 8px;
						}
						.plugin-update.gf_tooltips-plugin-update {
							background-color: #fff6e5;
						}
						.plugin-update.gf_tooltips-plugin-update .update-message {
							margin: 0 20px 6px 40px !important;
							line-height: 28px;
						}
                    </style>
                </td>
            </tr>
			<?php
			$row = ob_get_clean();
		}
		else if( !empty($license_key) && $license_status != 'valid' && $license_status != 'unknown' ) {
			ob_start();
			?>
            <tr class="plugin-update-tr">
                <td colspan="3" class="plugin-update gf_tooltips-plugin-update">
                    <div class="update-message">
                        Your license is invalid or expired. <a href="<?php echo admin_url('admin.php?page=gf_settings&subview=' . $this->_slug); ?>">Enter a valid license key</a> or <a href="<?php echo $this->_url; ?>" target="_blank">purchase a new one</a>.
                    </div>
                    <style>
						.plugin-update.gf_tooltips-plugin-update .update-message:before {
							content: "\f348";
							margin-top: 0;
							font-family: dashicons;
							font-size: 20px;
							position: relative;
							top: 5px;
							color: #d54e21;
							margin-right: 8px;
						}
						.plugin-update.gf_tooltips-plugin-update {
							background-color: #fff6e5;
						}
						.plugin-update.gf_tooltips-update .update-message {
							margin: 0 20px 6px 40px !important;
							line-height: 28px;
						}
                    </style>
                </td>
            </tr>
			<?php
			$row = ob_get_clean();
		}

		echo $row;
	}


	public function upgrade( $previous_version ) {
		if ( empty($previous_version) ) {
			return;
		}

		$this->old_license_data_cleanup();
	}


	/**********************************************
	 *
	 *   LICENSE
	 *
	 **********************************************/


	private const STATUS_TTL = WEEK_IN_SECONDS;   // cache of 'valid'/'expired'/etc.
	private const LAST_STATUS_TTL = MONTH_IN_SECONDS;  // last known (for UI/guardrails)
	private const CHECK_THROTTLE_TTL = HOUR_IN_SECONDS;   // burst guard when status cache is cold

	/**
	 * Prefix for all caches/options for this product.
	 * If you reuse this pattern in other addons, change the prefix per product.
	 */
	private function t_prefix(): string {
		return 'gf_tooltips_';
	}

	/** md5 hash of the license key (avoid storing raw key in option names) */
	private function key_hash( string $key ): string {
		return md5( trim( $key ) );
	}

	/** Build a transient/option key that is product-scoped and (optionally) license-scoped. */
	private function t_key( string $suffix, string $key = '' ): string {
		$hash = $key !== '' ? '_' . $this->key_hash( $key ) : '';
		return $this->t_prefix() . $suffix . $hash;
	}

	public function init_license() {

		$old_data_cleanup_key = $this->t_key('old_license_data_cleanup');
		if ( isset($_GET[$old_data_cleanup_key]) ) {
			$this->old_license_data_cleanup();
		}

		// constant overrides saved key
		$saved_key = $this->get_license_key();
		$const_key = $this->get_const_license_key();

		if ( ! empty( $const_key ) && $saved_key !== $const_key ) {
			// force constant into settings
			$settings = $this->get_plugin_settings();
			if ( false === $settings ) {
				$settings = array();
			}
			$settings['gf_tooltips_license_key'] = $const_key;
			$this->update_plugin_settings( $settings );

			if ( ! empty( $saved_key ) ) {
				// deactivate old key (no extra check call)
				$this->deactivate_license_key( $saved_key );
			}
			$key = $const_key;
		}
		else {
			$key = $saved_key;
		}

		// activate current key if needed
		$this->activate_license_key( $key );
	}

	public function get_const_license_key() {
		return defined( 'GF_TOOLTIPS_LICENSE' ) ? GF_TOOLTIPS_LICENSE : ''; // keep if constant may be absent on some installs
	}

	public function get_license_key() {
		return $this->get_plugin_setting( 'gf_tooltips_license_key' );
	}

	private function activate_license_key( $key, $force = false ) {
		if ( empty( $key ) ) {
			return;
		}

		$activated_key = $this->t_key( 'license_activated', $key );
		$activated     = get_option( $activated_key );

		if ( ! empty( $activated ) && $force !== true ) {
			// already activated, skip
			return;
		}

		$status = $this->get_license_status( $key );
		if ( $status === 'valid' ) {
			// already active for this site; update local flag
			update_option( $activated_key, '1', false );
			return;
		}

		$response = $this->perform_edd_license_request( 'activate_license', $key );
		$status   = rgobj( $response, 'license', '' );

		if ( ! empty( $status ) ) {
			$this->update_license_status( $key, $status );
			if ( $status === 'valid' ) {
				update_option( $activated_key, '1', false );
			}
		}
	}

	private function deactivate_license_key( $key, $force = false ) {
		if ( empty( $key ) ) {
			return;
		}

		// Use cached status only to decide whether to skip; no forced network check.
		$current_status = get_transient( $this->t_key( 'license_status', $key ) );
		if ( $current_status === 'site_inactive' && $force !== true ) {
			return;
		}

		// Single remote call to deactivate
		$this->perform_edd_license_request( 'deactivate_license', $key );

		// Clear local state
		delete_transient( $this->t_key( 'license_status', $key ) );
		delete_option( $this->t_key( 'license_activated', $key ) );

		// Track last known status for UI/guards
		set_transient( $this->t_key( 'last_status', $key ), 'site_inactive', self::LAST_STATUS_TTL );
	}

	/**
	 * Determine if the license key is valid so the appropriate icon can be displayed next to the field.
	 *
	 * @param string $value The current value of the license_key field.
	 * @param array  $field The field properties.
	 *
	 * @return bool|null
	 */
	public function license_feedback( $value, $field = null ) {
		$status = $this->get_license_status( $value );
		return is_null( $status ) ? null : ( $status === 'valid' );
	}

	private function update_license_status( $key, $status ) {
		// Per-key caches
		set_transient( $this->t_key( 'last_status', $key ),    $status, self::LAST_STATUS_TTL );
		set_transient( $this->t_key( 'license_status', $key ), $status, self::STATUS_TTL );
	}

	/**
	 * Handle license key validation (maybe activate)
	 * Note: This gets called AFTER settings are saved
	 *
	 * @param array  $field         The field properties.
	 * @param string $field_setting The submitted value of the license_key field.
	 */
	public function license_validation( $field, $field_setting ) {
		$prev_key_opt = $this->t_key( 'prev_key' );
		$previous_key = get_option( $prev_key_opt );

		if ( ! empty( $previous_key ) && $previous_key !== $field_setting ) {
			// new key submitted: deactivate old one
			$this->deactivate_license_key( $previous_key );
		}

		// We can skip force here; admin save is still cheap but respects caches if warm.
		$status    = $this->get_license_status( $field_setting, false );
		$activated = get_option( $this->t_key( 'license_activated', $field_setting ) );

		if ( $status === 'site_inactive' || empty( $activated ) ) {
			$this->activate_license_key( $field_setting, true );
		}

		update_option( $prev_key_opt, $field_setting, false );
	}

	/**
	 * Get the current license status with caching + throttle.
	 *
	 * Behavior:
	 * - If week cache exists → return it (no network).
	 * - If cache missing:
	 *     - If throttle (1h) is present and not forcing → return last known (or 'unknown') without network.
	 *     - Else perform a single network check, set throttle (1h), set caches.
	 */
	public function get_license_status( $key, $force = false ) {
		if ( empty( $key ) ) {
			return null;
		}

		$status_t = $this->t_key( 'license_status', $key );
		$throttle_t = $this->t_key( 'license_checked', $key );
		$last_t = $this->t_key( 'last_status', $key );

		// 1) Week-long per-key cache
		$status = get_transient( $status_t );
		if ( $status !== false && ! $force ) {
			return $status;
		}

		// 2) If not forcing and throttle exists, avoid another network hit
		if ( ! $force && get_transient( $throttle_t ) !== false ) {
			$last = get_transient( $last_t );
			$this->log_debug( "get_license_status() using last_status::{$last}" );
			return $last !== false ? $last : 'unknown';
		}

		// 3) Perform a single check, with throttle guard
		set_transient( $throttle_t, '1', self::CHECK_THROTTLE_TTL );

		$license_data = $this->perform_edd_license_request( 'check_license', $key );
		if ( ! is_object( $license_data ) ) {
			set_transient( $last_t, 'unknown', self::LAST_STATUS_TTL );
			return 'unknown';
		}

		$status = rgobj( $license_data, 'license', '' );
		if ( empty( $status ) ) {
			set_transient( $last_t, 'unknown', self::LAST_STATUS_TTL );
			return 'unknown';
		}

		$this->update_license_status( $key, $status );
		return $status;
	}

	/**
	 * Send a request to the EDD store url.
	 *
	 * @param string $edd_action The action to perform (check_license, activate_license or deactivate_license).
	 * @param string $license    The license key.
	 *
	 * @return object
	 */
	public function perform_edd_license_request( $edd_action, $license ) {

		$endpoint = $this->build_edd_sl_endpoint( GFTT_HOME );

		$args = array(
			'timeout'   => GFTT_TIMEOUT,
			'sslverify' => GFTT_SSL_VERIFY,
			'headers'   => array(
				'X-JetSloth-License' => '1',
			),
			'body' => array(
				'edd_action' => $edd_action,// activate_license | deactivate_license | check_license | get_version
				'license' => trim( $license ),
				'item_name' => GFTT_NAME,
				'version' => GFTT_VERSION,
				'url' => home_url(),
			),
		);

		$response = wp_remote_post( $endpoint, $args );

		if ( is_wp_error( $response ) ) {
			$this->log_debug( 'EDD request error: ' . $response->get_error_message() );
			return (object) array();
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );

		if ( $code < 200 || $code >= 300 ) {
			$this->log_debug( 'EDD HTTP ' . $code . ': ' . $body );
			return (object) array();
		}

		$result = json_decode( $body );
		//$this->log_debug( 'perform_edd_license_request() result:: ' . print_r( $result, true ) );

		return is_object( $result ) ? $result : (object) array();
	}

	/**
	 * Build a robust /edd-sl/ endpoint from an arbitrary base URL.
	 * Handles roots, subpaths, and double slashes.
	 */
	private function build_edd_sl_endpoint( string $base ): string {
		$base = trailingslashit( $base );
		if ( preg_match( '#/edd-sl/?$#i', $base ) ) {
			return $base;
		}
		return $base . 'edd-sl/';
	}


	/**********************************************
	 *
	 *   OLD LICENSE TRANSIENTS/OPTIONS CLEANUP
	 *
	 **********************************************/

	private function old_license_data_cleanup(): void {
		// Cleanup legacy (pre-hash) license transients/options on this site.
		$this->maybe_run_legacy_cleanup();

		// Also sweep every site in a network if this add-on is network-activated.
		if ( is_multisite() && $this->is_network_activated() ) {
			$this->cleanup_legacy_license_network_wide();
		}
	}

	/**
	 * Iterate all sites and run the single-site cleanup on each.
	 * Safe on medium networks thanks to chunked deletes in the helpers.
	 */
	private function cleanup_legacy_license_network_wide(): void {
		$this->log_debug('cleanup_legacy_license_network_wide()');
		$sites = get_sites( [ 'fields' => 'ids' ] );
		foreach ( $sites as $blog_id ) {
			switch_to_blog( $blog_id );
			$this->maybe_run_legacy_cleanup();
			restore_current_blog();
		}
	}

	/**
	 * Run on plugin load/upgrade to purge legacy license caches/options.
	 * Call this from plugins_loaded or your existing upgrade routine.
	 */
	public function maybe_run_legacy_cleanup() {
		$ver_opt = $this->t_prefix() . 'cleanup_ver';
		$stored = get_option( $ver_opt );
		$current = GFTT_VERSION;

		// Only run once per version
		if ( $stored === $current ) {
			return;
		}

		$this->cleanup_legacy_license_caches( $this->t_prefix() );

		update_option( $ver_opt, $current, false );
	}

	/**
	 * Delete legacy (pre-hash) license options/transients safely.
	 * @param string $prefix e.g. 'gf_fetcher_'
	 */
	private function cleanup_legacy_license_caches( string $prefix ) {

		$this->log_debug('cleanup_legacy_license_caches()');

		global $wpdb;

		// ---- Site options (non-transients) ----
		// 1) license_activated_{raw-key}
		$this->delete_options_like( "{$prefix}license_activated_%" );
		// 2) prev_key
		delete_option( "{$prefix}prev_key" );

		// ---- Site transients ----
		// Transients are stored as options: _transient_{key}, _transient_timeout_{key}
		$this->delete_transients_like( "{$prefix}license_status_%" ); // per-key legacy
		$this->delete_transients_like( "{$prefix}last_status" );      // global legacy
		$this->delete_transients_like( "{$prefix}license_checked" );  // global legacy

		// ---- Network (multisite) equivalents ----
		if ( is_multisite() ) {
			$this->delete_site_options_like( "{$prefix}license_activated_%" );
			delete_site_option( "{$prefix}prev_key" );

			$this->delete_site_transients_like( "{$prefix}license_status_%" );
			$this->delete_site_transients_like( "{$prefix}last_status" );
			$this->delete_site_transients_like( "{$prefix}license_checked" );
		}
	}

	/**
	 * Delete site options by LIKE pattern (safe, chunked).
	 */
	private function delete_options_like( string $like_pattern, int $limit = 500 ) {
		global $wpdb;
		$table = $wpdb->options;

		$pattern = $wpdb->esc_like( $like_pattern );
		$pattern = str_replace( ['\\%', '\\_'], ['%', '_'], $pattern ); // keep wildcards

		do {
			$rows = $wpdb->get_col( $wpdb->prepare(
				"SELECT option_name FROM {$table} WHERE option_name LIKE %s LIMIT %d",
				$pattern, $limit
			) );

			if ( empty( $rows ) ) {
				break;
			}

			foreach ( $rows as $name ) {
				delete_option( $name );
			}
		} while ( count( $rows ) === $limit );
	}

	/**
	 * Delete site transients whose transient *keys* match LIKE pattern.
	 * Handles both the value and the timeout rows.
	 */
	private function delete_transients_like( string $transient_key_like, int $limit = 500 ) {
		global $wpdb;
		$table    = $wpdb->options;
		$patternV = '_transient_' . $transient_key_like;
		$patternT = '_transient_timeout_' . $transient_key_like;

		$patternV = $wpdb->esc_like( $patternV );
		$patternT = $wpdb->esc_like( $patternT );
		$patternV = str_replace( ['\\%', '\\_'], ['%', '_'], $patternV );
		$patternT = str_replace( ['\\%', '\\_'], ['%', '_'], $patternT );

		do {
			$rows = $wpdb->get_col( $wpdb->prepare(
				"SELECT option_name FROM {$table} WHERE option_name LIKE %s OR option_name LIKE %s LIMIT %d",
				$patternV, $patternT, $limit
			) );

			if ( empty( $rows ) ) {
				break;
			}

			foreach ( $rows as $name ) {
				delete_option( $name );
			}
		} while ( count( $rows ) === $limit );
	}

	/**
	 * Delete *network* options by LIKE pattern (multisite).
	 */
	private function delete_site_options_like( string $like_pattern, int $limit = 500 ) {
		global $wpdb;
		$table = $wpdb->sitemeta;

		$pattern = $wpdb->esc_like( $like_pattern );
		$pattern = str_replace( ['\\%', '\\_'], ['%', '_'], $pattern );

		do {
			$rows = $wpdb->get_results( $wpdb->prepare(
				"SELECT meta_id, meta_key FROM {$table} WHERE meta_key LIKE %s LIMIT %d",
				$pattern, $limit
			) );

			if ( empty( $rows ) ) {
				break;
			}

			foreach ( $rows as $row ) {
				delete_site_option( $row->meta_key );
			}
		} while ( count( $rows ) === $limit );
	}

	/**
	 * Delete *network* transients matching LIKE pattern (multisite).
	 * Network transients live in sitemeta as _site_transient_{key} and _site_transient_timeout_{key}.
	 */
	private function delete_site_transients_like( string $transient_key_like, int $limit = 500 ) {
		global $wpdb;
		$table    = $wpdb->sitemeta;
		$patternV = '_site_transient_' . $transient_key_like;
		$patternT = '_site_transient_timeout_' . $transient_key_like;

		$patternV = $wpdb->esc_like( $patternV );
		$patternT = $wpdb->esc_like( $patternT );
		$patternV = str_replace( ['\\%', '\\_'], ['%', '_'], $patternV );
		$patternT = str_replace( ['\\%', '\\_'], ['%', '_'], $patternT );

		do {
			$rows = $wpdb->get_results( $wpdb->prepare(
				"SELECT meta_id, meta_key FROM {$table} WHERE meta_key LIKE %s OR meta_key LIKE %s LIMIT %d",
				$patternV, $patternT, $limit
			) );

			if ( empty( $rows ) ) {
				break;
			}

			foreach ( $rows as $row ) {
				// Strip the prefix to get the transient key and use core API to delete both rows
				if ( strpos( $row->meta_key, '_site_transient_timeout_' ) === 0 ) {
					$key = substr( $row->meta_key, strlen( '_site_transient_timeout_' ) );
				} elseif ( strpos( $row->meta_key, '_site_transient_' ) === 0 ) {
					$key = substr( $row->meta_key, strlen( '_site_transient_' ) );
				} else {
					continue;
				}
				delete_site_transient( $key );
			}
		} while ( count( $rows ) === $limit );
	}



	public function debug_output($data = '', $background='black', $color='white') {
		echo '<pre style="padding:20px; background:'.$background.'; color:'.$color.';">';
		print_r($data);
		echo '</pre>';
	}



}
