<?php
namespace HTMegaPro\Elementor\Widget;
use Elementor\Plugin as Elementor;
use Elementor\Widget_Base;
use Elementor\Repeater;
use Elementor\Controls_Manager;
use Elementor\TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\HTMega_Icon_manager;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class HTMega_Breadcrumbs_Element extends Widget_Base {

    public function get_name() {
        return 'htmega-breadcrumbs';
    }
    
    public function get_title() {
        return __( 'HT Breadcrumbs', 'htmega-pro' );
    }

    public function get_icon() {
        return 'htmega-icon eicon-justify-space-around-v';
    }

    public function get_categories() {
        return [ 'htmegapro-addons' , 'htmega_builder'];
    }
    
    public function get_keywords() {
        return [ 'htmega', 'ht mega', 'breadcrumb', 'page title', 'blog breadcrumb', 'htmega pro' ];
    }
    
    public function get_custom_help_url() {
		return 'https://wphtmega.com/docs/general-widgets/breadcrumbs-widget/';
	}

    protected function register_controls() {

        // General
        $this->start_controls_section(
            'breadcrumbs_general_section',
            [
                'label' => __( 'General', 'htmega-pro' ),
            ]
        );

            $this->add_control(
                'breadcrumbs_style',
                [
                    'label' => __( 'Style', 'htmega-pro' ),
                    'type' => 'htmega-preset-select',
                    'default' => '1',
                    'options' => [
                        '1'   => __( 'Style One', 'htmega-pro' ),
                        '2'   => __( 'Style Two', 'htmega-pro' ),
                        '3'   => __( 'Style Three', 'htmega-pro' ),
                        '4'   => __( 'Style Four', 'htmega-pro' ),
                        '5'   => __( 'Style Five', 'htmega-pro' ),
                        '6'   => __( 'Style Six', 'htmega-pro' ),
                        '7'   => __( 'Style Seven', 'htmega-pro' ),
                        '8'   => __( 'Style Eight', 'htmega-pro' ),
                        '9'   => __( 'Style Nine', 'htmega-pro' ),
                    ],
                ]
            );

            $this->add_control(
                'show_home',
                [
                    'label' => __('Show Home', 'htmega-pro'),
                    'type' => Controls_Manager::SWITCHER,
                    'default' => 'yes',
                    'label_on' => __('On', 'htmega-pro'),
                    'label_off' => __('Off', 'htmega-pro'),
                    'return_value' => 'yes',
                    'separator' => 'before',
                ]
            );

            $this->add_control(
                'home_text',
                [
                    'label' => __('Home Text', 'htmega-pro'),
                    'type' => Controls_Manager::TEXT,
                    'default' => __('Home', 'htmega-pro'),
                    'dynamic' => [
                        'active' => true,
                    ],
                    'condition' => [
                        'show_home' => 'yes',
                    ],
                ]
            );

            $this->add_control(
                'select_home_icon',
                [
                    'label' => __('Home Icon', 'htmega-pro'),
                    'type' => Controls_Manager::ICONS,
                    'label_block' => false,
                    'skin' => 'inline',
                    'default' => [
                        'value' => 'fas fa-home',
                        'library' => 'fa-solid',
                    ],
                    'condition' => [
                        'show_home' => 'yes',
                    ],
                ]
            );

            $this->add_control(
                'breadcrumbs_home_spacing',
                [
                    'label' => __('Spacing', 'htmega-pro'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 10,
                    ],
                    'range' => [
                        'px' => [
                            'max' => 50,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-item-link svg , {{WRAPPER}} .htmega-breadcrumb-item-link i' => 'margin-right: {{SIZE}}{{UNIT}};',
                    ],
                    'condition' => [
                        'select_home_icon[library]!' => '',
                        'show_home' => 'yes',
                    ]
                ]
            );

            $this->add_control(
                'breadcrumbs_svg_size',
                [
                    'label' => __('SVG Size', 'htmega-pro'),
                    'description' => __('Using the SVG icon will make it work.','htmega-pro'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 20,
                    ],
                    'range' => [
                        'px' => [
                            'max' => 100,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} ul.htmega-breadcrumb li.htmega-breadcrumb-item a svg' => 'width: {{SIZE}}{{UNIT}};height: {{SIZE}}{{UNIT}};',
                    ],
                    'condition' => [
                        'select_home_icon[library]' => 'svg',
                        'show_home' => 'yes',
                    ]
                ]
            );

            $this->add_responsive_control(
                'breadcrumbs_align',
                [
                    'label' => __('Alignment', 'htmega-pro'),
                    'type' => Controls_Manager::CHOOSE,
                    'default' => 'left',
                    'options' => [
                        'left' => [
                            'title' => __('Left', 'htmega-pro'),
                            'icon' => 'eicon-h-align-left',
                        ],
                        'center' => [
                            'title' => __('Center', 'htmega-pro'),
                            'icon' => 'eicon-h-align-center',
                        ],
                        'right' => [
                            'title' => __('Right', 'htmega-pro'),
                            'icon' => 'eicon-h-align-right',
                        ],
                    ],
                    'selectors_dictionary' => [
                        'left' => 'flex-start',
                        'center' => 'center',
                        'right' => 'flex-end',
                    ],
                    'separator' => 'before',
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-box ul.htmega-breadcrumb' => 'justify-content: {{VALUE}};',
                    ],
                    'condition'=>[
                        'breadcrumbs_style!' => ['3','4','5','6','8','9'],
                    ]
                ]
            );

            $this->add_responsive_control(
                'breadcrumbs_align_two',
                [
                    'label' => __('Alignment', 'htmega-pro'),
                    'type' => Controls_Manager::CHOOSE,
                    'default' => '',
                    'options' => [
                        'left' => [
                            'title' => __('Left', 'htmega-pro'),
                            'icon' => 'eicon-h-align-left',
                        ],
                        'center' => [
                            'title' => __('Center', 'htmega-pro'),
                            'icon' => 'eicon-h-align-center',
                        ],
                        'right' => [
                            'title' => __('Right', 'htmega-pro'),
                            'icon' => 'eicon-h-align-right',
                        ],
                    ],
                    'separator' => 'before',
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-box' => 'text-align: {{VALUE}};',
                    ],
                    'condition'=>[
                        'breadcrumbs_style' => ['3','4','5','6','8','9'],
                    ]
                ]
            );

            $this->add_control(
                'breadcrumbs_width',
                [
                    'label' => __('Width', 'htmega-pro'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 700,
                    ],
                    'range' => [
                        'px' => [
                            'max' => 1920,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} ul.htmega-breadcrumb.htmega-breadcrumb-7' => 'max-width: {{SIZE}}{{UNIT}};',
                    ],
                    'condition'=>[
                        'breadcrumbs_style' => '7',
                    ]
                ]
            );

            $this->add_control(
                'breadcrumbs_items_spacing',
                [
                    'label' => __('Spacing', 'htmega-pro'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 25,
                    ],
                    'range' => [
                        'px' => [
                            'max' => 100,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} ul.htmega-breadcrumb li.htmega-breadcrumb-item:not(.htmega-breadcrumb-item:last-child) , {{WRAPPER}} ul.htmega-breadcrumb li.htmega-breadcrumb-separator' => 'margin-right: {{SIZE}}{{UNIT}};',
                    ],
                    'condition'=>[
                        'breadcrumbs_style!'=>'9',
                    ]
                ]
            );

            $this->add_control(
                'separator_border_color',
                [
                    'label' => __('Color', 'htmega-pro'),
                    'type' => Controls_Manager::COLOR,
                    'default' => '#dddddd',
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-8 .htmega-breadcrumb-separator::after' => 'border-color: {{VALUE}}',
                    ],
                    'condition' => [
                        'breadcrumbs_style' => '8',
                    ]
                ]
            );

        $this->end_controls_section();

        // Breadcrumbs Separator Style
        $this->start_controls_section(
            'section_separator',
            [
                'label' => __('Separator', 'htmega-pro'),
                'condition' => [
                    'breadcrumbs_style!' => '8',
                ],
            ]
        );

            $this->add_control(
                'separator_type',
                [
                    'label' => __('Separator Type', 'htmega-pro'),
                    'type' => Controls_Manager::SELECT,
                    'default' => 'icon',
                    'options' => [
                        'text' => __('Text', 'htmega-pro'),
                        'icon' => __('Icon', 'htmega-pro'),
                    ],
                ]
            );

            $this->add_control(
                'separator_text',
            [
                    'label' => __('Separator', 'htmega-pro'),
                    'type' => Controls_Manager::TEXT,
                    'default' => __('>', 'htmega-pro'),
                    'condition' => [
                        'separator_type' => 'text',
                    ]
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'breadcrumbs_separator_text_typography',
                    'label' => __('Typography', 'htmega-pro'),
                    'selector' => '{{WRAPPER}} li.htmega-breadcrumb-separator span.htmega-breadcrumb-separator-text',
                    'condition' => [
                        'separator_type' => 'text',
                    ],
                ]
            );

            $this->add_control(
                'separator_text_color',
                [
                    'label' => __('Color', 'htmega-pro'),
                    'type' => Controls_Manager::COLOR,
                    'default' => '',
                    'selectors' => [
                        '{{WRAPPER}} li.htmega-breadcrumb-separator span.htmega-breadcrumb-separator-text' => 'color: {{VALUE}}',
                    ],
                    'condition' => [
                        'separator_type' => 'text',
                    ]
                ]
            );

            $this->add_control(
                'select_separator_icon',
                [
                    'label' => __('Separator', 'htmega-pro'),
                    'type' => Controls_Manager::ICONS,
                    'label_block' => false,
                    'skin' => 'inline',
                    'fa4compatibility' => 'separator_icon',
                    'default' => [
                        'value' => 'fas fa-angle-right',
                        'library' => 'fa-solid',
                    ],
                    'recommended' => [
                        'fa-regular' => [
                            'circle',
                            'square',
                            'window-minimize',
                        ],
                        'fa-solid' => [
                            'angle-right',
                            'angle-double-right',
                            'caret-right',
                            'chevron-right',
                            'bullseye',
                            'circle',
                            'dot-circle',
                            'genderless',
                            'greater-than',
                            'grip-lines',
                            'grip-lines-vertical',
                            'minus',
                        ],
                    ],
                    'condition' => [
                        'separator_type' => 'icon',
                    ],
                ]
            );

            $this->add_control(
                'separator_icon_color',
                [
                    'label' => __('Color', 'htmega-pro'),
                    'type' => Controls_Manager::COLOR,
                    'default' => '',
                    'selectors' => [
                        '{{WRAPPER}} li.htmega-breadcrumb-separator i' => 'color: {{VALUE}}',
                        '{{WRAPPER}} li.htmega-breadcrumb-separator svg' => 'fill: {{VALUE}}',
                    ],
                    'condition' => [
                        'separator_type' => 'icon',
                    ],
                ]
            );

            $this->add_control(
                'separator_icon_size',
                [
                    'label' => __('Size', 'htmega-pro'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 13,
                    ],
                    'range' => [
                        'px' => [
                            'max' => 100,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} li.htmega-breadcrumb-separator i' => 'font-size: {{SIZE}}{{UNIT}};',
                        '{{WRAPPER}} li.htmega-breadcrumb-separator svg' => 'width: {{SIZE}}{{UNIT}};height: {{SIZE}}{{UNIT}};',
                    ],
                    'condition' => [
                        'separator_type' => 'icon',
                        'select_separator_icon[library]!' => ''
                    ],
                ]
            );

        $this->end_controls_section();

        // Item Button Style
        $this->start_controls_section(
            'breadcrumbs_area_style',
            array(
                'label' => __( 'Breadcrumbs Area', 'htmega-pro' ),
                'tab' => Controls_Manager::TAB_STYLE,
            )
        );

            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => 'item_area_background',
                    'label' => __( 'Background Color', 'htmega-pro' ),
                    'types' => [ 'classic', 'gradient' ],
                    'exclude' => ['image'],
                    'selector' => '{{WRAPPER}} .htmega-breadcrumb-box ul.htmega-breadcrumb',
                ]
            );

			$this->add_control(
                'only_style_four_color',
                [
                    'label' => __('Before/After Border Color', 'htmega-pro'),
                    'type' => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-4 .htmega-breadcrumb-item-current::before' => 'border-left-color: {{VALUE}};',
                    ],
                    'condition' => [
                        'breadcrumbs_style' => '4',
                    ]
                ]
            );  

			$this->add_control(
                'only_style_six_color',
                [
                    'label' => __('Before/After Border Color', 'htmega-pro'),
                    'type' => Controls_Manager::COLOR,
                    'selectors' => [
                            '{{WRAPPER}} .htmega-breadcrumb-6 .htmega-breadcrumb-item-current::before' => 'border-left-color: {{VALUE}}',
                            '{{WRAPPER}} .htmega-breadcrumb-6 .htmega-breadcrumb-item-current::after' => 'border-top-color: {{VALUE}};border-bottom-color: {{VALUE}};',
                    ],
                    'condition' => [
                        'breadcrumbs_style' => '6',
                    ]
                ]
            );      

			$this->add_control(
                'only_style_eight_color',
                [
                    'label' => __('Before/After Border Color', 'htmega-pro'),
                    'type' => Controls_Manager::COLOR,
                    'selectors' => [
                            '{{WRAPPER}} .htmega-breadcrumb-8 .htmega-breadcrumb-item-current::before' => 'border-top-color: {{VALUE}};border-bottom-color: {{VALUE}};',
                    ],
                    'condition' => [
                        'breadcrumbs_style' => '8',
                    ]
                ]
            );            

            $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => 'breadcrumbs_area_border',
                    'label' => __('Border', 'htmega-pro'),
                    'placeholder' => '1px',
                    'default' => '1px',
                    'selector' => '{{WRAPPER}} .htmega-breadcrumb-box ul.htmega-breadcrumb',
                    'separator' =>'before'
                ]
            );

            $this->add_responsive_control(
                'breadcrumbs_area_padding',
                [
                    'label' => __('Padding', 'htmega-pro'),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%'],
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-box ul.htmega-breadcrumb' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );

            $this->add_responsive_control(
                'breadcrumbs_area_border_radius',
                [
                    'label' => __('Border Radius', 'htmega-pro'),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%'],
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-box ul.htmega-breadcrumb' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );

        $this->end_controls_section();

        // Item Button Style
        $this->start_controls_section(
            'breadcrumbs_breadcrumbs_style',
            array(
                'label' => __( 'Breadcrumbs Items', 'htmega-pro' ),
                'tab' => Controls_Manager::TAB_STYLE,
            )
        );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'breadcrumbs_typography',
                    'label' => __('Typography', 'htmega-pro'),
                    'selector' => '{{WRAPPER}} .htmega-breadcrumb-item-link , {{WRAPPER}} li.htmega-breadcrumb-item.htmega-breadcrumb-item-current',
                ]
            );

            $this->start_controls_tabs('breadcrumbs_tabs_style');

                $this->start_controls_tab(
                    'breadcrumbs_tab_normal',
                    [
                        'label' => __('Normal', 'htmega-pro'),
                    ]
                );

                    $this->add_control(
                        'breadcrumbs_color',
                        [
                            'label' => __('Color', 'htmega-pro'),
                            'type' => Controls_Manager::COLOR,
                            'default' => '',
                            'selectors' => [
                                '{{WRAPPER}} .htmega-breadcrumb-item-link' => 'color: {{VALUE}}',
                                '{{WRAPPER}} .htmega-breadcrumb-item-link svg path' => 'fill: {{VALUE}}'
                            ],
                        ]
                    );
            
                    $this->add_control(
                        'breadcrumbs_background_color',
                        [
                            'label' => __('Background Color', 'htmega-pro'),
                            'type' => Controls_Manager::COLOR,
                            'default' => '',
                            'selectors' => [
                                '{{WRAPPER}} .htmega-breadcrumb-item-link' => 'background-color: {{VALUE}}',
                            ],
                        ]
                    );

                    $this->add_group_control(
                        Group_Control_Border::get_type(),
                        [
                            'name' => 'breadcrumbs_border',
                            'label' => __('Border', 'htmega-pro'),
                            'placeholder' => '1px',
                            'default' => '1px',
                            'selector' => '{{WRAPPER}} .htmega-breadcrumb-item-link',
                        ]
                    );

                $this->end_controls_tab();

                $this->start_controls_tab(
                    'breadcrumbs_tab_hover',
                    [
                        'label' => __('Hover', 'htmega-pro'),
                    ]
                );

                    $this->add_control(
                        'breadcrumbs_hover_color',
                        [
                            'label' => __('Color', 'htmega-pro'),
                            'type' => Controls_Manager::COLOR,
                            'default' => '',
                            'selectors' => [
                                '{{WRAPPER}} li.htmega-breadcrumb-item:hover a.htmega-breadcrumb-item-link' => 'color: {{VALUE}}',
                                '{{WRAPPER}} li.htmega-breadcrumb-item:hover a.htmega-breadcrumb-item-link svg path' => 'fill: {{VALUE}}'
                            ],
                        ]
                    );
            
                    $this->add_control(
                        'breadcrumbs_hover_background_color',
                        [
                            'label' => __('Background Color', 'htmega-pro'),
                            'type' => Controls_Manager::COLOR,
                            'default' => '',
                            'selectors' => [
                                '{{WRAPPER}} li.htmega-breadcrumb-item:hover a.htmega-breadcrumb-item-link' => 'background-color: {{VALUE}}',
                            ],
                        ]
                    );

                    $this->add_control(
                        'breadcrumbs_border_color_hover',
                        [
                            'label' => __('Border Color', 'htmega-pro'),
                            'type' => Controls_Manager::COLOR,
                            'default' => '',
                            'selectors' => [
                                '{{WRAPPER}} li.htmega-breadcrumb-item:hover a.htmega-breadcrumb-item-link' => 'border-color: {{VALUE}}',
                            ],
                        ]
                    );

                $this->end_controls_tab();

            $this->end_controls_tabs();

            $this->add_responsive_control(
                'breadcrumbs_padding',
                [
                    'label' => __('Padding', 'htmega-pro'),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%'],
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-item-link , {{WRAPPER}} li.htmega-breadcrumb-item.htmega-breadcrumb-item-current' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                    'separator' => 'before',
                ]
            );

            $this->add_responsive_control(
                'breadcrumbs_border_radius',
                [
                    'label' => __('Border Radius', 'htmega-pro'),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%'],
                    'selectors' => [
                        '{{WRAPPER}} .htmega-breadcrumb-item-link' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );

        $this->end_controls_section();

        // Item Button Style
        $this->start_controls_section(
            'breadcrumb_last_item',
            array(
                'label' => __( 'Last Item', 'htmega-pro' ),
                'tab' => Controls_Manager::TAB_STYLE,
            )
        );

            $this->add_control(
                'last_item_color',
                [
                    'label' => __('Color', 'htmega-pro'),
                    'type' => Controls_Manager::COLOR,
                    'default' => '',
                    'selectors' => [
                        '{{WRAPPER}} li.htmega-breadcrumb-item.htmega-breadcrumb-item-current' => 'color: {{VALUE}}',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => 'last_item_background_color',
                    'label' => __( 'Background Color', 'htmega-pro' ),
                    'types' => [ 'classic', 'gradient' ],
                    'exclude' => ['image'],
                    'selector' => '{{WRAPPER}} li.htmega-breadcrumb-item.htmega-breadcrumb-item-current',
                    'condition'=>[
                        'breadcrumbs_style!' => ['3','4','6','8'],
                    ]
                ]
            );

            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => 'last_item_background_gradient',
                    'label' => __( 'Background Color', 'htmega-pro' ),
                    'types' => [ 'gradient' ],
                    'exclude' => ['image','classic'],
                    'selector' => '{{WRAPPER}} li.htmega-breadcrumb-item.htmega-breadcrumb-item-current',
                    'condition'=>[
                        'breadcrumbs_style' => ['3','4','6','8'],
                    ]
                ]
            );

            $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => 'last_item_border',
                    'label' => __('Border', 'htmega-pro'),
                    'placeholder' => '1px',
                    'default' => '1px',
                    'selector' => '{{WRAPPER}} li.htmega-breadcrumb-item.htmega-breadcrumb-item-current',
                    'separator' => 'before',
                ]
            );     

            $this->add_responsive_control(
                'last_item_padding',
                [
                    'label' => __('Padding', 'htmega-pro'),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%'],
                    'selectors' => [
                        '{{WRAPPER}} li.htmega-breadcrumb-item.htmega-breadcrumb-item-current' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );   

            $this->add_responsive_control(
                'last_item_radius',
                [
                    'label' => __('Border Radius', 'htmega-pro'),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%'],
                    'selectors' => [
                        '{{WRAPPER}} li.htmega-breadcrumb-item.htmega-breadcrumb-item-current' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );

        $this->end_controls_section();

    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        extract($settings);

        $query = $this->get_query();
        if ($query) {
            if ($query->have_posts()) {

                $this->render_breadcrumbs($query);

                wp_reset_postdata();
            }
        } else {
            $this->render_breadcrumbs();
        }
    }

    protected function get_query() {

        $settings = $this->get_settings_for_display();

        global $post;

        $post_type = 'any';

        $args = array(
            'post_type' => $post_type,
        );

        $post_query = new \WP_Query($args);

        return false;
    }

    protected function render_breadcrumbs($query = false) {
        $settings = $this->get_settings_for_display();

        $this->add_render_attribute('breadcrumbs-parent', 'class', 'htmega-breadcrumb-box');
        $this->add_render_attribute('breadcrumbs', 'class', array('htmega-breadcrumb', 'htmega-breadcrumb-' . esc_attr( $settings['breadcrumbs_style'] ) ));

        // If you have any custom post types with custom taxonomies, put the taxonomy name below (e.g. product_cat)
        $custom_taxonomy = 'product_cat';

        // Get the query & post information
        global $post, $wp_query;

        if (false === $query) {
            // Reset post data to parent query
            $wp_query->reset_postdata();

            // Set active query to native query
            $query = $wp_query;
        }

        // Do not display on the homepage
        if (!$query->is_front_page()) {

            // Build the breadcrums
            echo '<div ' . $this->get_render_attribute_string('breadcrumbs-parent') . '>';
            echo '<ul ' . $this->get_render_attribute_string('breadcrumbs') . '>';

            // Home page
            if ('yes' === $settings['show_home']) {
                $this->render_home_link();
            }

            if ($query->is_archive() && !$query->is_tax() && !$query->is_category() && !$query->is_tag()) {

                echo '<li>' . post_type_archive_title('', false) . '</li>';

            } elseif ($query->is_archive() && $query->is_tax() && !$query->is_category() && !$query->is_tag()) {

                // If post is a custom post type
                $post_type = get_post_type();

                // If it is a custom post type display name and link
                if ('post' !== $post_type) {

                    $post_type_object = get_post_type_object($post_type);
                    $post_type_archive = get_post_type_archive_link($post_type);

                    $this->add_render_attribute(
                        array(
                            'breadcrumbs-item-cpt' => array(
                                'class' => array(
                                    'htmega-breadcrumb-item',
                                    'htmega-breadcrumb-item-custom-post-type-' . $post_type,
                                ),
                            ),
                            'breadcrumbs-item-cpt-crumb' => array(
                                'class' => array(
                                    'htmega-breadcrumb-crumb-link',
                                    'htmega-breadcrumb-crumb-custom-post-type-' . $post_type,
                                ),
                                'href' => $post_type_archive,
                                'title' => $post_type_object->labels->name,
                            ),
                        )
                    );

                    echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-cpt') . '><a ' . $this->get_render_attribute_string('breadcrumbs-item-cpt-crumb') . '>' . esc_html( $post_type_object->labels->name ) . '</a></li>';

                    $this->render_separator();

                }

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-tax' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                                'htmega-breadcrumb-item-current',
                                'htmega-breadcrumb-item-archive',
                            ),
                        ),
                    )
                );

                $custom_tax_name = get_queried_object()->name;

                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-tax') . '>' . esc_html( $custom_tax_name ) . '</li>';

            } elseif ($query->is_single()) {

                $post_type = get_post_type();

                if ('post' !== $post_type) {

                    $post_type_object = get_post_type_object($post_type);
                    $post_type_archive = get_post_type_archive_link($post_type);

                    $this->add_render_attribute(
                        array(
                            'breadcrumbs-item-cpt' => array(
                                'class' => array(
                                    'htmega-breadcrumb-item',
                                    'htmega-breadcrumb-item-custom-post-type-' . esc_attr( $post_type ),
                                ),
                            ),
                            'breadcrumbs-item-cpt-crumb' => array(
                                'class' => array(
                                    'htmega-breadcrumb-crumb',
                                    'htmega-breadcrumb-crumb-custom-post-type-' . esc_attr( $post_type ),
                                ),
                                'href' => $post_type_archive,
                                'title' => $post_type_object->labels->name,
                            ),
                        )
                    );

                }

                // Get post category info
                $category = get_the_category();

                if (!empty($category)) {

                    // Get last category post is in
                    $values = array_values($category);
                    $last_category = reset($values);

                    $categories = array();
                    $get_cat_parents = rtrim(get_category_parents($last_category->term_id, true, ','), ',');
                    $cat_parents = explode(',', $get_cat_parents);
                    foreach ($cat_parents as $parent) {
                        $categories[] = get_term_by('name', $parent, 'category');
                    }

                    // Loop through parent categories and store in variable $cat_display
                    $cat_display = '';

                    foreach ($categories as $parent) {
                        if (!is_wp_error(get_term_link($parent))) {
                            $cat_display .= '<li class="htmega-breadcrumb-item htmega-breadcrumb-item-cat"><a class="htmega-breadcrumb-item-link" href="' . get_term_link($parent) . '">' . esc_html( $parent->name ) . '</a></li>';
                            $cat_display .= $this->render_separator(false);
                        }
                    }
                }

                // If it's a custom post type within a custom taxonomy
                $taxonomy_exists = taxonomy_exists($custom_taxonomy);
                $taxonomy_terms = [];
                if (empty($last_category) && !empty($custom_taxonomy) && $taxonomy_exists) {
                    $taxonomy_terms = get_the_terms($post->ID, $custom_taxonomy);
                }
                // Check if the post is in a category
                if (!empty($last_category)) {
                    echo $cat_display; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

                    $this->add_render_attribute(
                        array(
                            'breadcrumbs-item-post-cat' => array(
                                'class' => array(
                                    'htmega-breadcrumb-item',
                                    'htmega-breadcrumb-item-current',
                                    'htmega-breadcrumb-item-' . esc_attr( $post->ID ),
                                ),
                            )
                        )
                    );

                    echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-post-cat') . '">' . esc_html( get_the_title() ) . '</li>';

                    // Else if post is in a custom taxonomy
                } elseif ($taxonomy_terms) {

                    foreach ($taxonomy_terms as $index => $taxonomy) {
                        $cat_id = $taxonomy->term_id;
                        $cat_nicename = $taxonomy->slug;
                        $cat_link = get_term_link($taxonomy->term_id, $custom_taxonomy);
                        $cat_name = $taxonomy->name;

                        $this->add_render_attribute(
                            array(
                                'breadcrumbs-item-post-cpt' => array(
                                    'class' => array(
                                        'htmega-breadcrumb-item',
                                    ),
                                ),
                                'breadcrumbs-item-post-cpt-crumb' => array(
                                    'class' => array(
                                        'htmega-breadcrumb-crumb',
                                    ),
                                    'href' => $cat_link,
                                    'title' => $cat_name,
                                ),
                            )
                        );

                        echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-post-cpt') . '"><a ' . $this->get_render_attribute_string('breadcrumbs-item-post-cpt-crumb') . '>' . esc_html( $cat_name ) . '</a></li>';

                        $this->render_separator();
                    }

                    $this->add_render_attribute(
                        array(
                            'breadcrumbs-item-post' => array(
                                'class' => array(
                                    'htmega-breadcrumb-item',
                                    'htmega-breadcrumb-item-' . esc_attr( $post->ID ),
                                ),
                            )
                        )
                    );

                    echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-post') . '">' . esc_html( get_the_title() ) . '</li>';

                } else {

                    $this->add_render_attribute(
                        array(
                            'breadcrumbs-item-post' => array(
                                'class' => array(
                                    'htmega-breadcrumb-item',
                                    'htmega-breadcrumb-item-current',
                                    'htmega-breadcrumb-item-' . esc_attr( $post->ID ),
                                ),
                            )
                        )
                    );

                    echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-post') . '">' . esc_html( get_the_title() ) . '</li>';

                }
            } elseif ($query->is_category()) {

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-cat' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        )
                    )
                );

                // Category page
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-cat') . '>' . single_cat_title('', false) . '</li>';

            } elseif ($query->is_page()) {

                // Standard page
                if ($post->post_parent) {

                    // If child page, get parents
                    $anc = get_post_ancestors($post->ID);

                    // Get parents in the right order
                    $anc = array_reverse($anc);

                    // Parent page loop
                    if (!isset($parents)) {
                        $parents = null;
                    }
                    foreach ($anc as $ancestor) {
                        $parents .= '<li class="htmega-breadcrumb-item htmega-breadcrumb-item-parent-' . esc_attr( $ancestor ) . '"><a class="htmega-breadcrumb-item-link htmega-breadcrumb-cb-parent-' . esc_attr( $ancestor ) . '" href="' . get_permalink($ancestor) . '" title="' . get_the_title($ancestor) . '">' . get_the_title($ancestor) . '</a></li>';

                        $parents .= $this->render_separator(false);
                    }

                    // Display parent pages
                    echo $parents; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

                }

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-page' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                                'htmega-breadcrumb-item-current',
                                'htmega-breadcrumb-item-' . esc_attr( $post->ID ),
                            ),
                        )
                    )
                );

                // Just display current page if not parents
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-page') . '>' . esc_html( get_the_title() ) . '</li>';

            } elseif ($query->is_tag()) {

                // Tag page

                // Get tag information
                $term_id = get_query_var('tag_id');
                $taxonomy = 'post_tag';
                // $args = 'include=' . $term_id;
                // $terms = get_terms($taxonomy, $args);
                $terms = get_terms([
                    'taxonomy' => $taxonomy,
                    'include' => $term_id,
                ]);
                $get_term_id = $terms[0]->term_id;
                $get_term_slug = $terms[0]->slug;
                $get_term_name = $terms[0]->name;

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-tag' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                                'htmega-breadcrumb-item-current',
                                'htmega-breadcrumb-item-tag-' . esc_attr( $get_term_id ),
                                'htmega-breadcrumb-item-tag-' . esc_attr( $get_term_slug ),
                            ),
                        )
                    )
                );

                // Display the tag name
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-tag') . '>' . esc_html( $get_term_name ). '</li>';

            } elseif ($query->is_day()) {

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-year' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        ),
                        'breadcrumbs-item-year-crumb' => array(
                            'class' => array(
                                'htmega-breadcrumb-crumb',
                            ),
                            'href' => get_year_link(get_the_time('Y')),
                            'title' => get_the_time('Y'),
                        ),
                        'breadcrumbs-item-month' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        ),
                        'breadcrumbs-item-month-crumb' => array(
                            'class' => array(
                                'htmega-breadcrumb-crumb',
                            ),
                            'href' => get_month_link(get_the_time('Y'), get_the_time('m')),
                            'title' => get_the_time('M'),
                        ),
                        'breadcrumbs-item-day' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        )
                    )
                );

                // Year link
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-year') . '><a ' . $this->get_render_attribute_string('breadcrumbs-item-year-crumb') . '>' . esc_html( get_the_time('Y') ) . ' ' . esc_html__('Archives', 'htmega-pro') . '</a></li>';

                $this->render_separator();

                // Month link
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-month') . '><a ' . $this->get_render_attribute_string('breadcrumbs-item-month-crumb') . '>' . esc_html( get_the_time('M') ) . ' ' . esc_html__('Archives', 'htmega-pro') . '</a></li>';

                $this->render_separator();

                // Day display
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-day') . '>' . esc_html( get_the_time('jS') ) . ' ' . esc_html( get_the_time('M') ) . ' ' . esc_html__('Archives', 'htmega-pro') . '</li>';

            } elseif ($query->is_month()) {

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-year' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        ),
                        'breadcrumbs-item-year-crumb' => array(
                            'class' => array(
                                'htmega-breadcrumb-crumb',
                            ),
                            'href' => get_year_link(get_the_time('Y')),
                            'title' => get_the_time('Y'),
                        ),
                        'breadcrumbs-item-month' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        ),
                    )
                );

                // Year link
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-year') . '>' . esc_html( get_the_time('Y') ) . ' ' . esc_html__('Archives', 'htmega-pro') . '</li>';

                $this->render_separator();

                // Month display
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-month') . '>' . esc_html( get_the_time('M') ) . ' ' . esc_html__('Archives', 'htmega-pro') . '</li>';

            } elseif ($query->is_year()) {

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-year' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        ),
                    )
                );

                // Display year archive
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-year') . '>' . esc_html( get_the_time('Y') ) . ' ' . esc_html__('Archives', 'htmega-pro') . '</li>';

            } elseif ($query->is_author()) {

                // Get the author information
                global $author;
                $userdata = get_userdata($author);

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-author' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        )
                    )
                );

                // Display author name
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-author') . '>' . esc_html__('Author:', 'htmega-pro') . ' ' . esc_html( $userdata->display_name ) . '</li>';

            } elseif (get_query_var('paged')) {

                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-paged' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        )
                    )
                );

                // Paginated archives
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-paged') . '>' . esc_html__('Page', 'htmega-pro') . ' ' . esc_html( get_query_var('paged') ) . '</li>';

            } elseif ($query->is_search()) {

                // Search results page
                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-search' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        )
                    )
                );

                // Search results page
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-search') . '>' . esc_html__('Search results for:', 'htmega-pro') . ' ' . esc_html( get_search_query() ) . '</li>';

            } elseif ($query->is_404()) {
                $this->add_render_attribute(
                    array(
                        'breadcrumbs-item-error' => array(
                            'class' => array(
                                'htmega-breadcrumb-item',
                            ),
                        ),
                    )
                );

                // 404 page
                echo '<li ' . $this->get_render_attribute_string('breadcrumbs-item-error') . '>' . esc_html__('Error 404', 'htmega-pro') . '</li>';
            }

            echo '</ul>';
            echo '</div>';

        }

    }
    
    protected function get_separator() {
        $settings = $this->get_settings_for_display();

        ob_start();
        if ('icon' === $settings['separator_type']) {

            if (!isset($settings['separator_icon']) && !\Elementor\Icons_Manager::is_migration_allowed()) {
                // add old default
                $settings['separator_icon'] = 'fas fa-angle-right';
            }

            $has_icon = !empty($settings['separator_icon']);

            if ($has_icon) {
                $this->add_render_attribute('separator-icon', 'class', $settings['separator_icon']);
                $this->add_render_attribute('separator-icon', 'aria-hidden', 'true');
            }

            if (!$has_icon && !empty($settings['select_separator_icon']['value'])) {
                $has_icon = true;
            }
            $migrated = isset($settings['__fa4_migrated']['select_separator_icon']);
            $is_new = !isset($settings['separator_icon']) && \Elementor\Icons_Manager::is_migration_allowed();

                if ($has_icon) {
                    ?>
                        <?php
                        if ($is_new || $migrated) {
                            \Elementor\Icons_Manager::render_icon($settings['select_separator_icon'], array('aria-hidden' => 'true'));
                        } elseif (!empty($settings['separator_icon'])) { ?>
                            <i <?php echo $this->get_render_attribute_string('separator-icon'); ?>></i>
                        <?php }
                        ?>
                    <?php
                }
            } else {

            $this->add_inline_editing_attributes('separator_text');
            $this->add_render_attribute('separator_text', 'class', 'htmega-breadcrumb-separator-text');

            echo '<span ' . $this->get_render_attribute_string('separator_text') . '>' . wp_kses_post( $settings['separator_text'] ) . '</span>';

        }
        
        $separator = ob_get_contents();
        ob_end_clean();

        return $separator;
    }

    protected function render_separator($output = true)
    {
        $settings = $this->get_settings_for_display();
        
        if($settings['breadcrumbs_style'] != '9'){
            $html = '<li class="htmega-breadcrumb-separator">';
            $html .= $this->get_separator();
            $html .= '</li>';
    
            if (true === $output) {
                echo $html;
                return;
            }

            return $html;
        }
    }

    protected function render_home_link()
    {
        $settings = $this->get_settings_for_display();

        $this->add_render_attribute(
            array(
                'home_item' => array(
                    'class' => array(
                        'htmega-breadcrumb-item',
                        'htmega-breadcrumb-item-home',
                    ),
                ),
                'home_link' => array(
                    'class' => array(
                        'htmega-breadcrumb-item-link'
                    ),
                    'href' => get_home_url(),
                    'title' => esc_html( $settings['home_text'] ),
                )
            )
        );

        if (!isset($settings['home_icon']) && !\Elementor\Icons_Manager::is_migration_allowed()) {
            // add old default
            $settings['home_icon'] = 'fa fa-home';
        }

        $has_home_icon = !empty($settings['home_icon']);

        if ($has_home_icon) {
            $this->add_render_attribute('i', 'class', esc_attr( $settings['home_icon'] ) );
            $this->add_render_attribute('i', 'aria-hidden', 'true');
        }

        if (!$has_home_icon && !empty($settings['select_home_icon']['value'])) {
            $has_home_icon = true;
        }
        $migrated_home_icon = isset($settings['__fa4_migrated']['select_home_icon']);
        $is_new_home_icon = !isset($settings['home_icon']) && \Elementor\Icons_Manager::is_migration_allowed();
        ?>
        <li <?php echo $this->get_render_attribute_string('home_item'); ?>>
            <a <?php echo $this->get_render_attribute_string('home_link'); ?>>
                <?php if (!empty($settings['home_icon']) || (!empty($settings['select_home_icon']['value']) && $is_new_home_icon)) {?>
                    <?php if ($is_new_home_icon || $migrated_home_icon) {
                            \Elementor\Icons_Manager::render_icon($settings['select_home_icon'], array('aria-hidden' => 'true'));
                        } elseif (!empty($settings['home_icon'])) { ?>
                            <i <?php echo $this->get_render_attribute_string('i'); ?>></i>
                    <?php } ?>
                <?php }?>
                <?php echo esc_html( $settings['home_text'] ); ?>
            </a>
        </li>
        <?php

        $this->render_separator();
    }

}