<?php
namespace HTMegaPro\Elementor\Widget;
use Elementor\Plugin as Elementor;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Css_Filter;
use Elementor\Repeater;
use Elementor\Group_Control_Typography;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class HTMega_Event_Calendar_Element extends Widget_Base {
    public function get_name() {
        return 'htmega-event-calendar-addons';
    }
    
    public function get_title() {
        return __( 'Event Calendar', 'htmega-pro' );
    }

    public function get_icon() {
        return 'htmega-icon eicon-calendar';
    }

    public function get_categories() {
        return [ 'htmegapro-addons' ];
    }

    public function get_script_depends() {
        return [ 'full-calendar','dayjs','locales-all' ];
    }

    public function get_style_depends() {
        return ['full-calendar'];
    }

    public function get_keywords() {
        return [ 'htmega', 'ht mega', 'event', 'calendar', 'event calendar','google calender' ];
    }

    public function get_custom_help_url() {
		return 'https://wphtmega.com/docs/general-widgets/event-calendar-widget/';
	}
    
    protected function register_controls() {

        // Content
        $this->start_controls_section(
            'content_section',
            [
                'label' => __( 'Events', 'htmega-pro' ),
            ]
        );
            
            $this->add_control(
                'source_type',
                [
                    'label' => __( 'Source', 'htmega-pro' ),
                    'type' => Controls_Manager::SELECT,
                    'default' => 'custom',
                    'options' => [
                        'custom'      => __( 'Custom', 'htmega-pro' ),
                        'google'      => __( 'Google', 'htmega-pro' ),
                    ],
                ]
            );

            $repeater = new Repeater();

            $repeater->add_control(
                'title',
                [
                    'label' => esc_html__( 'Title', 'htmega-pro' ),
                    'type' => Controls_Manager::TEXT,
                    'default' => esc_html__( 'Event Title', 'htmega-pro' ),
                    'placeholder' => esc_html__( 'Type your event title here', 'htmega-pro' ),
                    'label_block'=>true,
                    'dynamic' => [
                        'active' => true,
                    ],
                ]
            );

            $repeater->add_control(
                'all_day',
                [
                    'label'        => esc_html__('All Day', 'htmega-pro'),
                    'type'         => Controls_Manager::SWITCHER,
                    'label_block'  => false,
                    'return_value' => 'yes',
                ]
            );

            $repeater->add_control(
                'start_date',
                [
                    'label'     => esc_html__('Start Date', 'htmega-pro'),
                    'type'      => Controls_Manager::DATE_TIME,
                    'default'   => gmdate('Y-m-d H:i', current_time('timestamp', 0)),
                    'condition' => [
                        'all_day' => '',
                    ],
                ]
            );

            $repeater->add_control(
                'end_date',
                [
                    'label'     => esc_html__('End Date', 'htmega-pro'),
                    'type'      => Controls_Manager::DATE_TIME,
                    'default'   => gmdate('Y-m-d H:i', strtotime("+59 minute", current_time('timestamp', 0))),
                    'condition' => [
                        'all_day' => '',
                    ],
                ]
            );

            $repeater->add_control(
                'allday_start_date',
                [
                    'label'          => esc_html__('Start Date', 'htmega-pro'),
                    'type'           => Controls_Manager::DATE_TIME,
                    'picker_options' => ['enableTime' => false],
                    'default'        => gmdate('Y-m-d', current_time('timestamp', 0)),
                    'condition'      => [
                        'all_day' => 'yes',
                    ],
                ]
            );

            $repeater->add_control(
                'allday_end_date',
                [
                    'label'          => esc_html__('End Date', 'htmega-pro'),
                    'type'           => Controls_Manager::DATE_TIME,
                    'picker_options' => ['enableTime' => false],
                    'default'        => gmdate('Y-m-d', current_time('timestamp', 0)),
                    'condition'      => [
                        'all_day' => 'yes',
                    ],
                ]
            );
            $repeater->add_control(
                'location',
                [
                    'label' => esc_html__( 'Location', 'htmega-pro' ),
                    'type' => Controls_Manager::TEXT,
                    'placeholder' => esc_html__( 'Type your event location here', 'htmega-pro' ),
                    'label_block'=>true,
                    'dynamic' => [
                        'active' => true,
                    ],
                ]
            );

            $repeater->add_control(
                'description',
                [
                    'label' => esc_html__( 'Description', 'htmega-pro' ),
                    'type' => Controls_Manager::WYSIWYG,
                    'default' => esc_html__( 'BioLorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Quis ipsum suspendisse ultrices gravida.Chemistry','htmega-pro' ),
                    'dynamic' => [
                        'active' => true,
                    ],
                ]
            );
            $repeater->add_control(
                'custom_style',
                [
                    'label'        => esc_html__('Custom Style', 'htmega-pro'),
                    'type'         => Controls_Manager::SWITCHER,
                    'label_block'  => false,
                    'return_value' => 'yes',
                ]
            );
            $repeater->add_control(
                'bg_color',
                [
                    'label'   => esc_html__('Background Color', 'htmega-pro'),
                    'type'    => Controls_Manager::COLOR,
                    'condition'      => [
                        'custom_style' => 'yes',
                    ],
                ]
            );

            $repeater->add_control(
                'text_color',
                [
                    'label'   => esc_html__('Text Color', 'htmega-pro'),
                    'type'    => Controls_Manager::COLOR,
                    'condition'      => [
                        'custom_style' => 'yes',
                    ],
                ]
            );
            $this->add_control(
                'event_list',
                [
                    'label'       => esc_html__('Event', 'htmega-pro'),
                    'type'        => Controls_Manager::REPEATER,
                    'fields'      => $repeater->get_controls(),
                    'default'     => [
                        [
                            'title' => esc_html__( 'Event Title','htmega-pro' ),
                            'description' => esc_html__( 'BioLorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Quis ipsum suspendisse ultrices gravida.Chemistry','htmega-pro' ),
                        ],
                    ],
                    'condition'   => [
                        'source_type' => 'custom',
                    ],
                    'title_field' => '{{ title }}',
                ]
            );

        $this->add_control(
            'google_calendar_api_key',
            [
                'label'       => __('API Key', 'htmega-pro'),
                'label_block' => true,
                'type'        => Controls_Manager::TEXT,
                'default' => '',
                'description' => sprintf( // translators: %s represents the link to Google Cloud Console
                    esc_html__('<a href="https://console.cloud.google.com/apis/credentials" target="_blank">%s</a>','htmega-pro'), esc_html__( 'Get API Key','htmega-pro' ) ), 
                'condition'   => [
                    'source_type' => 'google',
                ],
            ]
        );

        $this->add_control(
            'google_calendar_id',
            [
                'label'       => __('Calendar ID', 'htmega-pro'),
                'label_block' => true,
                'type'        => Controls_Manager::TEXT,
                'default' => '',
                'condition'   => [
                    'source_type' => 'google',
                ],
            ]
        );

        $this->add_control(
            'google_start_date',
            [
                'label'     => esc_html__('Start Date', 'htmega-pro'),
                'type'      => Controls_Manager::DATE_TIME,
                'default'   => '',
                'condition' => [
                    'source_type' => 'google',
                ],
            ]
        );

        $this->add_control(
            'google_end_date',
            [
                'label'     => esc_html__('End Date', 'htmega-pro'),
                'type'      => Controls_Manager::DATE_TIME,
                'default'   =>'',
                'condition' => [
                    'source_type' => 'google',
                ],
            ]
        );
        $this->end_controls_section();
       // Setting 
       $this->start_controls_section(
        'events_settings',
        [
            'label' => esc_html__( 'Settings', 'htmega-pro' ),
        ]
    );

        $this->add_control(
            'default_view',
            [
                'label'   => esc_html__('Default View', 'htmega-pro'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'timeGridDay'  => esc_html__('Day', 'htmega-pro'),
                    'timeGridWeek' => esc_html__('Week', 'htmega-pro'),
                    'dayGridMonth' => esc_html__('Month', 'htmega-pro'),
                    'listMonth'    => esc_html__('List', 'htmega-pro'),
                ],
                'default' => 'dayGridMonth',
                'label_block' => true,
            ]
        );

        $this->add_control(
            'first_day',
            [
                'label'   => esc_html__('First Day of Week', 'htmega-pro'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    '0' => esc_html__('Sunday', 'htmega-pro'),
                    '1' => esc_html__('Monday', 'htmega-pro'),
                    '2' => esc_html__('Tuesday', 'htmega-pro'),
                    '3' => esc_html__('Wednesday', 'htmega-pro'),
                    '4' => esc_html__('Thursday', 'htmega-pro'),
                    '5' => esc_html__('Friday', 'htmega-pro'),
                    '6' => esc_html__('Saturday', 'htmega-pro'),
                ],
                'default' => '0',
                'label_block' => true,
            ]
        );

        $this->add_control(
            'language',
            [
                'label'   => esc_html__('Language', 'htmega-pro'),
                'type'    => Controls_Manager::SELECT,
                'options' => htmega_addons_language_code(),
                'default' => 'en',
                'label_block' => true,
            ]
        );

    $this->end_controls_section();
        // Style
        // Toolbar Style tab section
        $this->start_controls_section(
            'calendar_toolbar_style',
            [
                'label' => esc_html__( 'Toolbar', 'htmega-pro' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
            
            $this->add_control(
                'toolbar_title_heading',
                [
                    'label' => esc_html__('Title', 'htmega-pro'),
                    'type'  => Controls_Manager::HEADING,
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name'     => 'toolbar_title_typography',
                    'label'    => esc_html__('Typography', 'htmega-pro'),
                    'selector' => '{{WRAPPER}} .htmega-event-calendar .fc-toolbar h2',
                ]
            );

            $this->add_control(
                'toolbar_title_color',
                [
                    'label'     => esc_html__('Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar .fc-toolbar h2' => 'color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_control(
                'toolbar_button_heading',
                [
                    'label'     => esc_html__('Button', 'htmega-pro'),
                    'type'      => Controls_Manager::HEADING,
                    'separator' => 'before',
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name'     => 'toolbar_button_typography',
                    'label'    => esc_html__('Typography', 'htmega-pro'),
                    'selector' => '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button',
                ]
            );

            $this->add_responsive_control(
                'toolbar_button_border_radius',
                [
                    'label'      => esc_html__('Border Radius', 'htmega-pro'),
                    'type'       => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%'],
                    'selectors'  => [
                        '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );

            $this->add_responsive_control(
                'toolbar_button_border_right_bg_radius',
                [
                    'label'      => esc_html__('Button Wrapper Border Radius', 'htmega-pro'),
                    'type'       => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%'],
                    'selectors'  => [
                        '{{WRAPPER}} .htmega-event-calendar .fc-header-toolbar .fc-toolbar-chunk:last-child' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );
            $this->add_control(
                'toolbar_button_background_right',
                [
                    'label'     => esc_html__('Button Wrapper Background', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar .fc-header-toolbar .fc-toolbar-chunk:last-child' => 'background-color: {{VALUE}};',
                    ],
                ]
            );
            $this->start_controls_tabs('toolbar_button_style_tabs');

                $this->start_controls_tab(
                    'toolbar_button_normal',
                    [
                        'label' => esc_html__( 'Normal', 'htmega-pro' ),
                    ]
                );
                    
                    $this->add_control(
                        'toolbar_button_color',
                        [
                            'label'     => esc_html__('Color', 'htmega-pro'),
                            'type'      => Controls_Manager::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button:not(.fc-button-active)' => 'color: {{VALUE}};',
                            ],
                        ]
                    );

                    $this->add_control(
                        'toolbar_button_background',
                        [
                            'label'     => esc_html__('Background', 'htmega-pro'),
                            'type'      => Controls_Manager::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button:not(.fc-button-active)' => 'background-color: {{VALUE}};',
                            ],
                        ]
                    );

                    $this->add_group_control(
                        Group_Control_Border::get_type(),
                        [
                            'name'     => 'toolbar_button_border',
                            'label'    => esc_html__('Border', 'htmega-pro'),
                            'selector' => '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button:not(.fc-button-active)',
                        ]
                    );
                $this->end_controls_tab();

                // Active Button Style
                $this->start_controls_tab(
                    'toolbar_button_active',
                    [
                        'label' => esc_html__( 'Active', 'htmega-pro' ),
                    ]
                );
                    $this->add_control(
                        'button_color_active',
                        [
                            'label'     => esc_html__('Color', 'htmega-pro'),
                            'type'      => Controls_Manager::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-button-active,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button-primary:not(:disabled):hover' => 'color: {{VALUE}};',
                            ],
                        ]
                    );

                    $this->add_control(
                        'button_background_active',
                        [
                            'label'     => esc_html__('Background', 'htmega-pro'),
                            'type'      => Controls_Manager::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-button-active,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button-primary:not(:disabled):hover' => 'background-color: {{VALUE}};',
                            ],
                        ]
                    );

                    $this->add_group_control(
                        Group_Control_Border::get_type(),
                        [
                            'name'     => 'button_border_active',
                            'label'    => esc_html__('Border', 'htmega-pro'),
                            'selector' => '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-button-active,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button-primary:not(:disabled):hover',
                        ]
                    );

                $this->end_controls_tab();

            $this->end_controls_tabs();
            $this->add_control(
                'separate_next_prev_style',
                [
                    'label'        => esc_html__('Different Style For Next/Prev Button?', 'htmega-pro'),
                    'type'         => Controls_Manager::SWITCHER,
                    'label_block'  => false,
                    'default' =>'no',
                    'return_value' => 'yes',
                    'separator' =>'before'
                ]
            );


            $this->start_controls_tabs('toolbar_button_style_tabsnext_prev',
                [
                    'condition' =>[
                        'separate_next_prev_style' =>'yes'
                    ]
                ]
            );

                $this->start_controls_tab(
                    'toolbar_button_normalnext_prev',
                    [
                        'label' => esc_html__( 'Normal', 'htmega-pro' ),
                    ]
                );
                    
                $this->add_control(
                    'toolbar_button_color_next_prev',
                    [
                        'label'     => esc_html__('Color', 'htmega-pro'),
                        'type'      => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-prev-button,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-next-button,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-today-button:disabled' => 'color: {{VALUE}};',
                        ],
                    ]
                );
    
                $this->add_control(
                    'toolbar_button_background_next_prev',
                    [
                        'label'     => esc_html__('Background', 'htmega-pro'),
                        'type'      => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-prev-button,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-next-button,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-today-button:disabled' => 'background-color: {{VALUE}};',
                        ],
                    ]
                );
    
                $this->add_group_control(
                    Group_Control_Border::get_type(),
                    [
                        'name'     => 'toolbar_button_border_next_prev',
                        'label'    => esc_html__('Border', 'htmega-pro'),
                        'selector' => '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-prev-button,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-next-button,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-today-button:disabled',
                    ]
                );
                $this->add_responsive_control(
                    'toolbar_button_border_next_prev_radius',
                    [
                        'label'      => esc_html__('Border Radius', 'htmega-pro'),
                        'type'       => Controls_Manager::DIMENSIONS,
                        'size_units' => ['px', '%'],
                        'selectors'  => [
                            '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-prev-button,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-next-button,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-today-button:disabled' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                    ]
                );
                $this->end_controls_tab();

                // Active Button Style
                $this->start_controls_tab(
                    'toolbar_button_active_next_prev',
                    [
                        'label' => esc_html__( 'Active', 'htmega-pro' ),
                    ]
                );
                    $this->add_control(
                        'button_color_active_next_prev',
                        [
                            'label'     => esc_html__('Color', 'htmega-pro'),
                            'type'      => Controls_Manager::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-prev-button:hover,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-next-button:hover,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-today-button' => 'color: {{VALUE}};',
                            ],
                        ]
                    );

                    $this->add_control(
                        'button_background_active_next_prev',
                        [
                            'label'     => esc_html__('Background', 'htmega-pro'),
                            'type'      => Controls_Manager::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-prev-button:hover,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-next-button:hover,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-today-button' => 'background-color: {{VALUE}};',
                            ],
                        ]
                    );

                    $this->add_group_control(
                        Group_Control_Border::get_type(),
                        [
                            'name'     => 'button_border_active_next_prev',
                            'label'    => esc_html__('Border', 'htmega-pro'),
                            'selector' => '{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-prev-button:hover,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-next-button:hover,{{WRAPPER}} .htmega-event-calendar .fc-toolbar.fc-header-toolbar .fc-button.fc-today-button',
                        ]
                    );

                $this->end_controls_tab();

            $this->end_controls_tabs();
        $this->end_controls_section();

        // Day Style
        $this->start_controls_section(
            'calendar_day',
            [
                'label' => esc_html__('Day', 'htmega-pro'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );
            
            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name'     => 'day_background',
                    'label'    => esc_html__('Background', 'htmega-pro'),
                    'types'    => ['classic', 'gradient'],
                    'selector' => '{{WRAPPER}} .htmega-calender .fc th',
                    'exclude'  => [
                        'image'
                    ],
                    'fields_options'=>[
                        'color'=>[
                            'label'=> esc_html__( 'Background Color', 'htmega-pro' ),
                        ],
                    ],
                ]
            );

            $this->add_control(
                'day_border_color',
                [
                    'label'     => esc_html__('Border Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar.fc .fc-col-header .fc-col-header-cell' => 'border-color: {{VALUE}};',

                    ],
                ]
            );

            $this->add_control(
                'day_color',
                [
                    'label'     => esc_html__('Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar.fc .fc-col-header,{{WRAPPER}} .htmega-event-calendar.fc .fc-col-header .fc-col-header-cell *' => 'color: {{VALUE}} !important;',

                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name'     => 'day_typography',
                    'label'    => esc_html__('Typography', 'htmega-pro'),
                    'selector' => '{{WRAPPER}} .htmega-event-calendar.fc .fc-col-header',
                ]
            );

            $this->add_responsive_control(
                'day_alignment',
                [
                    'label'     => esc_html__('Alignment', 'htmega-pro'),
                    'type'      => Controls_Manager::CHOOSE,
                    'options'   => [
                        'left'   => [
                            'title' => esc_html__('Left', 'htmega-pro'),
                            'icon'  => 'eicon-text-align-left',
                        ],
                        'center' => [
                            'title' => esc_html__('Center', 'htmega-pro'),
                            'icon'  => 'eicon-text-align-center',
                        ],
                        'right'  => [
                            'title' => esc_html__('Right', 'htmega-pro'),
                            'icon'  => 'eicon-text-align-right',
                        ],
                    ],
                    'default'   => 'center',
                    'selectors' => [
                        '{{WRAPPER}} .fc-col-header-cell .fc-scrollgrid-sync-inner' => 'text-align: {{VALUE}};',
                    ],
                ]
            );
            $this->add_responsive_control(
                'day_padding',
                [
                    'label' => esc_html__( 'Padding', 'htmega-pro' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'selectors' => [
                        '{{WRAPPER}} .fc-col-header-cell .fc-scrollgrid-sync-inner' => 'padding: {{TOP}}px {{RIGHT}}px {{BOTTOM}}px {{LEFT}}px;',
                    ],
                ]
            );
        $this->end_controls_section();

        // date Style
        $this->start_controls_section(
            'date_style',
            [
                'label' => esc_html__('Date', 'htmega-pro'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

            $this->add_control(
                'date_color',
                [
                    'label'     => esc_html__('Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar.fc .fc-daygrid-day-top .fc-daygrid-day-number' => 'color: {{VALUE}}',
                    ],
                ]
            );

            $this->add_control(
                'date_background',
                [
                    'type'      => Controls_Manager::COLOR,
                    'label'     => esc_html__('Background', 'htmega-pro'),
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar.fc .fc-daygrid-day' => 'background: {{VALUE}} !important',
                    ],
                ]
            );

            $this->add_control(
                'date_border_color',
                [
                    'type'      => Controls_Manager::COLOR,
                    'label'     => esc_html__('Border Color', 'htmega-pro'),
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar.fc .fc-daygrid-day' => 'border-color: {{VALUE}}',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name'     => 'date_typography',
                    'label'    => esc_html__('Typography', 'htmega-pro'),
                    'selector' => '{{WRAPPER}} .htmega-event-calendar.fc .fc-daygrid-day-top .fc-daygrid-day-number',
                ]
            );

            $this->add_responsive_control(
                'date_alignment',
                [
                    'label'     => esc_html__('Alignment', 'htmega-pro'),
                    'type'      => Controls_Manager::CHOOSE,
                    'options'   => [
                        'flex-end'   => [
                            'title' => esc_html__('Left', 'htmega-pro'),
                            'icon'  => 'eicon-text-align-left',
                        ],
                        'center' => [
                            'title' => esc_html__('Center', 'htmega-pro'),
                            'icon'  => 'eicon-text-align-center',
                        ],
                        'flex-start'  => [
                            'title' => esc_html__('Right', 'htmega-pro'),
                            'icon'  => 'eicon-text-align-right',
                        ],
                    ],
                    'default'   => 'center',
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar.fc .fc-daygrid-day-top' => 'justify-content: {{VALUE}};',
                    ],
                ]
            );

            $this->add_control(
                'date_today_heading',
                [
                    'label'     => esc_html__('Today', 'htmega-pro'),
                    'type'      => Controls_Manager::HEADING,
                    'separator' => 'before',
                ]
            );

            $this->add_control(
                'today_color',
                [
                    'label'     => esc_html__('Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar.fc .fc-daygrid-day.fc-day-today .fc-daygrid-day-number' => 'color: {{VALUE}}',
                    ],
                ]
            );

            $this->add_control(
                'today_background',
                [
                    'type'      => Controls_Manager::COLOR,
                    'label'     => esc_html__('Background', 'htmega-pro'),
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar.fc .fc-daygrid-day.fc-day-today,{{WRAPPER}} .htmega-calender .fc-timegrid-col.fc-day-today,{{WRAPPER}} .htmega-calender .fc-theme-standard .fc-list-day-cushion' => 'background: {{VALUE}}',
                    ],
                ]
            );

        $this->end_controls_section();

        // Event  style
        $this->start_controls_section(
            'event_style',
            [
                'label' => esc_html__('Events', 'htmega-pro'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );
            

            $this->add_control(
                'event_title_color',
                [
                    'label'     => esc_html__('Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender .fc-daygrid-event' => 'color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'event_title_typography',
                    'label' => esc_html__( 'Title Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .htmega-calender .fc-daygrid-event,{{WRAPPER}} .htmega-calender .fc-event-title',
                ]
            );
            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'event_time_typography',
                    'label' => esc_html__( 'Time Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .fc-timegrid-event .fc-event-time,{{WRAPPER}} .htmega-calender .fc-event-time',
                ]
            );

            $this->add_control(
                'event_background_color',
                [
                    'label'     => esc_html__('Background Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender .fc-daygrid-event,{{WRAPPER}} .fc-v-event' => 'background-color: {{VALUE}};',
                        '{{WRAPPER}} .fc-v-event' => 'border-color: {{VALUE}};',
                    ],
                ]
            );
        $this->end_controls_section();
        // Pop Up style
        $this->start_controls_section(
            'popup_style',
            [
                'label' => esc_html__('Popup', 'htmega-pro'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );
            
            $this->add_control(
                'popup_header_heading',
                [
                    'label' => esc_html__('Header', 'htmega-pro'),
                    'type'  => Controls_Manager::HEADING,
                ]
            );

            $this->add_control(
                'popup_header_title_color',
                [
                    'label'     => esc_html__('Title Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender-modal-title' => 'color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'popup_title_typography',
                    'label' => esc_html__( 'Title Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .htmega-calender-modal-title',
                ]
            );

            $this->add_control(
                'popup_header_location_color',
                [
                    'label'     => esc_html__('Location Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender-modal-location-wrap' => 'color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'popup_location_typography',
                    'label' => esc_html__( 'Location Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .htmega-calender-modal-location-wrap',
                ]
            );

            $this->add_responsive_control(
                'popup_location_icon_size',
                [
                    'label'      => esc_html__('Location Icon Size', 'htmega-pro'),
                    'type'       => Controls_Manager::SLIDER,
                    'size_units' => ['px'],
                    'selectors'  => [
                        '{{WRAPPER}} .htmega-calender-modal-location-wrap i' => 'font-size: {{SIZE}}{{UNIT}};',
                    ],
                ]
            );

            $this->add_control(
                'popup_header_background_color',
                [
                    'label'     => esc_html__('Header Background Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender-modal-header' => 'background-color: {{VALUE}};',
                    ],
                    'separator' => 'after'
                ]
            );

            $this->add_control(
                'popup_date_heading',
                [
                    'label' => esc_html__('Date', 'htmega-pro'),
                    'type'  => Controls_Manager::HEADING,
                ]
            );

            $this->add_control(
                'popup_date_color',
                [
                    'label'     => esc_html__('Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender-modal-date, {{WRAPPER}} .htmega-calender-modal-event' => 'color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'popup_date_typography',
                    'label' => esc_html__( 'Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .htmega-calender-modal-date',
                ]
            );

            $this->add_responsive_control(
                'popup_date_icon_size',
                [
                    'label'      => esc_html__('Icon Size', 'htmega-pro'),
                    'type'       => Controls_Manager::SLIDER,
                    'size_units' => ['px'],
                    'selectors'  => [
                        '{{WRAPPER}} .htmega-calender-modal-event i' => 'font-size: {{SIZE}}{{UNIT}};',
                    ],
                    'separator' => 'after'
                ]
            );

            $this->add_control(
                'popup_content_heading',
                [
                    'label' => esc_html__('Content', 'htmega-pro'),
                    'type'  => Controls_Manager::HEADING,
                ]
            );

            $this->add_control(
                'popup_content_color',
                [
                    'label'     => esc_html__('Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender-modal-description-wrap' => 'color: {{VALUE}};',
                        '{{WRAPPER}} .htmega-calender-modal-description' => 'color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'popup_content_typography',
                    'label' => esc_html__( 'Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .htmega-calender-modal-description-wrap,{{WRAPPER}} .htmega-calender-modal-description',
                    
                ]
            );

            $this->add_control(
                'popup_close_icon_heading',
                [
                    'label' => esc_html__('Close Icon', 'htmega-pro'),
                    'type'  => Controls_Manager::HEADING,
                    'separator' => 'before'
                ]
            );

            $this->add_control(
                'popup_close_icon_color',
                [
                    'label'     => esc_html__('Close Icon Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender-close-modal' => 'color: {{VALUE}}; border-color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_control(
                'popup_close_icon_hover_color',
                [
                    'label'     => esc_html__('Hover Background', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender-close-modal:hover' => 'background-color: {{VALUE}};border-color: {{VALUE}};',
                    ],
                ]
            );
        $this->end_controls_section();
        // Week & Day View style
        $this->start_controls_section(
            'week_day_style',
            [
                'label' => esc_html__('Week & Day View', 'htmega-pro'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );
            

            $this->add_control(
                'day_date_color',
                [
                    'label'     => esc_html__('Day and Date Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .fc .fc-timegrid-axis-cushion,{{WRAPPER}} .fc .fc-timegrid-slot-label-cushion' => 'color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'day_date_typography',
                    'label' => esc_html__( 'Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .fc .fc-timegrid-axis-cushion,{{WRAPPER}} .fc .fc-timegrid-slot-label-cushion',
                ]
            );


            $this->add_control(
                'day_date_background_color',
                [
                    'label'     => esc_html__('Table Background Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender .fc-theme-standard .fc-scrollgrid' => 'background-color: {{VALUE}};',
                    ],
                ]
            );
            $this->add_control(
                'day_date_border_color',
                [
                    'label'     => esc_html__('Border Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .fc-theme-standard td, {{WRAPPER}} .fc-theme-standard th, {{WRAPPER}} table thead:first-child tr:first-child th,{{WRAPPER}} .htmega-calender .fc-theme-standard .fc-scrollgrid' => 'border-color: {{VALUE}};',
                    ],
                ]
            );
            
            $this->add_control(
                'day_date_toeday_background_color',
                [
                    'label'     => esc_html__('Today Background Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-calender .fc-timegrid-col.fc-day-today' => 'background-color: {{VALUE}};',
                    ],
                ]
            );

        $this->end_controls_section();
        // List Style
        $this->start_controls_section(
            'calendar_list_view',
            [
                'label' => esc_html__('List view', 'htmega-pro'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

            $this->add_control(
                'list_view_header_heading',
                [
                    'label' => esc_html__('Header', 'htmega-pro'),
                    'type'  => Controls_Manager::HEADING,
                ]
            );

            $this->add_control(
                'list_view_header_color',
                [
                    'label'     => esc_html__('Text Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > th .fc-cell-shaded,{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > th .fc-cell-shaded a' => 'color: {{VALUE}};',
                    ],
                ]
            );
            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'list_view_header_typography',
                    'label' => esc_html__( 'Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > th .fc-cell-shaded,{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > th .fc-cell-shaded a',
                ]
            );
            $this->add_control(
                'list_header_background_color',
                [
                    'label'     => esc_html__('Background Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > th,{{WRAPPER}} .htmega-calender .fc-theme-standard .fc-list-day-cushion' => 'background-color: {{VALUE}};',
                    ],
                ]
            );

            $this->add_control(
                'list_view_content_heading',
                [
                    'label' => esc_html__('Content', 'htmega-pro'),
                    'type'  => Controls_Manager::HEADING,
                ]
            );

            $this->add_control(
                'list_view_text_color',
                [
                    'label'     => esc_html__('Text Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > td.fc-list-event-time' => 'color: {{VALUE}};',
                        '{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > td.fc-list-event-title' => 'color: {{VALUE}};',
                    ],
                ]
            );
            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'list_view_typography',
                    'label' => esc_html__( 'Typography', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > td.fc-list-event-title,{{WRAPPER}} .htmega-event-calendar .fc-list-table tr > td.fc-list-event-time',
                ]
            );
            $this->add_control(
                'list_bg_color',
                [
                    'label'     => esc_html__('Background Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .htmega-event-calendar .fc-list-table tr.fc-list-event td' => 'background-color: {{VALUE}};',
                    ],
                ]
            );
            $this->add_control(
                'list_dot_color',
                [
                    'label'     => esc_html__('Dot Color', 'htmega-pro'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .fc .fc-list-event-dot' => 'border-color: {{VALUE}};',
                    ],
                ]
            );

        $this->end_controls_section();

    }

    protected function render( $instance = [] ) {
        $settings   = $this->get_settings_for_display();
        $id         = $this->get_id();

        $this->add_render_attribute( 'area_attr', 'class', 'htmega-event-calendar' );
        $this->add_render_attribute( 'area_attr', 'id', 'htmega-event-calendar-'.$id );

        $event_list = [];
        if( $settings['source_type'] == 'custom' ){
            $events = $settings['event_list'];
            foreach ( $events as $event ) {

                if ($event['all_day'] == 'yes') {
                    $start  = $event["allday_start_date"];
                    $end    = gmdate('Y-m-d', strtotime("+1 days", strtotime($event["allday_end_date"])));
                    $pudate = gmdate( 'F d Y', strtotime( $event["allday_start_date"] ) );
                    $putime = '';
                } else {
                    $start  = $event["start_date"];
                    $end    = gmdate('Y-m-d H:i', strtotime($event["end_date"])).":01";
                    $pudate = gmdate( 'F d Y', strtotime( $event["start_date"] ) );
                    $putime = gmdate( 'g:i a', strtotime( $event["start_date"] ) ).' - '.gmdate( 'g:i a', strtotime( $event["end_date"] ) );
                }

                $title = ( !empty( $event['title'] ) ? $event['title'] : esc_html__( 'Event Title', 'htmega-pro' ) );
                $description = ( !empty( $event['description'] ) ? $event['description'] : '' );
                $location = ( !empty( $event['location'] ) ?$event['location'] : '' );

                $event_list[] = [
                    'title' => esc_html( $title ),
                    'start' => $start,
                    'end'   => $end,
                    'color' => esc_attr( $event['bg_color'] ),
                    'textColor' =>  esc_attr( $event['text_color'] ),
                    'display'   => 'block',
                    'extendedProps'=> [
                        'eventTitle'=> esc_html( $title ),
                        'location'=>  esc_html( $location ),
                        'date'=> $pudate,
                        'time'=> $putime,
                        'description'=> wp_kses_post( $description ),
                    ],
                ];
            }
        }

        ?>


        <div class="htmega-calender htmega-calender-01">
        <?php if (  $settings['source_type'] == 'google' && ( empty( $settings['google_calendar_api_key'] ) || empty( $settings['google_calendar_id'] ))) {
			$error_message = __('Please insert API key & Calendar ID.', 'htmega-pro');
			printf('<span class="htmega-error-message">%1$s</span>', esc_html( $error_message ) );
            return [];
		} else{ ?>
                <div <?php echo $this->get_render_attribute_string( 'area_attr' ); ?> >

                </div>
        <?php  }
            ?>
        
        </div>
 

        <script type="text/javascript">
            /* Event Calendar */
            ;jQuery(document).ready(function($) {
                var calendarEl = document.getElementById('htmega-event-calendar-<?php echo esc_js( $id ); ?>');
                // htmega Calendar Modal HTML Markup Js
                const htmegaCalendarModal = function(title, location, date, desc, backgroundColor, style) {
                    desc = (desc == undefined)?'':desc;
                    const markup = `<div class="htmega-calender-modal htmega-calender-modal-${style}">
                        <div class="htmega-calender-modal-content">
                            <div class="htmega-calender-modal-header" ${ backgroundColor && `style="background-color: ${backgroundColor}` }">
                                ${ title && `<h2 class="htmega-calender-modal-title">${title}</h2>`}
                                ${ location ? `<div class="htmega-calender-modal-location-wrap"><i class="fa fa-map-marker"></i> <span class="htmega-calender-modal-location">${location}</span>.</div>` : ''}
                                <button class="htmega-calender-close-modal">&times;</button>
                            </div>
                            <div class="htmega-calender-modal-body">
                                ${ date && `<div class="htmega-calender-modal-event"><i class="fa fa-calendar"></i><span class="htmega-calender-modal-date">${date}</span></div>`}
                                ${ desc && `<div class="htmega-calender-modal-description-wrap">
                                    <p class="htmega-calender-modal-description">${desc}</p>
                                </div>`}
                            </div>
                        </div>  
                    </div>
                    <div class="htmega-calender-overlay"></div>`
                    calendarEl.insertAdjacentHTML('beforeend', markup);
                }

                // Htmega Calendar Active Code 1
        
                var Calendar = FullCalendar.Calendar;
                let errorMessage = 0;
                var defaultview = '<?php echo esc_js( $settings['default_view'] ); ?>',
                            language = '<?php echo esc_js( $settings['language'] ); ?>',
                            firstDay = '<?php echo esc_js( $settings['first_day'] ); ?>';

                // initialize the calendar
                // -----------------------------------------------------------------
                var calendar = new Calendar(calendarEl, {

                    <?php  if( $settings['source_type'] == 'google' ){ ?>
                    googleCalendarApiKey:'<?php echo esc_js( $settings['google_calendar_api_key'] ); ?>',
                    validRange: {
                        start: '<?php echo esc_js( $settings['google_start_date'] ); ?>',
                        end: '<?php echo esc_js( $settings['google_end_date'] ); ?>'
                    },
                    <?php  } ?>
                    initialView: defaultview,
                    headerToolbar: {
                        left: 'prev,next today',
                        center: 'title',
                        end: 'dayGridMonth,timeGridWeek,timeGridDay,listMonth',
                    },
                    dateClick: function(info) {
                        //console.log(info.dateStr);
                        //alert('Resource ID: ' + info.resource.id);
                    },
                    editable: true,
                    droppable: true, // this allows things to be dropped onto the calendar
                    <?php  if( $settings['source_type'] == 'google' ) { ?>
                        events: {
                    // googleCalendarId: 'q01afuq5b849qdn1qtar1ed9no@group.calendar.google.com',
                        googleCalendarId: '<?php echo esc_js( $settings['google_calendar_id'] ); ?>',
                       
                        failure: function( errorObj ) {

                            const eorrMessage = document.createElement("span");
                            eorrMessage.innerText = '<?php echo esc_html__("Invalid API key or Calendar ID", "htmega-pro" ); ?>';
                            calendarEl.appendChild(eorrMessage);
                        }
                        
                    },
                    <?php  } else { ?>
                    events: <?php echo wp_json_encode( $event_list ); ?>,
                    <?php } ?>
                    locale: language,
                    firstDay: firstDay,
                    eventColor: '#F1F1FE',
                    eventTextColor: '#735BF2',
                    displayEventTime: true,
                    eventTimeFormat: { // like '14:30:00'
                        hour: '2-digit',
                        minute: '2-digit',
                        hour12: true
                    },
                    eventClick:  function(event, jsEvent, view) {
                        const {title, extendedProps, start, backgroundColor} = event.event,
                            {location, description} = extendedProps;

                        htmegaCalendarModal(title, location, dayjs(start).format('MMMM DD YYYY - hh:mm a'), description, backgroundColor, 1);
                        // Overlay & Close Modal JS
                        const htmegaModalOverlay = document.querySelector('.htmega-calender-overlay');
                        const htmegaModalclose = document.querySelector('.htmega-calender-close-modal');
                        const htMegaCloseModal = function() {
                            document.querySelector(".htmega-calender-modal").remove();
                            document.querySelector(".htmega-calender-overlay").remove();
                        };
                        htmegaModalOverlay.addEventListener("click", htMegaCloseModal);
                        htmegaModalclose.addEventListener("click", htMegaCloseModal);
                        event.el.removeAttribute("href");
                    },
                
                });
                    calendar.render();
            
            });
        </script>

        <?php
    }

}