<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class HTMega_Elementor_Widget_Scroll_Image extends Widget_Base {

    public function get_name() {
        return 'htmega-scrollimage-addons';
    }
    
    public function get_title() {
        return __( 'Scroll Image', 'htmega-pro' );
    }

    public function get_icon() {
        return 'htmega-icon eicon-exchange';
    }

    public function get_categories() {
        return [ 'htmega-addons' ];
    }

    public function get_keywords() {
        return ['image scroll', 'scroll image','image scrolling', 'single image scrolling', 'ht mega', 'htmega'];
    }

    public function get_help_url() {
        return 'https://wphtmega.com/docs/general-widgets/scroll-image-widget/';
    }
    protected function is_dynamic_content():bool {
		return false;
	}
    protected function register_controls() {

        $this->start_controls_section(
            'button_content',
            [
                'label' => __( 'Scroll Image', 'htmega-pro' ),
            ]
        );
            
            $this->add_control(
                'scroll_image',
                [
                    'label' => __( 'Choose Image', 'htmega-pro' ),
                    'type' => Controls_Manager::MEDIA,
                    'dynamic' => [
                        'active' => true, // Enable dynamic tags support
                    ],
                    'default' => [
                        'url' => Utils::get_placeholder_image_src(),
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Image_Size::get_type(),
                [
                    'name' => 'scroll_image_size',
                    'default' => 'large',
                    'separator' => 'none',
                ]
            );

            $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => 'scroll_inner_image_border',
                    'label' => __( 'Border', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .ht-scroll-image .thumb',
                ]
            );
   
            $this->add_control(
                'link_type',
                [
                    'label'       => __( 'Link', 'htmega-pro' ),
                    'type'        => Controls_Manager::SELECT,
                    'options'     => [
                        'none' => __('None', 'htmega-pro'),
                        'url'  => __( 'URL', 'htmega-pro' ),
                        'lightbox' => __( 'Lightbox', 'htmega-pro' ),
                    ],
                    'default'     => 'url',
                    'label_block' => true,
                ]
            );
    
            $this->add_control(
                'scroll_image_link',
                [
                    'label' => __( 'Custom Link', 'htmega-pro' ),
                    'show_label' => false,
                    'type' => Controls_Manager::URL,
                    'placeholder' => __( 'https://example.com/', 'htmega-pro' ),
                    'dynamic' => [
                        'active' => true,
                    ],
                    'condition'   => array(
                        'link_type'     => 'url',
                    ),
                ]
            );
            $this->add_control(
                'lightbox_required',
                [
                    'label' => __( 'Lightbox Note', 'htmega-pro'),
                    'show_label' => false,
                    'type' => \Elementor\Controls_Manager::RAW_HTML,
                    'raw' => __( "Ensure Elementor's Lightbox option is turned on.", 'htmega-pro' ),
                    'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
                    'condition' => [
                        'link_type' => 'lightbox',
                    ],
                ]
            );
            $this->add_responsive_control(
                'scroll_image_height',
                [
                    'label' => __( 'Container Height', 'htmega-pro' ),
                    'type' => Controls_Manager::SLIDER,
                    'size_units' => [ 'px', '%' ],
                    'range' => [
                        'px' => [
                            'min' => 1,
                            'step' => 1,
                            'max'=>5000,
                        ],
                        '%' => [
                            'min' => 0,
                            'max' => 100,
                        ],
                    ],
                    'default' => [
                        'unit' => 'px',
                        'size' => 600,
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image .thumb,
                        {{WRAPPER}} .ht-scroll-type-vertical .ht-scroll-image-wrapper,
                        {{WRAPPER}} .ht-scroll-type-horizontal .ht-scroll-image-wrapper' => 'min-height: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                    ],
                    'separator' => 'after',   
                ]
            );        
			$this->add_control(
				'image_height_width_notice',
				[
					'raw'             => __( "Ensure that the container's dimensions (height and width) are smaller than the image's actual dimensions (height and width) otherwise, scrolling will not function properly", 'htmega-pro' ),
					'type'            => Controls_Manager::RAW_HTML,
					'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
				]
			);
            $this->add_control(
                'htmega_enable_mask_image',
                array(
                    'label'     => esc_html__( 'Mask Image', 'htmega-pro' ),
                    'type'      => Controls_Manager::SWITCHER,
                )
            );
    
            $this->add_control(
                'htmega_mask_shape',
                [
                    'label'       => __( 'Shape', 'htmega-pro' ),
                    'type'        => Controls_Manager::SELECT,
                    'options'     => [
                        'circle' => esc_html__( 'Circle', 'htmega-pro' ),
                        'flower' => esc_html__( 'Flower', 'htmega-pro' ),
                        'sketch' => esc_html__( 'Sketch', 'htmega-pro' ),
                        'triangle' => esc_html__( 'Triangle', 'htmega-pro' ),
                        'blob' => esc_html__( 'Blob', 'htmega-pro' ),
                        'hexagon' => esc_html__( 'Hexagon', 'htmega-pro' ),
                        'custom' => esc_html__( 'Custom', 'htmega-pro' ),
                    ],
                    'default'     => 'circle',
                    'label_block' => true,
                    'condition'   => array(
                        'htmega_enable_mask_image' => 'yes',
                    ),
                ]
            );

            $this->add_control(
                'htmega_mask_image',
                array(
                    'label'       => esc_html__( 'Mask Image', 'htmega-pro' ),
                    'type'        => Controls_Manager::MEDIA,
                    'default'     => array(
                        'url' => '',
                    ),
                    'description' => esc_html__( 'Use a PNG image with the desired shape to create a mask around the scrolling image.', 'htmega-pro' ),
                    'selectors'   => array(
                        '{{WRAPPER}} .ht-scroll-image .thumb' => 'mask-image: url("{{URL}}");-webkit-mask-image: url("{{URL}}");',
                    ),
                    'condition'   => array(
                        'htmega_enable_mask_image' => 'yes',
                        'htmega_mask_shape' => 'custom',
                    ),
                )
            );
    
            $this->add_control(
                'mask_size',
                array(
                    'label'     => __( 'Mask Size', 'htmega-pro' ),
                    'type'      => Controls_Manager::SELECT,
                    'options'   => array(
                        'contain' => __( 'Contain', 'htmega-pro' ),
                        'cover'   => __( 'Cover', 'htmega-pro' ),
                        'custom'   => __( 'Custom', 'htmega-pro' ),
                    ),
                    'default'   => 'contain',
                    'selectors' => array(
                        '{{WRAPPER}} .ht-scroll-image .thumb' => 'mask-size: {{VALUE}};-webkit-mask-size: {{VALUE}};',
                    ),
                    'condition' => array(
                        'htmega_enable_mask_image' => 'yes',
                    ),
                )
            );
            $this->add_responsive_control(
                'mask_size_custom',
                [
                    'label' => __( 'Size', 'htmega-pro' ),
                    'type' => Controls_Manager::SLIDER,
                    'size_units' => [ 'px', '%' ],
                    'range' => [
                        'px' => [
                            'min' => 0,
                            'step' => 1,
                            'max'=> 1000,
                        ],
                        '%' => [
                            'min' => 0,
                            'max' => 100,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image .thumb' => 'mask-size: {{SIZE}}{{UNIT}};-webkit-mask-size: {{SIZE}}{{UNIT}};'
                    ], 
                    'condition' => array(
                        'htmega_enable_mask_image' => 'yes',
                        'mask_size' => 'custom',
                    ),
                ]
            );
            $this->add_control(
                'mask_position_cover',
                array(
                    'label'     => __( 'Mask Position', 'htmega-pro' ),
                    'type'      => Controls_Manager::SELECT,
                    'options'   => array(
                        'center center' => __( 'Center Center', 'htmega-pro' ),
                        'center left'   => __( 'Center Left', 'htmega-pro' ),
                        'center right'  => __( 'Center Right', 'htmega-pro' ),
                        'top center'    => __( 'Top Center', 'htmega-pro' ),
                        'top left'      => __( 'Top Left', 'htmega-pro' ),
                        'top right'     => __( 'Top Right', 'htmega-pro' ),
                        'bottom center' => __( 'Bottom Center', 'htmega-pro' ),
                        'bottom left'   => __( 'Bottom Left', 'htmega-pro' ),
                        'bottom right'  => __( 'Bottom Right', 'htmega-pro' ),
                    ),
                    'default'   => 'center center',
                    'selectors' => array(
                        '{{WRAPPER}} .ht-scroll-image .thumb' => 'mask-position: {{VALUE}}; -webkit-mask-position: {{VALUE}}',
                    ),
                    'condition' => array(
                        'htmega_enable_mask_image' => 'yes',
                        'mask_size'                  => 'cover',
                    ),
                )
            );
    
            $this->add_control(
                'mask_position_contain',
                array(
                    'label'     => __( 'Mask Position', 'htmega-pro' ),
                    'type'      => Controls_Manager::SELECT,
                    'options'   => array(
                        'center center' => __( 'Center Center', 'htmega-pro' ),
                        'top center'    => __( 'Top Center', 'htmega-pro' ),
                        'bottom center' => __( 'Bottom Center', 'htmega-pro' ),
                    ),
                    'default'   => 'center center',
                    'selectors' => array(
                        '{{WRAPPER}} .ht-scroll-image .thumb' => 'mask-position: {{VALUE}}; -webkit-mask-position: {{VALUE}}',
                    ),
                    'condition' => array(
                        'htmega_enable_mask_image' => 'yes',
                        'mask_size'                  => 'contain',
                    ),
                )
            );
            $this->add_control(
                'htmega_enable_overlay',
                    array(
                        'label'     => esc_html__( 'Overlay', 'htmega-pro' ),
                        'type'      => Controls_Manager::SWITCHER,
                    )
            );
        $this->end_controls_section();
        // Scrolling settings
        $this->start_controls_section(
            'scrolling_settings',
            [
                'label' => __( 'Settings', 'htmega-pro' ),
            ]
        );
        $this->add_control(
            'show_badge',
            [
                'label' => __( 'Show Badge', 'htmega-pro' ),
                'type'  => Controls_Manager::SWITCHER,
                'label_on' => __( 'Show', 'htmega-pro' ),
                'label_off' => __( 'Hide', 'htmega-pro' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'badge_text',
            [
                'label' => __( 'Badge Text', 'htmega-pro' ),
                'show_label' => false,
                'label_block' => true,
                'type' => Controls_Manager::TEXT,
                'placeholder' => __( 'Badge Text', 'htmega-pro' ),
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'show_badge' => 'yes',
                ]
            ]
        );        
        $this->add_control(
            'badge_position',
            [
                'label' => __( 'Position', 'htmega-pro'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __( 'Left', 'htmega-pro'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => __( 'Right', 'htmega-pro'),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
                'default' => 'left',
                'toggle' => false,
                'selectors_dictionary' => [
                    'left' => 'left:0 !important;right:auto !important',
                    'right' =>'right:0px !important;left:auto !important'
                ],
				'selectors' => [
                    '{{WRAPPER}} .ht-scroll-image span.htmega-badge' => '{{VALUE}}',
                ],
                'condition' => [
                    'show_badge' => 'yes',
                    'badge_text!' => '',
                ],
            ]
        );
        $this->add_control(
			'trigger_type',
            [
                'label'         => __( 'Trigger On', 'htmega-pro' ),
                'type'          => Controls_Manager::SELECT,
                'options'       => [
                    'hover'   => __( 'Hover', 'htmega-pro' ),
                    'scroll'  => __( 'Mouse Scroll', 'htmega-pro' ),
                ],
                'default'       => 'hover',
				'separator' => 'before',
            ]
        );

        $this->add_control(
			'scroll_type',
			[
				'label' => __( 'Scroll Type', 'htmega-pro' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'vertical' => [
						'title' => __( 'Vertical', 'htmega-pro' ),
						'icon' => 'eicon-navigation-vertical',
					],
					'horizontal' => [
						'title' => __( 'Horizontal', 'htmega-pro' ),
						'icon' => 'eicon-navigation-horizontal',
					],
				],
				'default' => 'vertical',
				'toggle' => false,
                'style_transfer' => true,
			]
		);

		$this->add_control(
			'vtr_direction',
			[
				'label' => __( 'Direction', 'htmega-pro' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => __( 'Top', 'htmega-pro' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => __( 'Bottom', 'htmega-pro' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'default' => 'top',
				'toggle' => false,
				'condition' => [
					'trigger_type' => 'hover',
					'scroll_type' => 'vertical',
				],
                'style_transfer' => true,
			]
		);

		$this->add_control(
			'htr_direction',
			[
				'label' => __( 'Direction', 'htmega-pro' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', 'htmega-pro' ),
						'icon' => 'eicon-h-align-left',
					],
					'right' => [
						'title' => __( 'Right', 'htmega-pro' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'default' => 'left',
				'toggle' => false,
				'condition' => [
					'trigger_type' => 'hover',
					'scroll_type' => 'horizontal',
				],
                'style_transfer' => true,
			]
		);

        $this->add_control(
            'speed_scroll_time',
            [
                'label'			=> __( 'Speed', 'htmega-pro' ),
                'type'			=> Controls_Manager::NUMBER,
                'min' => 0,
                'default'		=> 3,
                'selectors' => [
                    '{{WRAPPER}} .ht-scroll-image .thumb'   => 'transition-duration: {{Value}}s',
                ]
            ]
        );    
        $this->add_control(
			'indicator_icon',
            [
                'label' => __('Indicator Icon', 'htmega-pro'),
				'type'  => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', 'htmega-pro' ),
				'label_off' => __( 'Hide', 'htmega-pro' ),
				'return_value' => 'yes',
				'default' => 'yes',
                'condition'     => [
                    'trigger_type' => 'scroll',
                ],
            ]
		);
        $this->end_controls_section();
        // Style tab section
        $this->start_controls_section(
            'scroll_image_style_section',
            [
                'label' => __( 'Scroll Image Style', 'htmega-pro' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => 'scroll_image_background',
                    'label' => __( 'Background', 'htmega-pro' ),
                    'types' => [ 'classic', 'gradient' ],
                    'selector' => '{{WRAPPER}} .ht-scroll-image',
                ]
            );

            $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => 'scroll_image_border',
                    'label' => __( 'Border', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .ht-scroll-image',
                ]
            );

            $this->add_responsive_control(
                'scroll_image_border_radius',
                [
                    'label' => esc_html__( 'Border Radius', 'htmega-pro' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image' => 'border-radius: {{TOP}}px {{RIGHT}}px {{BOTTOM}}px {{LEFT}}px;',
                    ],
                ]
            );

            $this->add_responsive_control(
                'scroll_image_margin',
                [
                    'label' => __( 'Margin', 'htmega-pro' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => [ 'px', '%', 'em' ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                    'separator' =>'before',
                ]
            );

            $this->add_responsive_control(
                'scroll_image_padding',
                [
                    'label' => __( 'Padding', 'htmega-pro' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => [ 'px', '%', 'em' ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                    'separator' =>'before',
                ]
            );

            $this->add_group_control(
                Group_Control_Box_Shadow::get_type(),
                [
                    'name' => 'scroll_image_box_shadow',
                    'label' => __( 'Box Shadow', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .ht-scroll-image',
                ]
            );
            $this->add_control(
                'scroll_image_overlay_color',
                [
                    'label' => __( 'Overlay Color', 'htmega-pro' ),
                    'type' => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image-overlay' => 'background: {{VALUE}};',
                    ],
                    'condition'=> [
                        'htmega_enable_overlay' => 'yes'
                    ]
                ]
            );
        $this->end_controls_section();

        // Style tab section
        $this->start_controls_section(
            'scroll_badge_style_section',
            [
                'label' => __( 'Badge', 'htmega-pro' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition'=>[
                    'show_badge' => 'yes',
                ]
            ]
        );

            $this->add_control(
                'scroll_badge_offset_toggle',
                [
                    'label' => __( 'Offset', 'htmega-pro' ),
                    'type' => Controls_Manager::POPOVER_TOGGLE,
                    'label_off' => __( 'None', 'htmega-pro' ),
                    'label_on' => __( 'Custom', 'htmega-pro' ),
                    'return_value' => 'yes',
                    'condition'=>[
                        'show_badge' => 'yes',
                    ],
                ]
            );
    
            $this->start_popover();
    
            $this->add_responsive_control(
                'scroll_badge_left_right',
                [
                    'label' => __( 'Left', 'htmega-pro' ),
                    'type' => Controls_Manager::SLIDER,
                    'size_units' => [ 'px', '%' ],
                    'range' => [
                        'px' => [
                            'min' => -1000,
                            'step' => 1,
                            'max'=> 1000,
                        ],
                        '%' => [
                            'min' => -100,
                            'max' => 100,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image span.htmega-badge' => 'left: {{SIZE}}{{UNIT}}!important;',
                    ],
                    'condition'=>[
                        'badge_position' => 'left',
                    ],

                ]
            );  
            $this->add_responsive_control(
                'scroll_badge_right',
                [
                    'label' => __( 'Right', 'htmega-pro' ),
                    'type' => Controls_Manager::SLIDER,
                    'size_units' => [ 'px', '%' ],
                    'range' => [
                        'px' => [
                            'min' => -200,
                            'step' => 1,
                            'max'=> 200,
                        ],
                        '%' => [
                            'min' => -100,
                            'max' => 100,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image span.htmega-badge' => 'right: {{SIZE}}{{UNIT}}!important;',
                    ],
                    'condition'=>[
                        'badge_position' => 'right',
                    ],
                ]
            );  

            $this->add_responsive_control(
                'scroll_badge_top_bottom',
                [
                    'label' => __( 'Top-Bottom', 'htmega-pro' ),
                    'type' => Controls_Manager::SLIDER,
                    'size_units' => [ 'px', '%' ],
                    'range' => [
                        'px' => [
                            'min' => -1000,
                            'step' => 1,
                            'max'=> 1000,
                        ],
                        '%' => [
                            'min' => -100,
                            'max' => 100,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image span.htmega-badge' => 'top: {{SIZE}}{{UNIT}};',
                    ], 
                ]
            );

            $this->end_popover();            
                
            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => 'scroll_badge_background',
                    'label' => __( 'Background', 'htmega-pro' ),
                    'types' => [ 'classic', 'gradient' ],
                    'selector' => '{{WRAPPER}} .ht-scroll-image span.htmega-badge',
                    'condition'=>[
                        'show_badge' => 'yes',
                    ]
                ]
            );

            $this->add_control(
                'scroll_badge_text_color',
                [
                    'label' => __( 'Color', 'htmega-pro' ),
                    'type' => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image span.htmega-badge' => 'color: {{VALUE}};',
                    ],
                    'condition'=>[
                        'show_badge' => 'yes',
                    ]
                ]
            );

            $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => 'scroll_badge_border',
                    'label' => __( 'Border', 'htmega-pro' ),
                    'selector' => '{{WRAPPER}} .ht-scroll-image span.htmega-badge',
                ]
            );

            $this->add_responsive_control(
                'scroll_badge_border_radius',
                [
                    'label' => esc_html__( 'Border Radius', 'htmega-pro' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image span.htmega-badge' => 'border-radius: {{TOP}}px {{RIGHT}}px {{BOTTOM}}px {{LEFT}}px;',
                    ],
                ]
            );

            $this->add_responsive_control(
                'scroll_badge_margin',
                [
                    'label' => __( 'Margin', 'htmega-pro' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => [ 'px', '%', 'em' ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image span.htmega-badge' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                    'separator' =>'before',
                ]
            );

            $this->add_responsive_control(
                'scroll_badge_padding',
                [
                    'label' => __( 'Padding', 'htmega-pro' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => [ 'px', '%', 'em' ],
                    'selectors' => [
                        '{{WRAPPER}} .ht-scroll-image span.htmega-badge' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                    'separator' =>'before',
                ]
            );

            $this->add_group_control(
                Group_Control_Box_Shadow::get_type(),
                [
                    'name' => 'scroll_badge_box_shadow',
                    'exclude' => [
                        'box_shadow_position',
                    ],
                    'selector' => '{{WRAPPER}} .ht-scroll-image span.htmega-badge',
                ]
            );
    
            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'scroll_badge_typography',
                    'label' => __( 'Typography', 'htmega-pro' ),
                    'exclude' => [
                        'line_height'
                    ],
                    'default' => [
                        'font_size' => ['']
                    ],
                    'selector' => '{{WRAPPER}} .ht-scroll-image span.htmega-badge',
                ]
            );            

        $this->end_controls_section();
        // Style tab section
        $this->start_controls_section(
            'indicator_icon_style',
            [
                'label' => __( 'Indicator Icon', 'htmega-pro' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition'=>[
                    'indicator_icon' => 'yes',
                ]
            ]
        );
        $this->add_responsive_control(
            'indicator_size',
            [
                'label' => __( 'Size', 'htmega-pro' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'step' => 1,
                        'max'=> 300,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .ht-scroll-indicator svg' => 'width: {{SIZE}}{{UNIT}};'
                ], 
            ]
        );
        $this->add_control(
            'indicator_color',
            [
                'label' => __( 'Color', 'htmega-pro' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .ht-scroll-indicator svg path' => 'fill: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_section();
    }

    protected function render( $instance = [] ) {

        $settings   = $this->get_settings_for_display();
        $scroll_image_url = Group_Control_Image_Size::get_attachment_image_src( $settings['scroll_image']['id'], 'scroll_image_size', $settings);

        if ( !$scroll_image_url ) {
            $scroll_image_url = $settings['scroll_image']['url'];
        }
        if ( 'yes'=== $settings['show_badge'] && $settings['badge_text'] ) {
            $this->add_render_attribute('htbadge','class','htmega-badge');
        }

        $this->add_render_attribute('scrollimage_wrapper', 'class', 'ht-scroll-image');

        if ( 'hover' == $settings['trigger_type'] && 'vertical' == $settings['scroll_type'] ) {
            $this->add_render_attribute('scrollimage_wrapper', 'class', 'ht-vtr-direction-' . esc_attr( $settings['vtr_direction'] ) );
        } else if ( 'hover' == $settings['trigger_type'] && 'horizontal' == $settings['scroll_type'] ) {
            $this->add_render_attribute('scrollimage_wrapper', 'class', 'ht-htr-direction-' . esc_attr( $settings['htr_direction'] ) );
        }

        if ( 'scroll' == $settings['trigger_type'] ) {
            $this->add_render_attribute( 'scrollimage_wrapper', 'class', 'ht-scroll-type-' . esc_attr( $settings['scroll_type'] ) );
        }

        $this->add_render_attribute( 'scrollimage', 'class', 'thumb' );
        if ( 'hover' == $settings['trigger_type'] ) {
            $this->add_render_attribute('scrollimage', 'style', 'background-image: url('.esc_url( $scroll_image_url ).');');
        }
        if ( 'yes' == $settings['htmega_enable_mask_image'] &&  'custom' !== $settings['htmega_mask_shape'] ) {
            $this->add_render_attribute('scrollimage', 'style', 'mask-image: url('.ELEMENTOR_ASSETS_URL . 'mask-shapes/'.$settings['htmega_mask_shape'].'.svg'.'); -webkit-mask-image: url('.esc_url( ELEMENTOR_ASSETS_URL . 'mask-shapes/'.$settings['htmega_mask_shape'].'.svg' ).');');
        }

        // Link
        if ( isset( $settings['scroll_image_link'] ) ) {
            $this->add_link_attributes( 'url', $settings['scroll_image_link'] );

        }
		if ( 'lightbox' === $settings['link_type'] ) {
			$this->add_render_attribute( 'url', 'href', esc_url( $scroll_image_url ) );

		}
        if( !empty($settings['scroll_image_link']['url']) || 'lightbox' === $settings['link_type'] || 'link' === $settings['link_type'] ) { ?>
            <a <?php echo $this->get_render_attribute_string( 'url' ); ?> >
                <div <?php echo $this->get_render_attribute_string( 'scrollimage_wrapper' ); ?>>
                    <?php if( 'yes'=== $settings['show_badge'] && $settings['badge_text'] ): ?>
                        <span <?php $this->print_render_attribute_string('htbadge'); ?>>
                            <?php echo wp_kses_post($settings['badge_text']);?>
                        </span>
                    <?php endif; ?>
                    <div class="ht-scroll-image-wrapper">
                        <div <?php echo $this->get_render_attribute_string( 'scrollimage' ); ?> >
                            <?php
                            if ( 'scroll' == $settings['trigger_type'] ) {
                                echo '<img src="' . esc_url($scroll_image_url) . '" alt="' . esc_attr( $settings['scroll_image']['alt'] ) . '" />';
                            }
                            if ( 'yes' == $settings['htmega_enable_overlay'] ) {
                                echo '<div class="ht-scroll-image-overlay"></div>';
                            } ?>
                        </div>
                    </div>
                    <?php 
                        if ( 'yes' == $settings['indicator_icon'] ) {
                            echo $this->indicator_icon( $settings['scroll_type'] );
                        }
                    ?>
                </div>                        
            </a>                    
        <?php } else{ ?>
            <div <?php echo $this->get_render_attribute_string( 'scrollimage_wrapper' ); ?> >
                <?php if( 'yes'=== $settings['show_badge'] && $settings['badge_text'] ): ?>
                    <span <?php $this->print_render_attribute_string('htbadge'); ?>>
                        <?php echo wp_kses_post( $settings['badge_text'] );?>
                    </span>
                <?php endif; ?>
                <div class="ht-scroll-image-wrapper">
                    <div <?php echo $this->get_render_attribute_string( 'scrollimage' ); ?> >
                        <?php
                            if ( 'scroll' == $settings['trigger_type'] ) {
                                echo '<img src="' . esc_url( $scroll_image_url ) . '" alt="' . esc_attr( $settings['scroll_image']['alt'] ) . '" />';
                            }
                            if ( 'yes' == $settings['htmega_enable_overlay'] ) {
                                echo '<div class="ht-scroll-image-overlay"></div>';
                            }
                            ?>
                    </div>
                </div>
                <?php 
                    if ( 'yes' == $settings['indicator_icon'] ) {
                        echo $this->indicator_icon( $settings['scroll_type'] );
                    }
                ?>
            </div>
        <?php } 
    }

    function indicator_icon( $scroll_type ) {
        return  ( 'vertical' == $scroll_type ) ? '<div class="ht-scroll-indicator"><svg width="512" height="512" viewBox="0 0 512 512" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M254.931 506.374C252.692 506.374 250.546 505.484 248.963 503.902C247.38 502.319 246.491 500.172 246.491 497.934V427.267C246.491 425.028 247.38 422.882 248.963 421.299C250.546 419.716 252.692 418.827 254.931 418.827C257.169 418.827 259.316 419.716 260.898 421.299C262.481 422.882 263.37 425.028 263.37 427.267V498.159C263.312 500.358 262.397 502.447 260.82 503.982C259.244 505.516 257.131 506.374 254.931 506.374Z" fill="black"/>
        <path d="M254.931 506.374C252.864 506.387 250.864 505.647 249.304 504.292L204.293 464.907C202.607 463.415 201.583 461.314 201.445 459.067C201.308 456.819 202.07 454.609 203.562 452.923C205.054 451.237 207.155 450.213 209.402 450.075C211.65 449.938 213.86 450.699 215.546 452.192L260.557 491.576C262.233 493.06 263.256 495.145 263.403 497.378C263.551 499.612 262.811 501.813 261.345 503.504C260.547 504.418 259.56 505.148 258.453 505.643C257.345 506.139 256.144 506.388 254.931 506.374Z" fill="black"/>
        <path d="M254.931 506.374C253.728 506.38 252.537 506.127 251.44 505.632C250.343 505.136 249.365 504.411 248.573 503.504C247.107 501.813 246.367 499.612 246.515 497.378C246.662 495.145 247.685 493.06 249.361 491.576L294.372 452.192C296.058 450.699 298.268 449.938 300.516 450.075C302.763 450.213 304.864 451.237 306.356 452.923C307.848 454.609 308.61 456.819 308.473 459.067C308.335 461.314 307.311 463.415 305.625 464.907L260.614 504.292C259.039 505.66 257.017 506.401 254.931 506.374Z" fill="black"/>
        <path d="M254.931 95.648C252.692 95.648 250.546 94.7588 248.963 93.1761C247.38 91.5934 246.491 89.4468 246.491 87.2084V16.2599C246.491 14.0216 247.38 11.8749 248.963 10.2922C250.546 8.70948 252.692 7.82031 254.931 7.82031C257.169 7.82031 259.316 8.70948 260.898 10.2922C262.481 11.8749 263.37 14.0216 263.37 16.2599V87.1522C263.378 88.2652 263.165 89.3687 262.744 90.3991C262.323 91.4296 261.703 92.3666 260.918 93.1563C260.134 93.9459 259.201 94.5726 258.173 95.0002C257.146 95.4279 256.044 95.648 254.931 95.648Z" fill="black"/>
        <path d="M209.752 64.1968C208.027 64.2124 206.339 63.6995 204.915 62.7269C203.491 61.7544 202.399 60.3689 201.786 58.757C201.173 57.1452 201.069 55.3842 201.487 53.7112C201.905 52.0382 202.826 50.5334 204.125 49.3994L249.136 10.0148C249.956 9.26856 250.915 8.69115 251.959 8.31556C253.002 7.93996 254.109 7.77354 255.217 7.82578C256.324 7.87803 257.411 8.14792 258.414 8.62006C259.417 9.09219 260.318 9.75731 261.064 10.5775C261.81 11.3976 262.388 12.3567 262.763 13.4C263.139 14.4433 263.305 15.5503 263.253 16.6579C263.201 17.7655 262.931 18.852 262.459 19.8553C261.987 20.8586 261.322 21.7591 260.501 22.5054L215.49 61.89C213.921 63.3298 211.881 64.1497 209.752 64.1968Z" fill="black"/>
        <path d="M300.055 64.1969C297.988 64.2102 295.988 63.4701 294.428 62.1151L249.417 22.7305C247.731 21.2383 246.707 19.1374 246.569 16.8899C246.432 14.6424 247.194 12.4325 248.686 10.7463C250.178 9.06012 252.279 8.03576 254.526 7.89859C256.774 7.76143 258.984 8.52268 260.67 10.0149L305.681 49.3995C306.98 50.5335 307.901 52.0383 308.319 53.7113C308.737 55.3843 308.633 57.1452 308.02 58.7571C307.407 60.369 306.315 61.7544 304.891 62.727C303.467 63.6995 301.779 64.2125 300.055 64.1969Z" fill="black"/>
        <path d="M282.669 391.315H227.193C206.869 391.3 187.38 383.223 173.003 368.857C158.627 354.491 150.536 335.008 150.506 314.684V199.737C150.536 179.412 158.627 159.93 173.003 145.563C187.38 131.197 206.869 123.12 227.193 123.105H282.669C302.984 123.135 322.458 131.218 336.823 145.583C351.188 159.948 359.271 179.422 359.301 199.737V314.684C359.271 334.998 351.188 354.472 336.823 368.837C322.458 383.202 302.984 391.285 282.669 391.315ZM227.193 139.985C211.345 139.999 196.15 146.298 184.939 157.499C173.727 168.699 167.415 183.889 167.385 199.737V314.684C167.415 330.531 173.727 345.721 184.939 356.922C196.15 368.122 211.345 374.421 227.193 374.436H282.669C298.512 374.421 313.702 368.121 324.904 356.918C336.107 345.716 342.407 330.526 342.421 314.684V199.737C342.407 183.894 336.107 168.704 324.904 157.502C313.702 146.3 298.512 139.999 282.669 139.985H227.193Z" fill="black"/>
        <path d="M254.931 239.571C252.692 239.571 250.546 238.681 248.963 237.099C247.38 235.516 246.491 233.369 246.491 231.131V193.378C246.491 191.14 247.38 188.993 248.963 187.41C250.546 185.828 252.692 184.938 254.931 184.938C257.169 184.938 259.316 185.828 260.898 187.41C262.481 188.993 263.37 191.14 263.37 193.378V231.131C263.37 233.369 262.481 235.516 260.898 237.099C259.316 238.681 257.169 239.571 254.931 239.571Z" fill="black"/>
        </svg></div>' : '<div class="ht-scroll-indicator"><svg width="512" height="512" viewBox="0 0 512 512" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M474.546 257.499C474.546 259.737 473.657 261.884 472.074 263.467C470.492 265.049 468.345 265.938 466.107 265.938H395.44C393.201 265.938 391.055 265.049 389.472 263.467C387.889 261.884 387 259.737 387 257.499C387 255.261 387.889 253.114 389.472 251.531C391.055 249.949 393.201 249.059 395.44 249.059H466.332C468.531 249.118 470.62 250.033 472.155 251.609C473.689 253.186 474.547 255.299 474.546 257.499Z" fill="black"/>
        <path d="M474.546 257.499C474.56 259.565 473.82 261.566 472.465 263.126L433.08 308.136C431.588 309.823 429.487 310.847 427.239 310.984C424.992 311.121 422.782 310.36 421.096 308.868C419.41 307.376 418.385 305.275 418.248 303.027C418.111 300.78 418.872 298.57 420.364 296.884L459.749 251.873C461.233 250.197 463.318 249.174 465.551 249.026C467.785 248.879 469.986 249.619 471.677 251.085C472.591 251.883 473.321 252.87 473.816 253.977C474.312 255.084 474.561 256.286 474.546 257.499Z" fill="black"/>
        <path d="M474.546 257.498C474.553 258.702 474.3 259.893 473.804 260.99C473.309 262.087 472.584 263.065 471.677 263.856C469.986 265.323 467.785 266.063 465.551 265.915C463.318 265.768 461.233 264.745 459.749 263.069L420.364 218.058C418.872 216.371 418.111 214.162 418.248 211.914C418.385 209.667 419.41 207.566 421.096 206.073C422.782 204.581 424.992 203.82 427.239 203.957C429.487 204.094 431.588 205.119 433.08 206.805L472.465 251.816C473.833 253.39 474.574 255.413 474.546 257.498Z" fill="black"/>
        <path d="M122.832 256.305C122.832 258.543 121.942 260.69 120.36 262.272C118.777 263.855 116.63 264.744 114.392 264.744H43.4435C41.2052 264.744 39.0585 263.855 37.4758 262.272C35.8931 260.69 35.0039 258.543 35.0039 256.305C35.0039 254.066 35.8931 251.92 37.4758 250.337C39.0585 248.754 41.2052 247.865 43.4435 247.865H114.336C115.449 247.858 116.552 248.07 117.583 248.491C118.613 248.912 119.55 249.533 120.34 250.317C121.13 251.101 121.756 252.034 122.184 253.062C122.611 254.089 122.832 255.192 122.832 256.305Z" fill="black"/>
        <path d="M91.3804 301.484C91.396 303.208 90.8831 304.896 89.9105 306.32C88.938 307.744 87.5525 308.836 85.9406 309.449C84.3287 310.062 82.5678 310.166 80.8948 309.748C79.2218 309.33 77.717 308.409 76.583 307.11L37.1984 262.099C36.4522 261.279 35.8747 260.32 35.4992 259.277C35.1236 258.233 34.9571 257.126 35.0094 256.019C35.0616 254.911 35.3315 253.825 35.8037 252.821C36.2758 251.818 36.9409 250.918 37.761 250.171C38.5812 249.425 39.5403 248.848 40.5836 248.472C41.6269 248.096 42.7339 247.93 43.8415 247.982C44.9491 248.035 46.0356 248.304 47.0389 248.777C48.0422 249.249 48.9427 249.914 49.689 250.734L89.0736 295.745C90.5134 297.315 91.3333 299.354 91.3804 301.484Z" fill="black"/>
        <path d="M91.3805 211.181C91.3938 213.247 90.6537 215.247 89.2987 216.807L49.9141 261.818C48.4219 263.504 46.3209 264.529 44.0735 264.666C41.826 264.803 39.6161 264.042 37.9299 262.55C36.2437 261.057 35.2194 258.956 35.0822 256.709C34.945 254.462 35.7063 252.252 37.1985 250.565L76.5831 205.554C77.7171 204.255 79.2219 203.335 80.8949 202.917C82.5679 202.498 84.3288 202.603 85.9407 203.216C87.5526 203.829 88.938 204.92 89.9106 206.345C90.8831 207.769 91.3961 209.456 91.3805 211.181Z" fill="black"/>
        <path d="M282.669 391.209H227.193C206.869 391.194 187.38 383.118 173.003 368.751C158.627 354.385 150.536 334.903 150.506 314.578V199.631C150.536 179.307 158.627 159.824 173.003 145.458C187.38 131.091 206.869 123.015 227.193 123H282.669C302.984 123.03 322.458 131.113 336.823 145.478C351.188 159.842 359.271 179.316 359.301 199.631V314.578C359.271 334.893 351.188 354.367 336.823 368.732C322.458 383.096 302.984 391.18 282.669 391.209ZM227.193 139.879C211.345 139.894 196.15 146.192 184.939 157.393C173.727 168.594 167.415 183.783 167.385 199.631V314.578C167.415 330.426 173.727 345.615 184.939 356.816C196.15 368.017 211.345 374.315 227.193 374.33H282.669C298.512 374.315 313.702 368.015 324.904 356.813C336.107 345.61 342.407 330.421 342.421 314.578V199.631C342.407 183.789 336.107 168.599 324.904 157.397C313.702 146.194 298.512 139.894 282.669 139.879H227.193Z" fill="black"/>
        <path d="M254.931 239.571C252.692 239.571 250.546 238.681 248.963 237.099C247.38 235.516 246.491 233.369 246.491 231.131V193.378C246.491 191.14 247.38 188.993 248.963 187.41C250.546 185.828 252.692 184.938 254.931 184.938C257.169 184.938 259.316 185.828 260.898 187.41C262.481 188.993 263.37 191.14 263.37 193.378V231.131C263.37 233.369 262.481 235.516 260.898 237.099C259.316 238.681 257.169 239.571 254.931 239.571Z" fill="black"/>
        </svg></div>';
    }
    
}