/**
 * LearnDash Block ld-course-notstarted
 *
 * @since 2.5.9
 * @package LearnDash
 */

/**
 * LearnDash block functions
 */
import {
	ldlms_get_post_edit_meta,
	ldlms_get_custom_label,
	ldlms_get_integer_value
} from '../ldlms.js';

/**
 * Internal block libraries
 */
import { __, _x, sprintf} from '@wordpress/i18n';
import { registerBlockType } from '@wordpress/blocks';
import { InnerBlocks, InspectorControls, useBlockProps } from '@wordpress/block-editor';
import { PanelBody, TextControl, ToggleControl } from '@wordpress/components';

const block_title = sprintf(
	// translators: placeholder: Course.
	_x('LearnDash %s Not Started', 'placeholder: Course', 'learndash'), ldlms_get_custom_label('course')
);
const block_key = 'learndash/ld-course-notstarted';
registerBlockType(
	block_key,
	{
		title: block_title,
		description: sprintf(
			// translators: placeholder: Course.
			_x('This block shows the content if the user is enrolled into the %s but not yet started.', 'placeholder: Course', 'learndash'), ldlms_get_custom_label('course') ),
		icon: 'star-empty',
		category: 'learndash-blocks',
		supports: {
			customClassName: false,
		},
		apiVersion: 3,
		attributes: {
			course_id: {
				type: 'string',
				default: '',
			},
			user_id: {
				type: 'string',
				default: '',
			},
			autop: {
				type: 'boolean',
				default: true
			},
		},
		edit: props => {
			const { attributes: { course_id, user_id, autop }, setAttributes } = props;

			const blockProps = useBlockProps();

			const inspectorControls = (
				<InspectorControls key="controls">
					<PanelBody
						title={__('Settings', 'learndash')}
					>
						<TextControl
							label={sprintf(
								// translators: placeholder: Course.
								_x('%s ID', 'placeholder: Course', 'learndash'), ldlms_get_custom_label('course') ) }
							help={sprintf(
								// translators: placeholders: Course, Course.
								_x('Enter single %1$s ID. Leave blank if used within a %2$s.', 'placeholders: Course, Course', 'learndash'), ldlms_get_custom_label('course'), ldlms_get_custom_label('course') ) }
							value={course_id || ''}
							type={'number'}
							onChange={ function( new_course_id ) {
								if ( new_course_id != "" && new_course_id < 0 ) {
									setAttributes({ course_id: "0" });
								} else {
									setAttributes({ course_id: new_course_id });
								}
							}}						/>
						<TextControl
							label={__('User ID', 'learndash')}
							help={__('Enter specific User ID. Leave blank for current User.', 'learndash')}
							value={user_id || ''}
							type={'number'}
							onChange={ function( new_user_id ) {
								if ( new_user_id != "" && new_user_id < 0 ) {
									setAttributes({ user_id: "0" });
								} else {
									setAttributes({ user_id: new_user_id });
								}
							}}						/>
						<ToggleControl
							label={__('Auto Paragraph', 'learndash')}
							checked={!!autop}
							onChange={autop => setAttributes({ autop })}
						/>

					</PanelBody>
				</InspectorControls>
			);

			let ld_block_error_message = '';
			let preview_course_id = ldlms_get_integer_value(course_id);

			if (preview_course_id === 0) {
				preview_course_id = ldlms_get_post_edit_meta('course_id');
				preview_course_id = ldlms_get_integer_value(preview_course_id);

				if (preview_course_id == 0) {
					ld_block_error_message = sprintf(
						// translators: placeholders: Course, Course.
						_x('%1$s ID is required when not used within a %2$s.', 'placeholders: Course, Course', 'learndash'), ldlms_get_custom_label('course'), ldlms_get_custom_label('course'));
				}
			}

			if (ld_block_error_message.length) {
				ld_block_error_message = (<span className="learndash-block-error-message">{ld_block_error_message}</span>);
			}

			return (
				<div { ...blockProps }>
					{ inspectorControls }
					<span className="learndash-inner-header">{block_title}</span>
					<div className="learndash-block-inner">
						{ld_block_error_message}
						<InnerBlocks />
					</div>
				</div>
			);
		},

		save: props => {
			return (
				<InnerBlocks.Content />
			);
		}
	},
);
