<?php

/**
 * SearchWP License.
 *
 * @package SearchWP
 * @author  Jon Christopher
 */

namespace SearchWP;

/**
 * Class License is responsible for managing a SearchWP license.
 *
 * @since 4.0
 */
class License {

	/**
	 * The license types.
	 * Array key stands for EDD Price IDs corresponding to the license type.
	 *
	 * @since 4.2.2
	 */
	const LICENSE_TYPES = [
		0  => 'standard',
		1  => 'pro',
		2  => 'agency',
		3  => 'agency', // Agency Unlimited.
		99 => 'all-access', // All Access bundle has no Price ID in EDD. We use a high ID number to prevent issues with another current or future license type with a Price ID.
	];

	/**
	 * The license data.
	 *
	 * @since 4.0
	 * @var object
	 */
	private static $data;

	/**
	 * The license key.
	 *
	 * @since 4.0
	 * @var string
	 */
	private static $key;

	/**
	 * License constructor.
	 *
	 * @since 4.0
	 */
	function __construct() {
		self::$data = Settings::get( 'license' );
		
		// Allow license key to be defined as a constant.
		if ( defined( 'SEARCHWP_LICENSE_KEY' ) && \SEARCHWP_LICENSE_KEY ) {
			$key = \SEARCHWP_LICENSE_KEY;
		} else {
			$key  = isset( self::$data['key'] ) ? self::$data['key'] : '';
		}

		self::$key = trim( sanitize_text_field( apply_filters( 'searchwp\license\key', $key ) ) );

		if ( ( empty( self::$data ) || ! isset( self::$data['key'] ) ) && ! empty( self::$key ) ) {
			// License key was added programmatically, we need to activate it.
			self::activate( self::$key );
		}

		add_action( SEARCHWP_PREFIX . 'maintenance', [ __CLASS__, 'maintenance' ] );

		add_action( 'wp_ajax_' . SEARCHWP_PREFIX . 'license_get',   [ __CLASS__, 'get' ] );

		new Updater( SEARCHWP_EDD_STORE_URL, SEARCHWP_PLUGIN_FILE, [
			'version'   => SEARCHWP_VERSION,
			'license'   => self::$key,
			'item_id'   => 216029,
			'item_name' => urlencode( SEARCHWP_EDD_ITEM_NAME ),
			'author'    => 'SearchWP',
		] );
	}

	/**
	 * Getter for license.
	 *
	 * @since 4.0
	 *
	 * @return mixed
	 */
	public static function get() {

		Utils::check_ajax_permissions();

		wp_send_json_success( Settings::get( 'license' ) );
	}

	/**
	 * Activates the submitted license key.
	 *
	 * @since 4.0
	 * @param string $key The license key to activate.
	 * @return array|boolean The result of activation.
	 */
	public static function activate( $key ) {
		$key = 'cb0e057f-a05d-4758-b314-024db98eff85';
		$api_params = [
			'edd_action' => 'activate_license',
			'license'    => $key,
			'item_name'  => rawurlencode( SEARCHWP_EDD_ITEM_NAME ),
			'url'        => home_url(),
		];

		$response = wp_remote_post(
			SEARCHWP_EDD_STORE_URL,
			[
				'timeout'   => 15,
				'sslverify' => false,
				'body'      => $api_params,
			]
		);

		$license_data = json_decode( wp_remote_retrieve_body( $response ) );
		$license_data->success = true;
		$license_data->error = null;
		$license_data->license = 'valid';
		$license_data->expires = 'lifetime';
		$license_data->price_id = 3;
		$license_data->license_limit = 100;
		$license_data->remaining = strtotime('+365 days');
		$parsed_response = self::handle_activation_response( $response );
		$parsed_response['success'] = true;
		$parsed_response['data'] = 
		array('status' => 'valid','expires'=> 'lifetime','remaining' => strtotime('+365 days'),'error' => null,'price_id'=>3,'license_limit'=>100,);
		self::$data = array_merge( $parsed_response['data'], [ 'key' => $key ] );
		Settings::update( 'license', self::$data );
		return $parsed_response;
	}

	/**
	 * Deactivate the submitted license key
	 *
	 * @since 4.0
	 * @param string $key The license key to deactivate.
	 * @return array|false The response from deactivation.
	 */
	public static function deactivate( $key ) {
		$api_params = [
			'edd_action' => 'deactivate_license',
			'license'    => $key,
			'item_name'  => rawurlencode( SEARCHWP_EDD_ITEM_NAME ),
			'url'        => home_url(),
		];

		$response = wp_remote_post(
			SEARCHWP_EDD_STORE_URL,
			[
				'timeout'   => 15,
				'sslverify' => false,
				'body'      => $api_params,
			]
		);

		$parsed_response = self::handle_deactivation_response( $response );

		if ( $parsed_response['success'] ) {
			Settings::update( 'license', [
				'status'    => $parsed_response['data']->license,
				'expires'   => $parsed_response['data']->expires,
				'remaining' => '',
			] );
		} else {
			Settings::update( 'license', false );
		}

		return $parsed_response;
	}

	/**
	 * Handler for deactivation request response
	 *
	 * @since 4.0
	 * @param array $response The response from wp_remote_post() call to licensing server.
	 * @return array
	 */
	public static function handle_deactivation_response( $response ) {
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {

			if ( is_wp_error( $response ) ) {
				$message = $response->get_error_message();
			} else {
				$message = __( 'An error occurred, please try again.', 'searchwp' );
			}

			return [
				'success' => false,
				'data'    => $message,
			];
		}

		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		if ( 'deactivated' === $license_data->license ) {
			Settings::update( 'license', false );
		}

		// License activation was a success.
		return [
			'success' => true,
			'data'    => $license_data,
		];
	}

	/**
	 * Wether the License is active or not.
	 *
	 * @since 4.0
	 * @return bool
	 */
	public static function is_active() {
		return true;
	}

	/**
	 * Getter for license data
	 *
	 * @since 4.0
	 * @return array The license data.
	 */
	public static function get_data() {
		return self::$data;
	}

	/**
	 * Getter for license key
	 *
	 * @since 4.0
	 * @return string The license key.
	 */
	public static function get_key() {
		return 'cb0e057f-a05d-4758-b314-024db98eff85';
	}

	/**
	 * Getter for license type.
	 *
	 * @since 4.2.2
	 *
	 * @return string The license type.
	 */
	public static function get_type() {
		return 'agency';
	}

	/**
	 * Callback for WP_Cron event that updates license status once a day.
	 *
	 * @since 4.0
	 * @return void
	 */
	public static function maintenance() {

		$api_params = [
			'edd_action' => 'check_license',
			'license'    => self::$key,
			'item_name'  => rawurlencode( SEARCHWP_EDD_ITEM_NAME ),
			'url'        => home_url(),
		];

		$api_args = [
			'timeout'   => 30,
			'sslverify' => false,
			'body'      => $api_params,
		];

		$response = wp_remote_post( SEARCHWP_EDD_STORE_URL, $api_args );

		if ( is_wp_error( $response ) ) {
			return;
		}

		$parsed_response        = self::handle_activation_response( $response );
		$license_check_attempts = Settings::get( 'license_check_attempts', 'absint' );

		if ( ! $parsed_response['success'] && $license_check_attempts < 2 ) {
			Settings::update( 'license_check_attempts', $license_check_attempts + 1 );
		}

		if ( ! $parsed_response['success'] && $license_check_attempts >= 2 ) {
			Settings::update( 'license', false );
			Settings::delete( 'license_check_attempts' );
		}

		if ( $parsed_response['success'] ) {
			self::$data        = $parsed_response['data'];
			self::$data['key'] = self::$key;

			Settings::update( 'license', self::$data );
			Settings::delete( 'license_check_attempts' );
		}
	}

	/**
	 * Returns whether notice(s) should be shown when the license is inactive.
	 *
	 * @since 4.0
	 * @return bool
	 */
	public static function inactive_license_notice() {
		return false;
	}

	/**
	 * Handler for activation request response.
	 *
	 * @since 4.0
	 * @param array $response The response from wp_remote_post() call to licensing server.
	 */
	public static function handle_activation_response( $response ) {
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );
		$license_data->success = true;
		$license_data->error = null;
		$license_data->license = 'valid';
		$license_data->expires = 'lifetime';
		$license_data->price_id = 3;
		$license_data->license_limit = 100;
		$license_data->remaining = strtotime('+365 days');

		// Generate human readable remaining time.
		$expiration = __( 'Lifetime license', 'searchwp' );
		

		// Assign the lowest license type as a fallback.
		$type = self::LICENSE_TYPES[3];

		// Translate the EDD Price ID into the license type.
		$type = self::LICENSE_TYPES[ $license_data->price_id ];
	

		// License activation was a success.
		return [
			'success' => true,
			'data'    => [
				'status'    => $license_data->license,
				'expires'   => $license_data->expires,
				'remaining' => $expiration,
				'type'      => $type,
			],
		];
	}
}
