<?php
/**
 * The Plus Elementor Addon Installation Handler
 *
 * @package ThePlus
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Theplus_Element_Install_Plugin
 */
class Theplus_Element_Install_Plugin {

	/**
	 * Instance of this class.
	 *
	 * @var object
	 */
	private static $_instance;

	/**
	 * Get instance of this class.
	 *
	 * @return Theplus_Element_Install_Plugin
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		if ( is_admin() && current_user_can( 'manage_options' ) ) {
			add_action( 'admin_action_theplus_lite_install_plugin', array( __CLASS__, 'plus_free_install' ) );
		}
	}

	/**
	 * Install the free version of the plugin.
	 *
	 * This function handles the installation process of The Plus Addons for Elementor Lite.
	 * It performs the following actions:
	 * 1. Checks if the plugin is already installed
	 * 2. If installed, checks for available updates
	 * 3. If not installed, downloads and installs the plugin
	 * 4. Activates the plugin after successful installation
	 * 5. Displays appropriate success/error messages
	 *
	 * @since 1.0.0
	 * @version v6.3.14
	 * @access public
	 * @static
	 *
	 * @return void
	 */
	public static function plus_free_install() {

		include_once ABSPATH . 'wp-admin/includes/file.php';
		include_once ABSPATH . 'wp-admin/includes/misc.php';
		include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		include_once ABSPATH . 'wp-admin/includes/plugin.php';

		$plugin_slug = 'the-plus-addons-for-elementor-page-builder/theplus_elementor_addon.php';
		$plugin_zip  = 'https://downloads.wordpress.org/plugin/the-plus-addons-for-elementor-page-builder.latest-stable.zip';

		WP_Filesystem();
		$upgrader = new Plugin_Upgrader( new Automatic_Upgrader_Skin() );
		$upgrader->install( $plugin_zip );

		if ( file_exists( WP_PLUGIN_DIR . '/' . $plugin_slug ) ) {
			activate_plugin( $plugin_slug );
		}

		wp_safe_redirect( admin_url( 'admin.php?page=theplus_welcome_page' ) );
		exit;
	}

	/**
	 * Check if plugin is installed.
	 *
	 * @param string $slug Plugin slug.
	 * @return boolean
	 */
	public static function is_plugin_installed( $slug ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$all_plugins = get_plugins();

		if ( ! empty( $all_plugins[ $slug ] ) ) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Upgrade plugin.
	 *
	 * @param string $plugin_slug Plugin slug.
	 * @return boolean
	 */
	public static function upgrade_plugin( $plugin_slug ) {
		include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		wp_cache_flush();

		$upgrader = new \Plugin_Upgrader();
		$upgraded = $upgrader->upgrade( $plugin_slug );

		return $upgraded;
	}

	/**
	 * Install plugin.
	 *
	 * @param string $plugin_zip Plugin zip URL.
	 * @return boolean
	 */
	public static function install_plugin( $plugin_zip ) {
		include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		wp_cache_flush();

		$upgrader  = new \Plugin_Upgrader();
		$installed = $upgrader->install( $plugin_zip );

		return $installed;
	}
}

Theplus_Element_Install_Plugin::instance();
