<?php
/**
 * The file that defines the core plugin class
 *
 * @link    https://posimyth.com/
 * @since   6.3.14
 *
 * @package the-plus-addons-for-elementor-page-builder
 */

namespace TheplusAddons\Widgets;

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;

if ( ! defined( 'WPINC' ) ) {
	exit;
}

/*
 * Tooltip option Theplus.
 */
if ( ! class_exists( 'Tpaep_Plus_Tooltip' ) ) {

	/**
	 * Define Tpaep_Plus_Tooltip class
	 */
	class Tpaep_Plus_Tooltip {

		/**
		 * A reference to an instance of this class.
		 *
		 * @since 6.3.14
		 * @var   object
		 */
		private static $instance = null;

		/**
		 * Returns a singleton instance of the class.
		 *
		 * This method ensures that only one instance of the class is created (singleton pattern).
		 * If an instance doesn't exist, it creates one using the provided shortcodes.
		 *
		 * @since 6.3.14
		 *
		 * @param array $shortcodes Optional. An array of shortcodes to initialize the instance with.
		 * @return self The single instance of the class.
		 */
		public static function get_instance( $shortcodes = array() ) {

			if ( null === self::$instance ) {
				self::$instance = new self( $shortcodes );
			}

			return self::$instance;
		}

        private array $containers_data = [];

		/**
		 * Initalize integration hooks
		 * 
         * @since 6.3.14
		 *
		 * @return void
		 */
		public function __construct() {

			add_action( 'elementor/element/common/_section_style/after_section_end', array( $this, 'tpaep_register_controls' ), 10, 2 );

			add_filter( 'elementor/widget/render_content', array( $this, 'tpaep_render_element' ), 10, 2 );
		}

        /**
		 * Register controls for the Tooltip feature
		 *
		 * @since 6.3.14
		 */
        public function tpaep_register_controls( $element, $section_id ) {

            if ( ! $element || ! is_object( $element ) ) {
                return; 
            }

            $element->start_controls_section(
                'plus_section_tooltip_section',
                array(
                    'label' => esc_html__( 'Plus Tooltip', 'theplus' ),
                    'tab'   => Controls_Manager::TAB_ADVANCED,
                )
            );
            $element->add_control(
                'plus_tooltip',
                array(
                    'label'       => esc_html__( 'Tooltip', 'theplus' ),
                    'type'        => Controls_Manager::SWITCHER,
                    'label_on'    => esc_html__( 'Yes', 'theplus' ),
                    'label_off'   => esc_html__( 'No', 'theplus' ),
                    'render_type' => 'template',
                )
            );

            $element->start_controls_tabs( 'plus_tooltip_tabs' );
            $element->start_controls_tab(
                'plus_tooltip_content_tab',
                array(
                    'label'       => esc_html__( 'Content', 'theplus' ),
                    'render_type' => 'template',
                    'condition'   => array(
                        'plus_tooltip' => 'yes',
                    ),
                )
            );
                $element->add_control(
                    'plus_tooltip_content_type',
                    array(
                        'label'       => esc_html__( 'Content Type', 'theplus' ),
                        'type'        => Controls_Manager::SELECT,
                        'default'     => 'normal_desc',
                        'options'     => array(
                            'normal_desc'     => esc_html__( 'Text Content', 'theplus' ),
                            'content_wysiwyg' => esc_html__( 'WYSIWYG Editor', 'theplus' ),
                        ),
                        'render_type' => 'template',
                        'condition'   => array(
                            'plus_tooltip' => 'yes',
                        ),
                    )
                );
                $element->add_control(
                    'plus_tooltip_content_desc',
                    array(
                        'label'     => esc_html__( 'Description', 'theplus' ),
                        'type'      => Controls_Manager::TEXTAREA,
                        'rows'      => 5,
                        'default'   => esc_html__( 'Luctus nec ullamcorper mattis', 'theplus' ),
                        'condition' => array(
                            'plus_tooltip_content_type' => 'normal_desc',
                            'plus_tooltip'              => 'yes',
                        ),
                    )
                );
                $element->add_control(
                    'plus_tooltip_content_wysiwyg',
                    array(
                        'label'       => esc_html__( 'Tooltip Content', 'theplus' ),
                        'type'        => Controls_Manager::WYSIWYG,
                        'default'     => esc_html__( 'Luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'theplus' ),
                        'render_type' => 'template',
                        'condition'   => array(
                            'plus_tooltip_content_type' => 'content_wysiwyg',
                            'plus_tooltip'              => 'yes',
                        ),
                    )
                );
                $element->add_control(
                    'plus_tooltip_content_align',
                    array(
                        'label'     => esc_html__( 'Text Alignment', 'theplus' ),
                        'type'      => Controls_Manager::CHOOSE,
                        'default'   => 'center',
                        'options'   => array(
                            'left'   => array(
                                'title' => esc_html__( 'Left', 'theplus' ),
                                'icon'  => 'eicon-text-align-left',
                            ),
                            'center' => array(
                                'title' => esc_html__( 'Center', 'theplus' ),
                                'icon'  => 'eicon-text-align-center',
                            ),
                            'right'  => array(
                                'title' => esc_html__( 'Right', 'theplus' ),
                                'icon'  => 'eicon-text-align-right',
                            ),
                        ),
                        'selectors' => array(
                            '{{WRAPPER}} .tippy-tooltip .tippy-content' => 'text-align: {{VALUE}};',
                        ),
                        'condition' => array(
                            'plus_tooltip_content_type' => 'normal_desc',
                            'plus_tooltip'              => 'yes',
                        ),
                    )
                );
                $element->add_group_control(
                    Group_Control_Typography::get_type(),
                    array(
                        'name'      => 'plus_tooltip_content_typography',
                        'selector'  => '{{WRAPPER}} .tippy-tooltip .tippy-content',
                        'condition' => array(
                            'plus_tooltip_content_type' => array( 'normal_desc', 'content_wysiwyg' ),
                            'plus_tooltip'              => 'yes',
                        ),
                    )
                );
                $element->add_control(
                    'plus_tooltip_content_color',
                    array(
                        'label'     => esc_html__( 'Text Color', 'theplus' ),
                        'type'      => Controls_Manager::COLOR,
                        'selectors' => array(
                            '{{WRAPPER}} .tippy-tooltip .tippy-content,{{WRAPPER}} .tippy-tooltip .tippy-content p' => 'color: {{VALUE}}',
                        ),
                        'condition' => array(
                            'plus_tooltip_content_type' => array( 'normal_desc', 'content_wysiwyg' ),
                            'plus_tooltip'              => 'yes',
                        ),
                    )
                );
            $element->end_controls_tab();
            $element->start_controls_tab(
                'plus_tooltip_styles_tab',
                array(
                    'label'     => esc_html__( 'Style', 'theplus' ),
                    'condition' => array(
                        'plus_tooltip' => 'yes',
                    ),
                )
            );
                $element->add_group_control(
                    \Theplus_Tooltips_Option_Group::get_type(),
                    array(
                        'label'       => esc_html__( 'Tooltip Options', 'theplus' ),
                        'name'        => 'tooltip_opt',
                        'render_type' => 'template',
                        'condition'   => array(
                            'plus_tooltip' => array( 'yes' ),
                        ),
                    )
                );
                $element->add_group_control(
                    \Theplus_Tooltips_Option_Style_Group::get_type(),
                    array(
                        'label'       => esc_html__( 'Style Options', 'theplus' ),
                        'name'        => 'tooltip_style',
                        'render_type' => 'template',
                        'condition'   => array(
                            'plus_tooltip' => array( 'yes' ),
                        ),
                    )
                );
            $element->end_controls_tab();
            $element->end_controls_tabs();

            $element->end_controls_section();
        }
    
        /**
		 * Apply Tooltip settings before rendering the widget.
		 *
		 * @since 6.3.14
		 */
        public function tpaep_render_element( $content, $element ) {
            
            $settings  = $element->get_settings_for_display();
            $widget_id = $element->get_id();

            $plus_tooltip = ! empty( $settings['plus_tooltip'] ) ? $settings['plus_tooltip'] : '';

            if ( empty( $plus_tooltip ) || 'yes' !== $plus_tooltip ) {
				return $content;
			}

            if ( 'yes' === $plus_tooltip ) {

                $uid_widget = 'plus' . $widget_id;

                $tooltip_duration_in  = '';
                $tooltip_duration_out = '';

                $plus_before_content = '';
                $plus_after_content = '';

                $element->add_render_attribute( '_tooltip', 'data-tippy', '', true );
                if ( ! empty( $settings['plus_tooltip_content_type'] ) && $settings['plus_tooltip_content_type'] == 'normal_desc' ) {
                    $element->add_render_attribute( '_tooltip', 'title', wp_kses_post( $settings['plus_tooltip_content_desc'] ), true );
                } elseif ( ! empty( $settings['plus_tooltip_content_type'] ) && $settings['plus_tooltip_content_type'] == 'content_wysiwyg' ) {
                    $tooltip_content = $settings['plus_tooltip_content_wysiwyg'];
                    $element->add_render_attribute( '_tooltip', 'title', wp_kses_post( $tooltip_content ), true );
                }
            
                $plus_tooltip_position = ! empty( $settings['tooltip_opt_plus_tooltip_position'] ) ? $settings['tooltip_opt_plus_tooltip_position'] : 'top';
                $element->add_render_attribute( '_tooltip', 'data-tippy-placement', $plus_tooltip_position, true );
            
                $tooltip_interactive = isset( $settings['tooltip_opt_plus_tooltip_interactive'] ) && $settings['tooltip_opt_plus_tooltip_interactive'] == 'yes' ? 'true' : 'false';
                $element->add_render_attribute( '_tooltip', 'data-tippy-interactive', $tooltip_interactive, true );
            
                $plus_tooltip_theme = ! empty( $settings['tooltip_opt_plus_tooltip_theme'] ) ? $settings['tooltip_opt_plus_tooltip_theme'] : 'dark';
                $element->add_render_attribute( '_tooltip', 'data-tippy-theme', $plus_tooltip_theme, true );
            
                $tooltip_arrow = ( $settings['tooltip_opt_plus_tooltip_arrow'] != 'none' || empty( $settings['tooltip_opt_plus_tooltip_arrow'] ) ) ? 'true' : 'false';
                $element->add_render_attribute( '_tooltip', 'data-tippy-arrow', $tooltip_arrow, true );
            
                $plus_tooltip_arrow = ! empty( $settings['tooltip_opt_plus_tooltip_arrow'] ) ? $settings['tooltip_opt_plus_tooltip_arrow'] : 'sharp';
                $element->add_render_attribute( '_tooltip', 'data-tippy-arrowtype', $plus_tooltip_arrow, true );
            
                $plus_tooltip_animation = ! empty( $settings['tooltip_opt_plus_tooltip_animation'] ) ? $settings['tooltip_opt_plus_tooltip_animation'] : 'shift-toward';
                $element->add_render_attribute( '_tooltip', 'data-tippy-animation', $plus_tooltip_animation, true );
            
                $plus_tooltip_x_offset = isset( $settings['tooltip_opt_plus_tooltip_x_offset'] ) ? $settings['tooltip_opt_plus_tooltip_x_offset'] : 0;
                $plus_tooltip_y_offset = isset( $settings['tooltip_opt_plus_tooltip_y_offset'] ) ? $settings['tooltip_opt_plus_tooltip_y_offset'] : 0;
                $element->add_render_attribute( '_tooltip', 'data-tippy-offset', $plus_tooltip_x_offset . ',' . $plus_tooltip_y_offset, true );
            
                $tooltip_duration_in  = isset( $settings['tooltip_opt_plus_tooltip_duration_in'] ) ? $settings['tooltip_opt_plus_tooltip_duration_in'] : 250;
                $tooltip_duration_out = isset( $settings['tooltip_opt_plus_tooltip_duration_out'] ) ? $settings['tooltip_opt_plus_tooltip_duration_out'] : 200;
                $tooltip_trigger      = ! empty( $settings['tooltip_opt_plus_tooltip_triggger'] ) ? $settings['tooltip_opt_plus_tooltip_triggger'] : 'mouseenter';
                $tooltip_arrowtype    = ! empty( $settings['tooltip_opt_plus_tooltip_arrow'] ) ? $settings['tooltip_opt_plus_tooltip_arrow'] : 'sharp';
            
                $plus_before_content .= '<div id="' . esc_attr( $uid_widget ) . '" class=" plus-widget-wrapper" ' . $element->get_render_attribute_string( '_tooltip' ) . '>';
            
                $plus_inline_tippy_js = 'jQuery( document ).ready(function() {
                "use strict";
                    if(typeof tippy === "function"){
                        tippy( "#' . esc_attr( $uid_widget ) . '" , {
                            arrowType : "' . esc_attr( $tooltip_arrowtype ) . '",
                            duration : [' . esc_attr( $tooltip_duration_in ) . ',' . esc_attr( $tooltip_duration_out ) . '],
                            trigger : "' . esc_attr( $tooltip_trigger ) . '",
                            appendTo: document.querySelector("#' . esc_attr( $uid_widget ) . '")
                        });
                    }
                });';

                $plus_after_content .= wp_print_inline_script_tag( $plus_inline_tippy_js );

                $plus_after_content .= '</div>';

                $plus_output = $plus_before_content . $content . $plus_after_content;

                return $plus_output;
            }
        }
	}
}

Tpaep_Plus_Tooltip::get_instance();