<?php

class MeowPro_WR2X_Rest
{
  private $core;
  private $engine;
	private $namespace = 'wp-retina-2x/v1';

	public function __construct( $core, $engine ) {
		$this->core = $core;
		$this->engine = $engine;
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );
	}

	function rest_api_init() {
		if ( !current_user_can( 'administrator' ) ) {
			return;
		} 
		register_rest_route( $this->namespace, '/upload_retina', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_upload_retina' )
		) );
		register_rest_route( $this->namespace, '/upload_retina_fullsize', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_upload_retina_fullsize' )
		) );
		register_rest_route( $this->namespace, '/delete_retina_fullsize', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_delete_retina_fullsize' )
		) );
		register_rest_route( $this->namespace, '/upload_webp', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_upload_webp' )
		) );
		register_rest_route( $this->namespace, '/upload_webp_fullsize', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_upload_webp_fullsize' )
		) );
		register_rest_route( $this->namespace, '/delete_webp_fullsize', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_delete_webp_fullsize' )
		) );

		// AI
		register_rest_route( $this->namespace, '/generate_upscale_retina', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_generate_upscale_retina' )
		) );

  }

	function rest_delete_retina_fullsize( $request ) {
		if ( !current_user_can( 'upload_files' ) ) {
			$this->core->log( "You do not have permission to upload files." );
			return __( "You do not have permission to upload files.", 'wp-retina-2x' );
		}

		// Check errors
		$params = $request->get_json_params();
		$mediaId = isset( $params['mediaId'] ) ? (int)$params['mediaId'] : null;
		if ( empty( $mediaId ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "The Media ID is required." ] );
		}

		// Delete Retina
		$this->engine->delete_retina_fullsize( $mediaId );
		$info = $this->core->get_media_status_one( $mediaId );

		// Delete the history for the AI generated sizes
		$history = $this->core->get_option( 'ai_history', [] );
		if ( isset( $history[ $mediaId ][ 'full_size' ] ) ) {
			unset( $history[ $mediaId ][ 'full_size' ]  );
			$this->core->update_option( 'ai_history', $history );
		}

		return new WP_REST_Response( [ 'success' => true, 'data' => $info  ], 200 );
	}

	function rest_delete_webp_fullsize( $request ) {
		if ( !current_user_can( 'upload_files' ) ) {
			$this->core->log( "You do not have permission to upload files." );
			return __( "You do not have permission to upload files.", 'wp-retina-2x' );
		}

		// Check errors
		$params = $request->get_json_params();
		$mediaId = isset( $params['mediaId'] ) ? (int)$params['mediaId'] : null;
		if ( empty( $mediaId ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "The Media ID is required." ] );
		}

		// Delete WebP
		$this->engine->delete_webp_fullsize( $mediaId );
		$info = $this->core->get_media_status_one( $mediaId );
		return new WP_REST_Response( [ 'success' => true, 'data' => $info  ], 200 );
	}

	function rest_upload_retina( $request ) {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		$files = $request->get_file_params();
		$file = $files['file'];
		$tmpfname = $file['tmp_name'];
		$filename = $file['name'];

		// Check errors
		if ( empty( $tmpfname ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "A file is required." ] );
		}
		$error = $this->check_upload( $tmpfname, $filename );
		if ( !empty( $error ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $error ], 200 );
		}

		$ftype = wp_check_filetype( $filename );
		$image = wp_get_image_editor( $tmpfname, array ( 'mime_type' => $ftype['type'] ) );

		if ( is_wp_error( $image ) ) {
			$this->core->log( 'Error getting image editor: ' . $image->get_error_message() );
			return new WP_REST_Response( [ 'success' => false, 'message' => $image->get_error_message() ] );
		}

		$size = $image->get_size();

		try {
			// Halve the size of the uploaded image
			if ( $size['width'] >= $size['height'] ) $image->resize( round($size['width'] * .5), null );
			else $image->resize( null, round($size['height'] * .5) );
			$image->set_quality( get_option('wr2x_quality', 75 ) );
			$halved = $image->save( $tmpfname . 'H', $ftype['type'] );
			if ( !$halved ) throw new Exception( "Failed to halve the uploaded image" );
			if ( is_wp_error($halved) ) throw new Exception( $halved->get_error_message() );

			// Upload the halved image
			$content = file_get_contents( $halved['path'] );
			if ( $content === false ) throw new Exception( "Couldn't read the uploaded file: {$halved['file']}" );
			$uploaded = wp_upload_bits( $filename, null, $content );
			if ( isset($uploaded['error']) && $uploaded['error'] ) throw new Exception( $uploaded['error'] );

			// Register the file as a new attachment
			$attachTo = 0; // TODO Support specifying which post the media attach to
			$attachment = array (
				'post_mime_type' => $ftype['type'],
				'post_parent' => $attachTo,
				'post_title' => preg_replace( '/\.[^.]+$/', '', $filename ),
				'post_content' => '',
				'post_status' => 'inherit'
			);
			$attachmentId = wp_insert_attachment( $attachment, $uploaded['file'], $attachTo );
			if ( !$attachmentId ) throw new Exception( "Couldn't add an attachment file: {$uploaded['file']}" );
			if ( is_wp_error($attachmentId) ) throw new Exception( $attachmentId->get_error_message() );
			$meta = wp_generate_attachment_metadata( $attachmentId, $uploaded['file'] );
			wp_update_attachment_metadata( $attachmentId, $meta );
			return $this->rest_upload_retina_fullsize( $request, $attachmentId );
		} 
		catch ( Exception $e ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $e->getMessage() ] );
		}
	}

	function rest_upload_webp( $request ) {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		$files = $request->get_file_params();
		$file = $files['file'];
		$tmpfname = $file['tmp_name'];
		$filename = $file['name'];

		// Check errors
		if ( empty( $tmpfname ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "A file is required." ] );
		}
		$error = $this->check_upload( $tmpfname, $filename );
		if ( !empty( $error ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $error ], 200 );
		}

		$ftype = wp_check_filetype( $filename );
		$image = wp_get_image_editor( $tmpfname, array ( 'mime_type' => $ftype['type'] ) );
		
		if ( is_wp_error( $image ) ) {
			$this->core->log( 'Error getting image editor: ' . $image->get_error_message() );
			return new WP_REST_Response( [ 'success' => false, 'message' => $image->get_error_message() ] );
		}
		
		$size = $image->get_size();

		try {
			// Halve the size of the uploaded image
			if ( $size['width'] >= $size['height'] ) {
				$image->resize( round($size['width'] * .5), null );
			} else {
				$image->resize( null, round($size['height'] * .5) );
			}
			$quality = $this->core->get_option( 'quality' );
			if ( empty( $quality ) ) {
				$quality = apply_filters( 'jpeg_quality', 75 );
			}
			$image->set_quality( $quality );
			$halved = $image->save( $tmpfname . 'H', $ftype['type'] );
			if ( !$halved ) {
				throw new Exception( "Failed to halve the uploaded image" );
			}
			if ( is_wp_error($halved) ) {
				throw new Exception( $halved->get_error_message() );
			}

			// Upload the halved image
			$content = file_get_contents( $halved['path'] );
			if ( $content === false ) {
				throw new Exception( "Couldn't read the uploaded file: {$halved['file']}" );
			}
			$uploaded = wp_upload_bits( $filename, null, $content );
			if ( isset($uploaded['error']) && $uploaded['error'] ) {
				throw new Exception( $uploaded['error'] );
			}

			// Register the file as a new attachment
			$attachTo = 0; // TODO Support specifying which post the media attach to
			$attachment = array (
				'post_mime_type' => $ftype['type'],
				'post_parent' => $attachTo,
				'post_title' => preg_replace( '/\.[^.]+$/', '', $filename ),
				'post_content' => '',
				'post_status' => 'inherit'
			);
			$attachment_id = wp_insert_attachment( $attachment, $uploaded['file'], $attachTo );
			if ( !$attachment_id ) {
				throw new Exception( "Couldn't add an attachment file: {$uploaded['file']}" );
			}
			if ( is_wp_error($attachment_id) ) {
				throw new Exception( $attachment_id->get_error_message() );
			}
			$meta = wp_generate_attachment_metadata( $attachment_id, $uploaded['file'] );
			wp_update_attachment_metadata( $attachment_id, $meta );
			return $this->rest_upload_webp_fullsize( $request, $attachment_id );
		} 
		catch ( Exception $e ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $e->getMessage() ] );
		}
	}

	function check_upload( $tmpfname, $filename ) {
		if ( !current_user_can( 'upload_files' ) ) {
			$this->core->log( "You do not have permission to upload files." );
			unlink( $tmpfname );
			return __( "You do not have permission to upload files.", 'wp-retina-2x' );
		}
		$file_info = getimagesize( $tmpfname );
		if ( empty( $file_info ) ) {
			$this->core->log( "The file is not an image or the upload went wrong." );
			unlink( $tmpfname );
			return __( "The file is not an image or the upload went wrong.", 'wp-retina-2x' );
		}
		$filedata = wp_check_filetype_and_ext( $tmpfname, $filename );
		if ( $filedata["ext"] == "" ) {
			$this->core->log( "You cannot use this file (wrong extension? wrong type?)." );
			unlink( $tmpfname );
			return __( "You cannot use this file (wrong extension? wrong type?).", 'wp-retina-2x' );
		}
		return null;
	}

	function rest_generate_upscale_retina( $request ) {
	
		$params = $request->get_json_params();
	
		if ( ! isset( $params['mediaId'] ) || empty( $params['mediaId'] ) ) {
			return new WP_REST_Response( [
				'success' => false,
				'message' => 'The Media ID is required.'
			], 400 );
		}

		$service = $this->core->get_option( 'ai_service', 'none' );
		$mediaId = intval( $params['mediaId'] );
	
		// Call the upscale function
		$res = $this->core->use_ai_upscale( $mediaId, $service );
	
		if ( $res['status'] ) {
			$history = $this->core->get_option( 'ai_history', [] );
			$history[ $mediaId ]['full_size'] = [
				'time' => time(),
				'operation' => 'upscale',
				'service' => $service,

				'width' => $res['meta']['output']['width'],
				'height' => $res['meta']['output']['height'],
			];

			$this->core->update_option( 'ai_history', $history );

			// Success - Get the temp URL
			$tmp_url = $res['data'];

			// Download the image from the temp URL
			// Check if the download_url() function is available
			if ( ! function_exists( 'download_url' ) ) {
				require_once ABSPATH . 'wp-admin/includes/file.php';
			}

			$temp_file = download_url( $tmp_url );
			if ( is_wp_error( $temp_file ) ) {
				return new WP_REST_Response( [
					'success' => false,
					'message' => 'Error downloading image: ' . $temp_file->get_error_message(). ' - ' . $tmp_url
				], 500 );
			}
	
			// Prepare parameters for rest_upload_retina_fullsize
			$file_params = [
				'file'     => [
					'name'     => basename( get_attached_file( $mediaId ) ),
					'tmp_name' => $temp_file,
				],
			];
	
			$body_params = [
				'mediaId' => $mediaId,
			];

			$request = [
				'body_params' => $body_params,
				'file_params' => $file_params,
			];
	
			// Call rest_upload_retina_fullsize with array params
			$upload_response = $this->rest_upload_retina_fullsize( $request );

			// Clean up temporary file
			unlink( $temp_file );

			// Check if the upload was successful
			if ( ! $upload_response->data['success'] ) {
				return new WP_REST_Response( [
					'success' => false,
					'message' => 'Error uploading image: ' . $upload_response->data['message']
				], 500 );
			}
	
			// Return the upload response
			return new WP_REST_Response( [
				'success' => true,
				'data' => $upload_response->data['data'],
				'history' => $history
			], 200 );

		
		} else {
			// Failure - return error message
			return new WP_REST_Response( [
				'success' => false,
				'message' => $res['message']
			], 500 );
		}
	}

	function rest_upload_retina_fullsize( $request, $mediaId = null ) {
		require_once ABSPATH . 'wp-admin/includes/image.php';

		$params = is_array( $request ) ? $request['body_params'] : $request->get_body_params();
		$mediaId = $mediaId ? $mediaId : $params['mediaId'];


		$files = is_array( $request ) ? $request['file_params'] : $request->get_file_params();
		$file = $files['file'];
		$tmpfname = $file['tmp_name'];

		if ( empty( $mediaId ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "The Media ID is required." ] );
		}
		if ( empty( $tmpfname ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "A file is required." ] );
		}
		$error = $this->check_upload( $tmpfname, $file['name'] );
		if ( !empty( $error ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $error ], 200 );
		}

		$meta = wp_get_attachment_metadata( $mediaId );
		$current_file = get_attached_file( $mediaId );
		$pathinfo = pathinfo( $current_file );
		$retinafile = trailingslashit( $pathinfo['dirname'] ) . $pathinfo['filename'] . $this->core->retina_extension() . $pathinfo['extension'];
		do_action( 'wr2x_before_upload_retina', $mediaId, $retinafile );

		// Let's clean everything first
		if ( file_exists( $retinafile ) )
			unlink( $retinafile );

		// Insert the new file and delete the temporary one
		list( $width, $height ) = getimagesize( $tmpfname );
		if ( !$this->core->are_dimensions_ok( $width, $height, $meta['width'] * 2, $meta['height'] * 2 ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "This image has a resolution of {$width}×{$height} but your Full-Size image requires a retina image of at least " . ( $meta['width'] * 2 ) . "x" . ( $meta['height'] * 2 ) . "." ], 200 );
		}
		$this->engine->resize( $tmpfname, $meta['width'] * 2, $meta['height'] * 2, null, $retinafile );
		chmod( $retinafile, 0644 );
		unlink( $tmpfname );
		
		// Finished
		do_action( 'wr2x_upload_retina', $mediaId, $retinafile );
		$info = $this->core->get_media_status_one( $mediaId );
		return new WP_REST_Response( [ 'success' => true, 'data' => $info  ], 200 );
	}

	function rest_upload_webp_fullsize( $request, $media_id = null ) {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		$params = $request->get_body_params();
		$media_id = $media_id ? $media_id : $params['mediaId'];
		$files = $request->get_file_params();
		$file = $files['file'];
		$tmp_fname = $file['tmp_name'];

		if ( empty( $media_id ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "The Media ID is required." ] );
		}
		if ( empty( $tmp_fname ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "A file is required." ] );
		}
		$error = $this->check_upload( $tmp_fname, $file['name'] );
		if ( !empty( $error ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $error ], 200 );
		}

		$meta = wp_get_attachment_metadata( $media_id );
		$current_file = get_attached_file( $media_id );
		$pathinfo = pathinfo( $current_file );
		$webp_file = trailingslashit( $pathinfo['dirname'] ) . $pathinfo['filename'] . '.' . $pathinfo['extension'] . $this->core->webp_avif_extension();
		do_action( 'wr2x_before_upload_webp', $media_id, $webp_file );

		// Let's clean everything first
		if ( file_exists( $webp_file ) )
			unlink( $webp_file );

		// Insert the new file and delete the temporary one
		$this->engine->resize( $tmp_fname, $meta['width'], $meta['height'], null, $webp_file );
		chmod( $webp_file, 0644 );
		unlink( $tmp_fname );
		
		// Finished
		do_action( 'wr2x_upload_webp', $media_id, $webp_file );
		$info = $this->core->get_media_status_one( $media_id );
		return new WP_REST_Response( [ 'success' => true, 'data' => $info  ], 200 );
	}

}

?>