<?php

namespace WeDevs\Wpuf\Pro\Fields;

/**
 * Facebook URL Field Class
 *
 * @since 4.2.0
 */
class Field_Facebook extends Field_Social {

    const FACEBOOK_SVG_ICON = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 640"><!--!Font Awesome Free v7.0.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2025 Fonticons, Inc.--><path d="M160 96C124.7 96 96 124.7 96 160L96 480C96 515.3 124.7 544 160 544L258.2 544L258.2 398.2L205.4 398.2L205.4 320L258.2 320L258.2 286.3C258.2 199.2 297.6 158.8 383.2 158.8C399.4 158.8 427.4 162 438.9 165.2L438.9 236C432.9 235.4 422.4 235 409.3 235C367.3 235 351.1 250.9 351.1 292.2L351.1 320L434.7 320L420.3 398.2L351 398.2L351 544L480 544C515.3 544 544 515.3 544 480L544 160C544 124.7 515.3 96 480 96L160 96z"/></svg>';

    public function __construct() {
        $this->name                 = __( 'Facebook', 'wpuf-pro' );
        $this->input_type           = 'facebook_url';
        $this->icon                 = 'facebook';
        $this->icon_svg             = self::FACEBOOK_SVG_ICON;
        $this->base_url             = 'https://facebook.com/';
        $this->username_pattern     = '^[a-zA-Z0-9\.]{1,50}$';
        $this->url_pattern          = '/^(?:https?:\/\/)?(?:www\.)?(?:facebook\.com|fb\.com)\/([A-Za-z0-9\.]{1,50})\/?$/i';
        $this->max_username_length  = 50;
        $this->example_username     = 'username';
    }

    /**
     * Get the field props
     *
     * @return array
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'       => 'facebook_url',
            'required'         => 'no',
            'name'             => 'wpuf_social_facebook',
            'meta_key'         => 'wpuf_social_facebook',
            'help'             => __( 'Enter your Facebook username or full URL', 'wpuf-pro' ),
            'css'              => '',
            'placeholder'      => __( 'username', 'wpuf-pro' ),
            'default'          => '',
            'size'             => 40,
            'width'            => 'large',
            'id'               => 0,
            'is_new'           => true,
            'show_icon'        => 'yes',
            'icon'              => '<svg class="wpuf-facebook-svg" width="20" height="25" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" aria-label="Facebook" role="img"><path d="M14.1061 6.68815H11.652C10.7822 6.68815 10.0752 7.3899 10.0688 8.25975L9.99768 17.8552M8.40234 11.6676H12.4046M2.08398 9.9987C2.08398 6.26675 2.08398 4.40077 3.24335 3.2414C4.40273 2.08203 6.2687 2.08203 10.0007 2.08203C13.7326 2.08203 15.5986 2.08203 16.758 3.2414C17.9173 4.40077 17.9173 6.26675 17.9173 9.9987C17.9173 13.7306 17.9173 15.5966 16.758 16.756C15.5986 17.9154 13.7326 17.9154 10.0007 17.9154C6.2687 17.9154 4.40273 17.9154 3.24335 16.756C2.08398 15.5966 2.08398 13.7306 2.08398 9.9987Z" stroke="#079669" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            'readonly'         => 'no',
            'open_in_new_window' => 'yes',
            'nofollow'         => 'yes',
            'username_validation' => 'strict',
            'wpuf_cond'        => '',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Render the social media field
     *
     * @since 4.1.7
     *
     * @param array $field_settings
     * @param int $form_id
     * @param string $type
     * @param int $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        // Enqueue Facebook validation script
        wp_enqueue_script( 'wpuf-frontend-facebook-validation' );

        if ( isset( $post_id ) && $post_id != 0 ) {
            $value = $this->get_user_data( $post_id, $field_settings['name'] );
        } else {
            $value = $field_settings['default'];
        }

        $show_icon = isset( $field_settings['show_icon'] ) ? $field_settings['show_icon'] : 'no';
        $readonly = isset( $field_settings['readonly'] ) ? $field_settings['readonly'] : 'no';
        $css_class = isset( $field_settings['css'] ) ? $field_settings['css'] : '';
        $width_class = isset( $field_settings['width'] ) ? 'wpuf-' . $field_settings['width'] : 'wpuf-medium';
        ?>

        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <div class="wpuf-label-with-icon">
                <?php
                $this->print_label( $field_settings, $form_id );
                // Show icon next to label if enabled
                if ( $show_icon === 'yes' && ! empty( $this->icon_svg ) ) : ?>
                    <span class="wpuf-social-label-icon">
                        <?php
                        echo wp_kses( $this->icon_svg, $this->get_allowed_svg_attributes() );
                        ?>
                    </span>
                <?php endif; ?>
            </div>

            <div class="wpuf-fields">
                <div class="wpuf-social-input-wrapper <?php echo esc_attr( $width_class ); ?>">
                    <?php
                    $input_type = 'text';
                    ?>
                    <input
                        class="textfield <?php echo esc_attr( $input_type ); ?> <?php echo 'wpuf_' . $field_settings['name'] . '_' . $form_id; ?> <?php echo esc_attr( $css_class ); ?>"
                        id="<?php echo $field_settings['name'] . '_' . $form_id; ?>"
                        type="<?php echo esc_attr( $input_type ); ?>"
                        data-required="<?php echo $field_settings['required'] ?>"
                        data-type="<?php echo esc_attr( $input_type ); ?>"
                        data-username-validation="<?php echo isset( $field_settings['username_validation'] ) ? esc_attr( $field_settings['username_validation'] ) : 'strict'; ?>"
                        name="<?php echo esc_attr( $field_settings['name'] ); ?>"
                        placeholder="<?php echo esc_attr( $field_settings['placeholder'] ); ?>"
                        value="<?php echo esc_attr( $value ); ?>"
                        size="<?php echo esc_attr( isset( $field_settings['size'] ) ? $field_settings['size'] : 40 ); ?>"
                        <?php if ( isset( $this->username_pattern ) && ! empty( $this->username_pattern ) ): ?>
                        pattern="<?php echo esc_attr( $this->username_pattern ); ?>"
                        <?php endif; ?>
                        title="<?php echo esc_attr( sprintf( __( 'Enter a valid %s username or URL', 'wpuf-pro' ), $this->name ) ); ?>"
                        <?php echo $readonly === 'yes' ? 'readonly' : ''; ?>
                    />
                </div>

                <?php $this->help_text( $field_settings ); ?>
            </div>

            <?php if ( isset( $field_settings['word_restriction'] ) && $field_settings['word_restriction'] ) : ?>
                <div class="wpuf-wordlimit-message wpuf-help"></div>
            <?php endif; ?>
        </li>

        <?php $this->conditional_logic( $field_settings, $form_id ); ?>

        <?php
    }

    /**
     * Get options settings for the form builder
     *
     * @return array
     */
    public function get_options_settings() {
        // Get default options but exclude fields that would duplicate parent class settings
        $default_options = $this->get_default_option_settings( false, array('dynamic') );

        $settings = [
            [
                'name'      => 'meta_key',
                'title'     => __( 'Meta Key', 'wpuf-pro' ),
                'type'      => 'text',
                'default'   => 'wpuf_social_facebook',
                'section'   => 'basic',
                'priority'  => 15,
                'readonly'  => true,
                'help_text' => __( 'The meta key used to store this field data in the database', 'wpuf-pro' ),
            ],
            [
                'name'      => 'placeholder',
                'title'     => __( 'Placeholder', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'basic',
                'priority'  => 15.5,
                'help_text' => __( 'Placeholder text for the input field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'default',
                'title'     => __( 'Default value', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 15.7,
                'help_text' => __( 'Enter a default value for this field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'username_validation',
                'title'     => __( 'Username Validation', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'strict' => __( 'Strict (Facebook username format)', 'wpuf-pro' ),
                    'allow_full_url' => __( 'Allow full URLs', 'wpuf-pro' ),
                ],
                'default'   => 'strict',
                'section'   => 'basic',
                'priority'  => 16,
                'inline'    => true,
                'help_text' => __( 'Choose validation type: strict username format or allow full URLs', 'wpuf-pro' ),
            ],
            [
                'name'      => 'show_icon',
                'title'     => __( 'Icon show', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                    'no'  => __( 'No', 'wpuf-pro' ),
                ],
                'default'   => 'yes',
                'section'   => 'basic',
                'priority'  => 17,
                'inline'    => true,
                'help_text' => __( 'Show the social platform icon next to the field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'readonly',
                'title'     => __( 'Read Only', 'wpuf-pro' ),
                'type'      => 'checkbox',
                'is_single_opt' => true,
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                ],
                'section'   => 'basic',
                'priority'  => 18,
                'help_text' => __( 'Make this field read-only', 'wpuf-pro' ),
            ],
            [
                'name'      => 'size',
                'title'     => __( 'Size', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 20,
                'help_text' => __( 'Size of the input field', 'wpuf-pro' ),
            ],
        ];

        return array_merge( $default_options, $settings );
    }
}
