<?php

namespace WeDevs\Wpuf\Pro\Fields;

/**
 * Instagram URL Field Class
 *
 * @since 4.2.0
 */
class Field_Instagram extends Field_Social {

    const INSTAGRAM_SVG_ICON = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 640"><!--!Font Awesome Free v7.0.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2025 Fonticons, Inc.--><path d="M290.4 275.7C274 286 264.5 304.5 265.5 323.8C266.6 343.2 278.2 360.4 295.6 368.9C313.1 377.3 333.8 375.5 349.6 364.3C366 354 375.5 335.5 374.5 316.2C373.4 296.8 361.8 279.6 344.4 271.1C326.9 262.7 306.2 264.5 290.4 275.7zM432.7 207.3C427.5 202.1 421.2 198 414.3 195.3C396.2 188.2 356.7 188.5 331.2 188.8C327.1 188.8 323.3 188.9 320 188.9C316.7 188.9 312.8 188.9 308.6 188.8C283.1 188.5 243.8 188.1 225.7 195.3C218.8 198 212.6 202.1 207.3 207.3C202 212.5 198 218.8 195.3 225.7C188.2 243.8 188.6 283.4 188.8 308.9C188.8 313 188.9 316.8 188.9 320C188.9 323.2 188.9 327 188.8 331.1C188.6 356.6 188.2 396.2 195.3 414.3C198 421.2 202.1 427.4 207.3 432.7C212.5 438 218.8 442 225.7 444.7C243.8 451.8 283.3 451.5 308.8 451.2C312.9 451.2 316.7 451.1 320 451.1C323.3 451.1 327.2 451.1 331.4 451.2C356.9 451.5 396.2 451.9 414.3 444.7C421.2 442 427.4 437.9 432.7 432.7C438 427.5 442 421.2 444.7 414.3C451.9 396.3 451.5 356.9 451.2 331.3C451.2 327.1 451.1 323.2 451.1 319.9C451.1 316.6 451.1 312.8 451.2 308.5C451.5 283 451.9 243.6 444.7 225.5C442 218.6 437.9 212.4 432.7 207.1L432.7 207.3zM365.6 251.8C383.7 263.9 396.2 282.7 400.5 304C404.8 325.3 400.3 347.5 388.2 365.6C382.2 374.6 374.5 382.2 365.6 388.2C356.7 394.2 346.6 398.3 336 400.4C314.7 404.6 292.5 400.2 274.4 388.1C256.3 376 243.8 357.2 239.5 335.9C235.2 314.6 239.7 292.4 251.7 274.3C263.7 256.2 282.6 243.7 303.9 239.4C325.2 235.1 347.4 239.6 365.5 251.6L365.6 251.6zM394.8 250.5C391.7 248.4 389.2 245.4 387.7 241.9C386.2 238.4 385.9 234.6 386.6 230.8C387.3 227 389.2 223.7 391.8 221C394.4 218.3 397.9 216.5 401.6 215.8C405.3 215.1 409.2 215.4 412.7 216.9C416.2 218.4 419.2 220.8 421.3 223.9C423.4 227 424.5 230.7 424.5 234.5C424.5 237 424 239.5 423.1 241.8C422.2 244.1 420.7 246.2 419 248C417.3 249.8 415.1 251.2 412.8 252.2C410.5 253.2 408 253.7 405.5 253.7C401.7 253.7 398 252.6 394.9 250.5L394.8 250.5zM544 160C544 124.7 515.3 96 480 96L160 96C124.7 96 96 124.7 96 160L96 480C96 515.3 124.7 544 160 544L480 544C515.3 544 544 515.3 544 480L544 160zM453 453C434.3 471.7 411.6 477.6 386 478.9C359.6 480.4 280.4 480.4 254 478.9C228.4 477.6 205.7 471.7 187 453C168.3 434.3 162.4 411.6 161.2 386C159.7 359.6 159.7 280.4 161.2 254C162.5 228.4 168.3 205.7 187 187C205.7 168.3 228.5 162.4 254 161.2C280.4 159.7 359.6 159.7 386 161.2C411.6 162.5 434.3 168.3 453 187C471.7 205.7 477.6 228.4 478.8 254C480.3 280.3 480.3 359.4 478.8 385.9C477.5 411.5 471.7 434.2 453 452.9L453 453z"/></svg>';

    public function __construct() {
        $this->name                 = __( 'Instagram', 'wpuf-pro' );
        $this->input_type           = 'instagram_url';
        $this->icon                 = 'instagram';
        $this->icon_svg             = self::INSTAGRAM_SVG_ICON;
        $this->base_url             = 'https://instagram.com/';
        $this->username_pattern     = '^[a-zA-Z0-9_\.]{1,30}$';
        $this->url_pattern          = '/^(?:https?:\/\/)?(?:www\.)?(?:instagram\.com|instagr\.am)\/([A-Za-z0-9_\.]{1,30})\/?$/i';
        $this->max_username_length  = 30;
        $this->example_username     = 'username';
    }

    /**
     * Get the field props
     *
     * @return array
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'       => 'instagram_url',
            'required'         => 'no',
            'name'             => 'wpuf_social_instagram',
            'meta_key'         => 'wpuf_social_instagram',
            'help'             => __( 'Enter your Instagram username or full URL', 'wpuf-pro' ),
            'css'              => '',
            'placeholder'      => __( 'username', 'wpuf-pro' ),
            'default'          => '',
            'size'             => 40,
            'width'            => 'large',
            'id'               => 0,
            'is_new'           => true,
            'show_icon'        => 'yes',
            'icon'              => '<svg class="wpuf-instagram-svg" width="20" height="25" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" aria-label="Instagram" role="img"><path d="M2.24335 2.2414L1.71302 1.71107L1.71302 1.71107L2.24335 2.2414ZM15.758 2.2414L16.2883 1.71108L16.2883 1.71106L15.758 2.2414ZM15.758 15.756L16.2883 16.2864L16.2883 16.2863L15.758 15.756ZM2.24335 15.756L1.71301 16.2864L1.71303 16.2864L2.24335 15.756ZM13.5905 5.16536C14.0047 5.16536 14.3405 4.82958 14.3405 4.41536C14.3405 4.00115 14.0047 3.66536 13.5905 3.66536V5.16536ZM13.583 3.66536C13.1688 3.66536 12.833 4.00115 12.833 4.41536C12.833 4.82958 13.1688 5.16536 13.583 5.16536V3.66536ZM1.08398 8.9987H1.83398C1.83398 7.11152 1.83558 5.77159 1.97222 4.75527C2.10596 3.76052 2.35657 3.18884 2.77368 2.77173L2.24335 2.2414L1.71302 1.71107C0.97076 2.45333 0.641695 3.39432 0.485593 4.5554C0.332392 5.6949 0.333984 7.15392 0.333984 8.9987H1.08398ZM2.24335 2.2414L2.77368 2.77173C3.19079 2.35462 3.76248 2.104 4.75722 1.97026C5.77354 1.83362 7.11347 1.83203 9.00065 1.83203V1.08203V0.332031C7.15588 0.332031 5.69685 0.330438 4.55735 0.48364C3.39627 0.639742 2.45529 0.968807 1.71302 1.71107L2.24335 2.2414ZM9.00065 1.08203V1.83203C10.8878 1.83203 12.2277 1.83362 13.2441 1.97026C14.2388 2.104 14.8105 2.35462 15.2277 2.77174L15.758 2.2414L16.2883 1.71106C15.546 0.968806 14.605 0.639742 13.4439 0.48364C12.3044 0.330438 10.8454 0.332031 9.00065 0.332031V1.08203ZM15.758 2.2414L15.2276 2.77172C15.6447 3.18883 15.8954 3.76052 16.0291 4.75526C16.1657 5.77158 16.1673 7.11152 16.1673 8.9987H16.9173H17.6673C17.6673 7.15392 17.6689 5.6949 17.5157 4.5554C17.3596 3.39433 17.0306 2.45334 16.2883 1.71108L15.758 2.2414ZM16.9173 8.9987H16.1673C16.1673 10.8859 16.1657 12.2258 16.0291 13.2421C15.8954 14.2369 15.6447 14.8086 15.2276 15.2257L15.758 15.756L16.2883 16.2863C17.0306 15.5441 17.3596 14.6031 17.5157 13.442C17.6689 12.3025 17.6673 10.8435 17.6673 8.9987H16.9173ZM15.758 15.756L15.2277 15.2257C14.8105 15.6428 14.2388 15.8934 13.2441 16.0271C12.2277 16.1638 10.8878 16.1654 9.00065 16.1654V16.9154V17.6654C10.8454 17.6654 12.3044 17.667 13.4439 17.5138C14.605 17.3577 15.546 17.0286 16.2883 16.2864L15.758 15.756ZM9.00065 16.9154V16.1654C7.11347 16.1654 5.77354 16.1638 4.75722 16.0271C3.76247 15.8934 3.19078 15.6428 2.77367 15.2257L2.24335 15.756L1.71303 16.2864C2.4553 17.0286 3.39628 17.3577 4.55735 17.5138C5.69685 17.667 7.15588 17.6654 9.00065 17.6654V16.9154ZM2.24335 15.756L2.77369 15.2257C2.35658 14.8086 2.10596 14.2369 1.97222 13.2421C1.83558 12.2258 1.83398 10.8859 1.83398 8.9987H1.08398H0.333984C0.333984 10.8435 0.332392 12.3025 0.485593 13.442C0.641695 14.6031 0.970759 15.5441 1.71301 16.2864L2.24335 15.756ZM12.7507 8.9987H12.0007C12.0007 10.6556 10.6575 11.9987 9.00065 11.9987V12.7487V13.4987C11.4859 13.4987 13.5007 11.484 13.5007 8.9987H12.7507ZM9.00065 12.7487V11.9987C7.3438 11.9987 6.00065 10.6556 6.00065 8.9987H5.25065H4.50065C4.50065 11.484 6.51537 13.4987 9.00065 13.4987V12.7487ZM5.25065 8.9987H6.00065C6.00065 7.34184 7.3438 5.9987 9.00065 5.9987V5.2487V4.4987C6.51537 4.4987 4.50065 6.51342 4.50065 8.9987H5.25065ZM9.00065 5.2487V5.9987C10.6575 5.9987 12.0007 7.34184 12.0007 8.9987H12.7507H13.5007C13.5007 6.51342 11.4859 4.4987 9.00065 4.4987V5.2487ZM13.5905 4.41536V3.66536H13.583V4.41536V5.16536H13.5905V4.41536Z" fill="#079669"/></svg>',
            'readonly'         => 'no',
            'open_in_new_window' => 'yes',
            'nofollow'         => 'yes',
            'username_validation' => 'strict',
            'wpuf_cond'        => '',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Render the social media field
     *
     * @since 4.1.7
     *
     * @param array $field_settings
     * @param int $form_id
     * @param string $type
     * @param int $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        // Enqueue Instagram validation script
        wp_enqueue_script( 'wpuf-frontend-instagram-validation' );

        if ( isset( $post_id ) && $post_id != 0 ) {
            $value = $this->get_user_data( $post_id, $field_settings['name'] );
        } else {
            $value = $field_settings['default'];
        }

        $show_icon = isset( $field_settings['show_icon'] ) ? $field_settings['show_icon'] : 'no';
        $readonly = isset( $field_settings['readonly'] ) ? $field_settings['readonly'] : 'no';
        $css_class = isset( $field_settings['css'] ) ? $field_settings['css'] : '';
        $width_class = isset( $field_settings['width'] ) ? 'wpuf-' . $field_settings['width'] : 'wpuf-medium';
        ?>

        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <div class="wpuf-label-with-icon">
                <?php
                $this->print_label( $field_settings, $form_id );
                // Show icon next to label if enabled
                if ( $show_icon === 'yes' && ! empty( $this->icon_svg ) ) : ?>
                    <span class="wpuf-social-label-icon">
                        <?php
                        echo wp_kses( $this->icon_svg, $this->get_allowed_svg_attributes() );
                        ?>
                    </span>
                <?php endif; ?>
            </div>

            <div class="wpuf-fields">
                <div class="wpuf-social-input-wrapper <?php echo esc_attr( $width_class ); ?>">
                    <?php
                    $input_type = 'text';
                    ?>
                    <input
                        class="textfield <?php echo esc_attr( $input_type ); ?> <?php echo 'wpuf_' . $field_settings['name'] . '_' . $form_id; ?> <?php echo esc_attr( $css_class ); ?>"
                        id="<?php echo $field_settings['name'] . '_' . $form_id; ?>"
                        type="<?php echo esc_attr( $input_type ); ?>"
                        data-required="<?php echo $field_settings['required'] ?>"
                        data-type="<?php echo esc_attr( $input_type ); ?>"
                        data-username-validation="<?php echo isset( $field_settings['username_validation'] ) ? esc_attr( $field_settings['username_validation'] ) : 'strict'; ?>"
                        name="<?php echo esc_attr( $field_settings['name'] ); ?>"
                        placeholder="<?php echo esc_attr( $field_settings['placeholder'] ); ?>"
                        value="<?php echo esc_attr( $value ); ?>"
                        size="<?php echo esc_attr( isset( $field_settings['size'] ) ? $field_settings['size'] : 40 ); ?>"
                        <?php if ( isset( $this->username_pattern ) && ! empty( $this->username_pattern ) ): ?>
                        pattern="<?php echo esc_attr( $this->username_pattern ); ?>"
                        <?php endif; ?>
                        title="<?php echo esc_attr( sprintf( __( 'Enter a valid %s username or URL', 'wpuf-pro' ), $this->name ) ); ?>"
                        <?php echo $readonly === 'yes' ? 'readonly' : ''; ?>
                    />
                </div>

                <?php $this->help_text( $field_settings ); ?>
            </div>

            <?php if ( isset( $field_settings['word_restriction'] ) && $field_settings['word_restriction'] ) : ?>
                <div class="wpuf-wordlimit-message wpuf-help"></div>
            <?php endif; ?>
        </li>

        <?php $this->conditional_logic( $field_settings, $form_id ); ?>

        <?php
    }

    /**
     * Get options settings for the form builder
     *
     * @return array
     */
    public function get_options_settings() {
        // Get default options but exclude fields that would duplicate parent class settings
        $default_options = $this->get_default_option_settings( false, array('dynamic') );

        $settings = [
            [
                'name'      => 'meta_key',
                'title'     => __( 'Meta Key', 'wpuf-pro' ),
                'type'      => 'text',
                'default'   => 'wpuf_social_instagram',
                'section'   => 'basic',
                'priority'  => 15,
                'readonly'  => true,
                'help_text' => __( 'The meta key used to store this field data in the database', 'wpuf-pro' ),
            ],
            [
                'name'      => 'placeholder',
                'title'     => __( 'Placeholder', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'basic',
                'priority'  => 15.5,
                'help_text' => __( 'Placeholder text for the input field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'default',
                'title'     => __( 'Default value', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 15.7,
                'help_text' => __( 'Enter a default value for this field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'username_validation',
                'title'     => __( 'Username Validation', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'strict' => __( 'Strict (Instagram username format)', 'wpuf-pro' ),
                    'allow_full_url' => __( 'Allow full URLs', 'wpuf-pro' ),
                ],
                'default'   => 'strict',
                'section'   => 'basic',
                'priority'  => 16,
                'inline'    => true,
                'help_text' => __( 'Choose validation type: strict username format or allow full URLs', 'wpuf-pro' ),
            ],
            [
                'name'      => 'show_icon',
                'title'     => __( 'Icon show', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                    'no'  => __( 'No', 'wpuf-pro' ),
                ],
                'default'   => 'yes',
                'section'   => 'basic',
                'priority'  => 17,
                'inline'    => true,
                'help_text' => __( 'Show the social platform icon next to the field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'readonly',
                'title'     => __( 'Read Only', 'wpuf-pro' ),
                'type'      => 'checkbox',
                'is_single_opt' => true,
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                ],
                'section'   => 'basic',
                'priority'  => 18,
                'help_text' => __( 'Make this field read-only', 'wpuf-pro' ),
            ],
            [
                'name'      => 'size',
                'title'     => __( 'Size', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 20,
                'help_text' => __( 'Size of the input field', 'wpuf-pro' ),
            ],
        ];

        return array_merge( $default_options, $settings );
    }
}