<?php

namespace WeDevs\Wpuf\Pro\Fields;

/**
 * LinkedIn URL Field Class
 *
 * @since 4.2.0
 */
class Field_Linkedin extends Field_Social {

    const LINKEDIN_SVG_ICON = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 640"><!--!Font Awesome Free v7.0.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2025 Fonticons, Inc.--><path d="M512 96L127.9 96C110.3 96 96 110.5 96 128.3L96 511.7C96 529.5 110.3 544 127.9 544L512 544C529.6 544 544 529.5 544 511.7L544 128.3C544 110.5 529.6 96 512 96zM231.4 480L165 480L165 266.2L231.5 266.2L231.5 480L231.4 480zM198.2 160C219.5 160 236.7 177.2 236.7 198.5C236.7 219.8 219.5 237 198.2 237C176.9 237 159.7 219.8 159.7 198.5C159.7 177.2 176.9 160 198.2 160zM480.3 480L413.9 480L413.9 376C413.9 351.2 413.4 319.3 379.4 319.3C344.8 319.3 339.5 346.3 339.5 374.2L339.5 480L273.1 480L273.1 266.2L336.8 266.2L336.8 295.4L337.7 295.4C346.6 278.6 368.3 260.9 400.6 260.9C467.8 260.9 480.3 305.2 480.3 362.8L480.3 480z"/></svg>';

    public function __construct() {
        $this->name                 = __( 'LinkedIn', 'wpuf-pro' );
        $this->input_type           = 'linkedin_url';
        $this->icon                 = 'linkedin';
        $this->icon_svg             = self::LINKEDIN_SVG_ICON;
        $this->base_url             = 'https://linkedin.com/in/';
        $this->username_pattern     = '^[a-zA-Z0-9\-]{3,100}$';
        $this->url_pattern          = '/^(?:https?:\/\/)?(?:www\.)?(?:linkedin\.com)\/(?:in|company)\/([A-Za-z0-9\-]{3,100})\/?$/i';
        $this->max_username_length  = 100;
        $this->example_username     = 'john-doe';
    }

    /**
     * Get the field props
     *
     * @return array
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'       => 'linkedin_url',
            'required'         => 'no',
            'name'             => 'wpuf_social_linkedin',
            'meta_key'         => 'wpuf_social_linkedin',
            'help'             => __( 'Enter your LinkedIn username or full URL', 'wpuf-pro' ),
            'css'              => '',
            'placeholder'      => __( 'username', 'wpuf-pro' ),
            'default'          => '',
            'size'             => 40,
            'width'            => 'large',
            'id'               => 0,
            'is_new'           => true,
            'show_icon'        => 'yes',
            'icon'              => '<svg class="wpuf-linkedin-svg" width="20" height="25" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" aria-label="LinkedIn" role="img"><path d="M4.83398 7.33203V13.1654M8.16732 9.83203V13.1654M8.16732 9.83203C8.16732 8.45128 9.28657 7.33203 10.6673 7.33203C12.0481 7.33203 13.1673 8.45128 13.1673 9.83203V13.1654M8.16732 9.83203V7.33203M4.84066 4.83203H4.83317M1.08398 8.9987C1.08398 5.26675 1.08398 3.40077 2.24335 2.2414C3.40273 1.08203 5.2687 1.08203 9.00065 1.08203C12.7326 1.08203 14.5986 1.08203 15.758 2.2414C16.9173 3.40077 16.9173 5.26675 16.9173 8.9987C16.9173 12.7306 16.9173 14.5966 15.758 15.756C14.5986 16.9154 12.7326 16.9154 9.00065 16.9154C5.2687 16.9154 3.40273 16.9154 2.24335 15.756C1.08398 14.5966 1.08398 12.7306 1.08398 8.9987Z" stroke="#079669" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            'readonly'         => 'no',
            'open_in_new_window' => 'yes',
            'nofollow'         => 'yes',
            'username_validation' => 'strict',
            'wpuf_cond'        => '',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Render the social media field
     *
     * @since 4.1.7
     *
     * @param array $field_settings
     * @param int $form_id
     * @param string $type
     * @param int $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        // Enqueue LinkedIn validation script
        wp_enqueue_script( 'wpuf-frontend-linkedin-validation' );

        if ( isset( $post_id ) && $post_id != 0 ) {
            $value = $this->get_user_data( $post_id, $field_settings['name'] );
        } else {
            $value = $field_settings['default'];
        }

        $show_icon = isset( $field_settings['show_icon'] ) ? $field_settings['show_icon'] : 'no';
        $readonly = isset( $field_settings['readonly'] ) ? $field_settings['readonly'] : 'no';
        $css_class = isset( $field_settings['css'] ) ? $field_settings['css'] : '';
        $width_class = isset( $field_settings['width'] ) ? 'wpuf-' . $field_settings['width'] : 'wpuf-medium';
        ?>

        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <div class="wpuf-label-with-icon">
                <?php
                $this->print_label( $field_settings, $form_id );
                // Show icon next to label if enabled
                if ( $show_icon === 'yes' && ! empty( $this->icon_svg ) ) : ?>
                    <span class="wpuf-social-label-icon">
                        <?php
                        echo wp_kses( $this->icon_svg, $this->get_allowed_svg_attributes() );
                        ?>
                    </span>
                <?php endif; ?>
            </div>

            <div class="wpuf-fields">
                <div class="wpuf-social-input-wrapper <?php echo esc_attr( $width_class ); ?>">
                    <?php
                    $input_type = 'text';
                    ?>
                    <input
                        class="textfield <?php echo esc_attr( $input_type ); ?> <?php echo 'wpuf_' . $field_settings['name'] . '_' . $form_id; ?> <?php echo esc_attr( $css_class ); ?>"
                        id="<?php echo $field_settings['name'] . '_' . $form_id; ?>"
                        type="<?php echo esc_attr( $input_type ); ?>"
                        data-required="<?php echo $field_settings['required'] ?>"
                        data-type="<?php echo esc_attr( $input_type ); ?>"
                        data-username-validation="<?php echo isset( $field_settings['username_validation'] ) ? esc_attr( $field_settings['username_validation'] ) : 'strict'; ?>"
                        name="<?php echo esc_attr( $field_settings['name'] ); ?>"
                        placeholder="<?php echo esc_attr( $field_settings['placeholder'] ); ?>"
                        value="<?php echo esc_attr( $value ); ?>"
                        size="<?php echo esc_attr( isset( $field_settings['size'] ) ? $field_settings['size'] : 40 ); ?>"
                        <?php if ( isset( $this->username_pattern ) && ! empty( $this->username_pattern ) ): ?>
                        pattern="<?php echo esc_attr( $this->username_pattern ); ?>"
                        <?php endif; ?>
                        title="<?php echo esc_attr( sprintf( __( 'Enter a valid %s username or URL', 'wpuf-pro' ), $this->name ) ); ?>"
                        <?php echo $readonly === 'yes' ? 'readonly' : ''; ?>
                    />
                </div>

                <?php $this->help_text( $field_settings ); ?>
            </div>

            <?php if ( isset( $field_settings['word_restriction'] ) && $field_settings['word_restriction'] ) : ?>
                <div class="wpuf-wordlimit-message wpuf-help"></div>
            <?php endif; ?>
        </li>

        <?php $this->conditional_logic( $field_settings, $form_id ); ?>

        <?php
    }

    /**
     * Get options settings for the form builder
     *
     * @return array
     */
    public function get_options_settings() {
        // Get default options but exclude fields that would duplicate parent class settings
        $default_options = $this->get_default_option_settings( false, array('dynamic') );

        $settings = [
            [
                'name'      => 'meta_key',
                'title'     => __( 'Meta Key', 'wpuf-pro' ),
                'type'      => 'text',
                'default'   => 'wpuf_social_linkedin',
                'section'   => 'basic',
                'priority'  => 15,
                'readonly'  => true,
                'help_text' => __( 'The meta key used to store this field data in the database', 'wpuf-pro' ),
            ],
            [
                'name'      => 'placeholder',
                'title'     => __( 'Placeholder', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'basic',
                'priority'  => 15.5,
                'help_text' => __( 'Placeholder text for the input field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'default',
                'title'     => __( 'Default value', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 15.7,
                'help_text' => __( 'Enter a default value for this field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'username_validation',
                'title'     => __( 'Username Validation', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'strict' => __( 'Strict (LinkedIn username format)', 'wpuf-pro' ),
                    'allow_full_url' => __( 'Allow full URLs', 'wpuf-pro' ),
                ],
                'default'   => 'strict',
                'section'   => 'basic',
                'priority'  => 16,
                'inline'    => true,
                'help_text' => __( 'Choose validation type: strict username format or allow full URLs', 'wpuf-pro' ),
            ],
            [
                'name'      => 'show_icon',
                'title'     => __( 'Icon show', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                    'no'  => __( 'No', 'wpuf-pro' ),
                ],
                'default'   => 'yes',
                'section'   => 'basic',
                'priority'  => 17,
                'inline'    => true,
                'help_text' => __( 'Show the social platform icon next to the field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'readonly',
                'title'     => __( 'Read Only', 'wpuf-pro' ),
                'type'      => 'checkbox',
                'is_single_opt' => true,
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                ],
                'section'   => 'basic',
                'priority'  => 18,
                'help_text' => __( 'Make this field read-only', 'wpuf-pro' ),
            ],
            [
                'name'      => 'size',
                'title'     => __( 'Size', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 20,
                'help_text' => __( 'Size of the input field', 'wpuf-pro' ),
            ],
        ];

        return array_merge( $default_options, $settings );
    }
}