<?php

namespace WeDevs\Wpuf\Pro\Fields;

/**
 * Twitter URL Field Class
 *
 * @since 4.2.0
 */
class Field_Twitter extends Field_Social {

    const TWITTER_SVG_ICON = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 640"><!--!Font Awesome Free v7.0.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2025 Fonticons, Inc.--><path d="M160 96C124.7 96 96 124.7 96 160L96 480C96 515.3 124.7 544 160 544L480 544C515.3 544 544 515.3 544 480L544 160C544 124.7 515.3 96 480 96L160 96zM457.1 180L353.3 298.6L475.4 460L379.8 460L305 362.1L219.3 460L171.8 460L282.8 333.1L165.7 180L263.7 180L331.4 269.5L409.6 180L457.1 180zM419.3 431.6L249.4 206.9L221.1 206.9L392.9 431.6L419.3 431.6z"/></svg>';

    public function __construct() {
        $this->name                 = __( 'X (Twitter)', 'wpuf-pro' );
        $this->input_type           = 'twitter_url';
        $this->icon                 = 'twitter';
        $this->icon_svg             = self::TWITTER_SVG_ICON;
        $this->base_url             = 'https://x.com/';
        $this->username_pattern     = '^[a-zA-Z0-9_]{1,15}$';
        $this->url_pattern          = '/^(?:https?:\/\/)?(?:www\.)?(?:twitter\.com|x\.com)\/([A-Za-z0-9_]{1,15})\/?$/i';
        $this->max_username_length  = 15;
        $this->example_username     = 'username';
    }

    /**
     * Get the field props
     *
     * @return array
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'       => 'twitter_url',
            'required'         => 'no',
            'name'             => 'wpuf_social_twitter',
            'meta_key'         => 'wpuf_social_twitter',
            'help'             => __( 'Enter your X (Twitter) username (without @) or full URL', 'wpuf-pro' ),
            'css'              => '',
            'placeholder'      => __( 'username', 'wpuf-pro' ),
            'default'          => '',
            'size'             => 40,
            'width'            => 'large',
            'id'               => 0,
            'is_new'           => true,
            'show_icon'        => 'yes',
            'icon'              => '<svg class="wpuf-twitter-svg" width="20" height="25" viewBox="0 0 22 22" fill="none" xmlns="http://www.w3.org/2000/svg" aria-label="X (Twitter)" role="img"><path d="M6 16L10.1936 11.8065M10.1936 11.8065L6 6H8.77778L11.8065 10.1935M10.1936 11.8065L13.2222 16H16L11.8065 10.1935M16 6L11.8065 10.1935M1.5 11C1.5 6.52166 1.5 4.28249 2.89124 2.89124C4.28249 1.5 6.52166 1.5 11 1.5C15.4784 1.5 17.7175 1.5 19.1088 2.89124C20.5 4.28249 20.5 6.52166 20.5 11C20.5 15.4783 20.5 17.7175 19.1088 19.1088C17.7175 20.5 15.4784 20.5 11 20.5C6.52166 20.5 4.28249 20.5 2.89124 19.1088C1.5 17.7175 1.5 15.4783 1.5 11Z" stroke="#079669" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            'readonly'         => 'no',
            'open_in_new_window' => 'yes',
            'nofollow'         => 'yes',
            'username_validation' => 'strict',
            'wpuf_cond'        => '',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Render the social media field
     *
     * @since 4.1.7
     *
     * @param array $field_settings
     * @param int $form_id
     * @param string $type
     * @param int $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        // Enqueue Twitter validation script
        wp_enqueue_script( 'wpuf-frontend-twitter-validation' );

        if ( isset( $post_id ) && $post_id != 0 ) {
            $value = $this->get_user_data( $post_id, $field_settings['name'] );
        } else {
            $value = $field_settings['default'];
        }

        $show_icon = isset( $field_settings['show_icon'] ) ? $field_settings['show_icon'] : 'no';
        $readonly = isset( $field_settings['readonly'] ) ? $field_settings['readonly'] : 'no';
        $css_class = isset( $field_settings['css'] ) ? $field_settings['css'] : '';
        $width_class = isset( $field_settings['width'] ) ? 'wpuf-' . $field_settings['width'] : 'wpuf-medium';
        ?>

        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <div class="wpuf-label-with-icon">
                <?php
                $this->print_label( $field_settings, $form_id );
                // Show icon next to label if enabled
                if ( $show_icon === 'yes' && ! empty( $this->icon_svg ) ) : ?>
                    <span class="wpuf-social-label-icon">
                        <?php
                        echo wp_kses( $this->icon_svg, $this->get_allowed_svg_attributes() );
                        ?>
                    </span>
                <?php endif; ?>
            </div>

            <div class="wpuf-fields">
                <div class="wpuf-social-input-wrapper <?php echo esc_attr( $width_class ); ?>">
                    <?php
                    $input_type = 'text';
                    ?>

                    <span class="wpuf-at-prefix">@</span>
                    <input
                        class="textfield <?php echo esc_attr( $input_type ); ?> wpuf-twitter-with-prefix <?php echo 'wpuf_' . $field_settings['name'] . '_' . $form_id; ?> <?php echo esc_attr( $css_class ); ?>"
                        id="<?php echo $field_settings['name'] . '_' . $form_id; ?>"
                        type="<?php echo esc_attr( $input_type ); ?>"
                        data-required="<?php echo $field_settings['required'] ?>"
                        data-type="<?php echo esc_attr( $input_type ); ?>"
                        data-username-validation="<?php echo isset( $field_settings['username_validation'] ) ? esc_attr( $field_settings['username_validation'] ) : 'strict'; ?>"
                        data-show-at-prefix="yes"
                        name="<?php echo esc_attr( $field_settings['name'] ); ?>"
                        placeholder="<?php echo esc_attr( $field_settings['placeholder'] ); ?>"
                        value="<?php echo esc_attr( $value ); ?>"
                        size="<?php echo esc_attr( isset( $field_settings['size'] ) ? $field_settings['size'] : 40 ); ?>"
                        pattern="<?php echo esc_attr( $this->username_pattern ); ?>"
                        title="<?php echo esc_attr( sprintf( __( 'Enter a valid %s username (without @)', 'wpuf-pro' ), $this->name ) ); ?>"
                        <?php echo $readonly === 'yes' ? 'readonly' : ''; ?>
                    />

                </div>

                <?php $this->help_text( $field_settings ); ?>
            </div>

            <?php if ( isset( $field_settings['word_restriction'] ) && $field_settings['word_restriction'] ) : ?>
                <div class="wpuf-wordlimit-message wpuf-help"></div>
            <?php endif; ?>
        </li>

        <?php $this->conditional_logic( $field_settings, $form_id ); ?>

        <?php
    }

    /**
     * Get options settings for the form builder
     *
     * @return array
     */
    public function get_options_settings() {
        // Get default options but exclude fields that would duplicate parent class settings
        $default_options = $this->get_default_option_settings( false, array('dynamic') );

        $settings = [
            [
                'name'      => 'meta_key',
                'title'     => __( 'Meta Key', 'wpuf-pro' ),
                'type'      => 'text',
                'default'   => 'wpuf_social_twitter',
                'section'   => 'basic',
                'priority'  => 15,
                'readonly'  => true,
                'help_text' => __( 'The meta key used to store this field data in the database', 'wpuf-pro' ),
            ],
            [
                'name'      => 'placeholder',
                'title'     => __( 'Placeholder', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'basic',
                'priority'  => 15.5,
                'help_text' => __( 'Placeholder text for the input field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'default',
                'title'     => __( 'Default value', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 15.7,
                'help_text' => __( 'Enter a default value for this field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'username_validation',
                'title'     => __( 'Username Validation', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'strict' => __( 'Strict (X (Twitter) username format)', 'wpuf-pro' ),
                    'allow_full_url' => __( 'Allow full URLs', 'wpuf-pro' ),
                ],
                'default'   => 'strict',
                'section'   => 'basic',
                'priority'  => 16,
                'inline'    => true,
                'help_text' => __( 'Choose validation type: strict username format or allow full URLs', 'wpuf-pro' ),
            ],
            [
                'name'      => 'show_icon',
                'title'     => __( 'Icon show', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                    'no'  => __( 'No', 'wpuf-pro' ),
                ],
                'default'   => 'yes',
                'section'   => 'basic',
                'priority'  => 17,
                'inline'    => true,
                'help_text' => __( 'Show the social platform icon next to the field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'readonly',
                'title'     => __( 'Read Only', 'wpuf-pro' ),
                'type'      => 'checkbox',
                'is_single_opt' => true,
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                ],
                'section'   => 'basic',
                'priority'  => 18,
                'help_text' => __( 'Make this field read-only', 'wpuf-pro' ),
            ],
            [
                'name'      => 'size',
                'title'     => __( 'Size', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 20,
                'help_text' => __( 'Size of the input field', 'wpuf-pro' ),
            ],
        ];

        return array_merge( $default_options, $settings );
    }
}
