<?php
/*
Plugin Name: Mailchimp
Plugin URI: https://wedevs.com/docs/wp-user-frontend-pro/modules/add-users-to-mailchimp-subscribers-list-upon-registration-from-frontend/
Thumbnail Name: wpuf-mailchimp.png
Description: Add subscribers to Mailchimp mailing list when they registers via WP User Frontend Pro
Version: 0.2.1
Author: weDevs
Author URI: http://wedevs.com/
License: GPL2
*/

/**
 * Copyright (c) 2014 weDevs (email: info@wedevs.com). All rights reserved.
 *
 * Released under the GPL license
 * http://www.opensource.org/licenses/gpl-license.php
 *
 * This is an add-on for WordPress
 * http://wordpress.org/
 *
 * **********************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * **********************************************************************
 */

// don't call the file directly
if ( !defined( 'ABSPATH' ) ) exit;

// if ( is_admin() ) {
//    require_once dirname( __FILE__ ) . '/lib/wedevs-updater.php';
//    new WeDevs_Plugin_Update_Checker( plugin_basename( __FILE__ ), 'wpuf-mailchimp' );
// }

/**
 * WPUF_Mailchimp class
 *
 * @class WPUF_Mailchimp The class that holds the entire WPUF_Mailchimp plugin
 */
class WPUF_Mailchimp {

    /**
     * Constructor for the WPUF_Mailchimp class
     *
     * Sets up all the appropriate hooks and actions
     * within our plugin.
     *
     * @uses is_admin()
     * @uses add_action()
     */
    public function __construct() {
        $this->define_constants();
        add_action( 'wpuf_admin_menu', array( $this, 'add_mailchimp_menu' ) );

        add_action( 'wpuf_after_register', array( $this, 'subscribe_user_after_registration' ), 10, 3 );

        // Load form related scripts and styles
        add_action( 'wpuf_form_builder_enqueue_after_components', [ $this, 'enqueue_scripts' ], 11 );

        // registration form setting items
        add_filter( 'wpuf_registration_form_module_settings_menu_sub_items', [ $this, 'add_settings_sub_menu' ] );
        add_filter( 'wpuf_registration_form_builder_setting_menu_contents', [ $this, 'add_settings_fields' ] );
        add_action( 'wpuf_after_registration_form_settings_field_enable_double_optin', [ $this, 'add_conditional_field' ], 10, 3 );
    }

    /**
     * Define constants
     *
     * @return void
     */
    public function define_constants() {
        define( 'WPUF_MCHIMP_FILE', __FILE__ );
        define( 'WPUF_MCHIMP_ROOT', dirname( __FILE__ ) );
        define( 'WPUF_MCHIMP_INCLUDES', WPUF_MCHIMP_ROOT . '/includes' );
        define( 'WPUF_MCHIMP_VIEWS', WPUF_MCHIMP_ROOT . '/includes/views' );
        define( 'WPUF_MCHIMP_ROOT_URI', plugins_url( '', __FILE__ ) );
        define( 'WPUF_MCHIMP_ASSET_URI', WPUF_MCHIMP_ROOT_URI . '/assets' );
        define( 'WPUF_MCHIMP_TEMPLATES', WPUF_MCHIMP_ROOT . '/templates' );
    }

    /**
     * Add conditional fields using Vue component
     *
     * @since 4.1.0
     *
     * @param $field
     * @param $value
     * @param $form_settings
     *
     * @return void
     */
    public function add_conditional_field( $field, $value, $form_settings ) {
        $conditional_logic_settings = isset( $form_settings['integrations']['mailchimp']['wpuf_cond'] )
            ? $form_settings['integrations']['mailchimp']['wpuf_cond']
            : [
                'condition_status' => 'no',
                'cond_logic' => 'all',
                'conditions' => []
            ];
        ?>
        <integration-conditional-logic
            integration-name="mailchimp"
            settings-path="integrations.mailchimp.wpuf_cond"
            :current-settings="<?php echo esc_attr( wp_json_encode( $conditional_logic_settings ) ); ?>"
            label="<?php esc_attr_e( 'Conditional Logic', 'wpuf-pro' ); ?>">
        </integration-conditional-logic>
        <?php
    }

    /**
     * Add fields on registration form settings
     *
     * @since 4.1.0
     *
     * @param $fields
     *
     * @return array
     */
    public function add_settings_fields( $fields ) {
        $lists = get_option( 'wpuf_mc_lists');

        $fields['modules']['mailchimp']['enable_mailchimp'] = [
            'label' => __( 'Enable Mailchimp', 'wpuf-pro' ),
            'type'  => 'toggle',
            'help_text' => __( 'Activate to connect your registration form with Mailchimp for seamless email list management', 'wpuf-pro' ),
        ];

        if ( $lists ) {
            $options = [];
            foreach ( $lists as $list ) {
                $options[$list['id']] = $list['name'];
            }

            $fields['modules']['mailchimp']['mailchimp_list'] = [
                'label'   => __( 'Select Preferred List', 'wpuf-pro' ),
                'type'    => 'select',
                'options' => $options,
                'help_text'   => __( 'Select the Mailchimp list where new sign-ups will be added as subscribers for email campaigns', 'wpuf-pro' ),
            ];
        } else {
            $fields['modules']['mailchimp']['mailchimp_list'] = [
                'type'  => 'note',
                'note'  => __( 'You are not connected with mailchimp. Insert your API key first', 'wpuf-pro' ),
            ];
        }

        $fields['modules']['mailchimp']['enable_double_optin'] = [
            'label' => __( 'Double Optin', 'wpuf-pro' ),
            'type'  => 'toggle',
            'help_text' => __( 'Require new subscribers to confirm their email via an additional opt-in email', 'wpuf-pro' ),
        ];

        return $fields;
    }

    /**
     * Add sub-menu settings heading on registration form
     *
     * @since 4.1.0
     *
     * @param $settings
     *
     * @return mixed
     */
    public function add_settings_sub_menu( $settings ) {
        $settings['mailchimp'] = [
            'label' => __( 'Mailchimp', 'wpuf-pro' ),
            'icon'  => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M16.4837 10.9759C16.613 10.9605 16.7453 10.9605 16.874 10.9759C16.9449 10.815 16.9573 10.5336 16.8923 10.2308C16.8002 9.77958 16.6739 9.5041 16.4093 9.54744C16.1478 9.59078 16.1389 9.91851 16.231 10.3697C16.2865 10.6232 16.3786 10.8429 16.4831 10.9759H16.4837ZM14.2323 11.3345C14.42 11.4182 14.5334 11.4734 14.58 11.4242C14.6078 11.3933 14.6013 11.3375 14.5552 11.2633C14.4631 11.1119 14.2724 10.9569 14.0693 10.8708C13.6447 10.6915 13.1558 10.7503 12.7838 11.0252C12.6575 11.1178 12.5411 11.2478 12.5565 11.325C12.5624 11.3499 12.5813 11.3683 12.6244 11.3743C12.7289 11.3868 13.0885 11.2045 13.5072 11.1766C13.7994 11.1582 14.0451 11.2508 14.2328 11.3345H14.2323ZM13.8544 11.5512C13.6081 11.5916 13.4759 11.6717 13.3902 11.7459C13.3135 11.8107 13.2704 11.8849 13.2704 11.9347L13.2887 11.9811L13.3288 11.9965C13.3843 11.9965 13.5072 11.9437 13.5072 11.9437C13.7593 11.851 14.0297 11.8202 14.2943 11.8635C14.4171 11.876 14.4755 11.8849 14.5033 11.8421C14.5092 11.8297 14.5186 11.8047 14.4938 11.7649C14.4383 11.6694 14.1892 11.5108 13.8544 11.5512ZM15.7244 12.3486C15.8903 12.4287 16.0722 12.3978 16.1306 12.2773C16.192 12.1538 16.1058 11.9864 15.9399 11.9062C15.774 11.8225 15.5921 11.857 15.5307 11.9775C15.4723 12.101 15.5585 12.2648 15.7244 12.3486ZM16.7914 11.4087C16.6561 11.4057 16.5422 11.5542 16.5422 11.7424C16.5392 11.9282 16.6437 12.0796 16.779 12.0826C16.9142 12.0855 17.0252 11.9371 17.0281 11.7519C17.0311 11.5661 16.9236 11.4117 16.7914 11.4087ZM7.73411 14.7667C7.70045 14.7264 7.64495 14.7388 7.5924 14.7513C7.54929 14.7608 7.50619 14.7667 7.46603 14.7697C7.37392 14.7697 7.28771 14.7264 7.23516 14.6522C7.17375 14.5566 7.1767 14.4141 7.24461 14.2532L7.27531 14.179C7.38632 13.935 7.56464 13.5265 7.36152 13.137C7.21095 12.8431 6.96177 12.6609 6.66004 12.6234C6.37071 12.586 6.08492 12.6882 5.88475 12.9019C5.58656 13.2326 5.5405 13.6844 5.59896 13.8453C5.62022 13.9041 5.65446 13.9195 5.67867 13.9225C5.72768 13.9284 5.80149 13.8916 5.85109 13.7646L5.86349 13.7242C5.88475 13.6559 5.9249 13.5294 5.98631 13.4303C6.15223 13.1768 6.48762 13.1055 6.7368 13.2724C6.94583 13.4083 7.02908 13.6684 6.93992 13.9124C6.89387 14.0424 6.8171 14.2835 6.83541 14.4847C6.86907 14.8896 7.11529 15.0505 7.33672 15.0689C7.55224 15.0784 7.70281 14.9544 7.74297 14.8647C7.76422 14.8119 7.74592 14.781 7.73352 14.7656L7.73411 14.7667ZM20.1925 14.0276C20.183 13.9967 20.1311 13.799 20.0573 13.5639L19.9067 13.159C20.2019 12.7137 20.2078 12.3147 20.1683 12.0891C20.1252 11.8107 20.0112 11.5726 19.778 11.3286C19.5471 11.081 19.0706 10.8275 18.4004 10.6393L18.0497 10.5437C18.0497 10.5282 18.0314 9.70893 18.016 9.35923C18.0066 9.10572 17.9853 8.70674 17.8625 8.31727C17.7178 7.78827 17.4627 7.32459 17.1456 7.02773C18.019 6.11579 18.5663 5.11064 18.5634 4.25095C18.5604 2.59034 16.5398 2.08925 14.0486 3.12824L13.5196 3.35385C13.1966 3.03562 12.8765 2.7168 12.5506 2.39857C9.70865 -0.0967908 0.827353 9.84489 3.66575 12.2631L4.28693 12.7921C4.12101 13.2433 4.06255 13.7289 4.11451 14.2116C4.18242 14.8766 4.52076 15.5136 5.07403 16.0052C5.5966 16.4719 6.28568 16.7687 6.95646 16.7687C8.06064 19.3324 10.5855 20.9063 13.5438 20.9959C16.7205 21.0886 19.3871 19.5889 20.5031 16.8928C20.5769 16.704 20.8875 15.8509 20.8875 15.0992C20.8875 14.3416 20.4629 14.0294 20.1925 14.0294V14.0276ZM7.19914 16.0438C7.10407 16.0622 7.00251 16.0687 6.9039 16.0652C5.94734 16.0402 4.91106 15.1717 4.80655 14.1386C4.69259 12.9975 5.27066 12.1224 6.29513 11.9152C6.43684 11.8843 6.58092 11.8748 6.72558 11.8813C7.3007 11.9152 8.14626 12.3575 8.33698 13.6161C8.5094 14.7323 8.23837 15.8675 7.19914 16.0468V16.0438ZM6.1292 11.2324C5.49268 11.3559 4.92996 11.721 4.58512 12.2221C4.37904 12.0523 3.9976 11.718 3.93029 11.588C3.38292 10.5366 4.53021 8.49894 5.3297 7.34537C7.31015 4.50031 10.4131 2.34514 11.8497 2.73461C12.0835 2.80289 12.8553 3.70533 12.8553 3.70533C12.8553 3.70533 11.4192 4.50624 10.0871 5.62539C8.29447 7.01704 6.9411 9.03625 6.1292 11.2318V11.2324ZM16.1944 15.6205C16.2157 15.611 16.2316 15.5867 16.2281 15.5647C16.2251 15.5338 16.2003 15.5118 16.1696 15.5154C16.1696 15.5154 14.6686 15.741 13.2479 15.2156C13.4015 14.7115 13.8136 14.8938 14.4348 14.9437C15.4002 15.0024 16.3692 14.8974 17.3009 14.6314C17.944 14.4455 18.7866 14.081 19.4444 13.5585C19.6658 14.0472 19.7425 14.5851 19.7425 14.5851C19.7425 14.5851 19.915 14.5542 20.0561 14.6438C20.1913 14.7276 20.2899 14.9039 20.222 15.3521C20.0868 16.1934 19.7301 16.8768 19.1332 17.5073C18.7612 17.9122 18.3183 18.2465 17.8229 18.4905C17.5489 18.636 17.2632 18.7565 16.9709 18.8556C14.7194 19.5948 12.416 18.7814 11.672 17.0341C11.6135 16.9011 11.5645 16.7622 11.5244 16.6197C11.2079 15.4691 11.4753 14.0869 12.3179 13.2183C12.367 13.1625 12.4195 13.0978 12.4195 13.0141C12.4195 12.9458 12.3764 12.8752 12.3398 12.8223C12.0475 12.3925 11.0295 11.6628 11.2327 10.2462C11.3803 9.22861 12.263 8.51438 13.0873 8.55475L13.2964 8.56722C13.653 8.58859 13.9636 8.6355 14.2559 8.64737C14.7477 8.66875 15.1906 8.59809 15.7138 8.15875C15.8891 8.01032 16.0302 7.8803 16.2706 7.84052C16.2954 7.83755 16.3597 7.81262 16.4825 7.81915C16.6119 7.82508 16.7347 7.86546 16.8421 7.9367C17.2637 8.21812 17.3222 8.89851 17.3435 9.39604C17.3559 9.68043 17.3895 10.3668 17.4019 10.5651C17.4297 11.0163 17.5466 11.0816 17.7834 11.1588C17.9186 11.2051 18.0449 11.2389 18.2262 11.2888C18.783 11.4467 19.1119 11.607 19.318 11.8113C19.4444 11.9383 19.4993 12.0743 19.5182 12.2037C19.5861 12.6864 19.1491 13.2801 17.993 13.821C16.7288 14.4117 15.1977 14.5601 14.1396 14.4426L13.7676 14.3992C12.922 14.2846 12.439 15.3854 12.9462 16.1436C13.2722 16.6292 14.1673 16.9474 15.0589 16.9474C17.1043 16.9474 18.6785 16.0664 19.2631 15.3052L19.3091 15.237C19.3398 15.1936 19.315 15.1722 19.2784 15.1966C18.8019 15.5243 16.6768 16.8323 14.407 16.4398C14.407 16.4398 14.1301 16.3935 13.878 16.2944C13.6778 16.2172 13.2598 16.0224 13.2078 15.589C15.0406 16.1614 16.1938 15.6199 16.1938 15.6199L16.1944 15.6205ZM9.78541 7.3329C10.4898 6.51358 11.3567 5.80231 12.1319 5.40334C12.1597 5.3879 12.1875 5.41878 12.1751 5.44371C12.1136 5.55533 11.9938 5.79934 11.9566 5.97865C11.9507 6.00655 11.9814 6.02792 12.0026 6.01249C12.4886 5.68179 13.3282 5.32616 14.0663 5.27985C14.097 5.27985 14.1124 5.32022 14.0876 5.33863C13.9648 5.43421 13.8567 5.54286 13.7617 5.66635C13.7463 5.68773 13.7617 5.71563 13.7865 5.71563C14.3061 5.72157 15.0353 5.90443 15.5089 6.17338C15.5425 6.19179 15.5213 6.25353 15.4841 6.24463C14.7643 6.07779 13.5863 5.95074 12.3628 6.25413C11.271 6.52308 10.4379 6.93452 9.82851 7.38277C9.79781 7.40414 9.76061 7.36377 9.78541 7.3329Z" fill="#9CA3AF"/>
</svg>',
        ];

        return $settings;
    }

    /**
     * Initializes the WPUF_Mailchimp() class
     *
     * Checks for an existing WPUF_Mailchimp() instance
     * and if it doesn't find one, creates it.
     */
    public static function init() {
        static $instance = false;

        if ( ! $instance ) {
            $instance = new WPUF_Mailchimp();
        }

        return $instance;
    }

    /**
     * Enqueue admin scripts
     *
     * Allows plugin assets to be loaded.
     *
     * @uses wp_enqueue_script()
     * @uses wp_enqueue_style
     */
    public function enqueue_scripts() {
        // Ensure form builder scripts are loaded first
        $dependencies = [ 'wpuf-form-builder' ];

        /**
         * All styles goes here
         */
        wp_enqueue_style( 'wpuf-mc-styles', WPUF_MCHIMP_ASSET_URI . '/css/style.css', $dependencies, WPUF_PRO_VERSION );

        /**
         * The integration conditional logic component is now included in the
         * main form builder components file (wpuf-form-builder-components-pro.js)
         * which is already enqueued by the form builder system.
         * No additional script enqueuing needed.
         */
    }



    /**
     * Add Mailchimp Submenu in WPUF
     */
    public function add_mailchimp_menu() {
        add_submenu_page( 'wp-user-frontend', __( 'Mailchimp', 'wpuf-pro' ), __( 'Mailchimp', 'wpuf-pro' ), 'manage_options', 'wpuf_mailchimp', array($this, 'mailchimp_page') );
    }

    /**
     * Submenu Call Back Page
     */
    public function mailchimp_page() {
        require_once dirname( __FILE__ ) . '/templates/mailchimp-template.php';
    }

    /**
     * Send Subscribe request in Mailchimp
     * @param  integer $user_id
     * @param  integer $form_id
     * @param  array $form_settings
     */
    public function subscribe_user_after_registration( $user_id, $form_id, $form_settings ) {
        if ( ! isset( $form_settings['enable_mailchimp'] ) || $form_settings['enable_mailchimp'] == 'no' ) {
            return;
        }

        $wpuf_cond          = isset( $form_settings['integrations']['mailchimp']['wpuf_cond'] ) ? $form_settings['integrations']['mailchimp']['wpuf_cond'] : '';
        $conditional_logic  = isset( $wpuf_cond['condition_status'] ) ? $wpuf_cond['condition_status'] : 'no';
        $condition_logic    = isset( $wpuf_cond['cond_logic'] ) ? $wpuf_cond['cond_logic'] : 'all';
        $conditions         = isset( $wpuf_cond['conditions'] ) && is_array( $wpuf_cond['conditions'] ) ? $wpuf_cond['conditions'] : [];

        if ( $conditional_logic !== 'yes' || empty( $conditions ) ) {
            return;
        }

        $condition_results = [];

        foreach ( $conditions as $condition ) {
            $condition_name     = isset( $condition['name'] ) ? $condition['name'] : '';
            $condition_operator = isset( $condition['operator'] ) ? $condition['operator'] : '=';
            $condition_option   = isset( $condition['option'] ) ? $condition['option'] : '';
            $cond_field_value   = isset( $_POST[$condition_name] ) ? $_POST[$condition_name] : '';

            if ( empty( $condition_name ) ) {
                continue;
            }

            $result = false;
            $value = $cond_field_value;

            if ( is_array( $cond_field_value ) ) {
                if ( in_array( $condition_option, $cond_field_value ) ) {
                    $value = $condition_option;
                }
            }

            switch ( $condition_operator ) {
                case '=':
                    $result = ( $condition_option == $value );
                    break;
                case '!=':
                    $result = ( $condition_option != $value );
                    break;
                case '!=empty':
                    $result = !empty( $value );
                    break;
                case '==empty':
                    $result = empty( $value );
                    break;
                case '==contains':
                    $result = ( strpos( $value, $condition_option ) !== false );
                    break;
                default:
                    $result = ( $condition_option == $value );
                    break;
            }

            $condition_results[] = $result;
        }

        // Apply logic rule (all/any)
        if ( 'all' === $condition_logic ) {
            // All conditions must be true
            if ( in_array( false, $condition_results, true ) ) {
                return;
            }
        } else {
            // Any condition must be true
            if ( ! in_array( true, $condition_results, true ) ) {
                return;
            }
        }

        require_once dirname( __FILE__ ) . '/classes/mailchimp.php';
        $user = get_user_by( 'id', $user_id );

        $MailChimp = new MailChimp( get_option( 'wpuf_mailchimp_api_key' ) );

        $result = $MailChimp->call( 'lists/' . $form_settings['mailchimp_list'], array(
            'email_address' => $user->user_email,
            'status'        => 'subscribed',
            'merge_fields'  => array(
                'FNAME'         => $user->first_name,
                'LNAME'         =>$user->last_name,
            ),
            'double_optin'  => ( isset( $form_settings['enable_double_optin'] ) && $form_settings['enable_double_optin'] == 'yes' ) ? true : false
        ) );
    }

} // WPUF_Mailchimp

$wpuf_mailchimp = WPUF_Mailchimp::init();
