# WPUF User Directory Shortcode - Complete Flow Documentation

## Overview
This document provides a comprehensive guide to how the WPUF User Directory shortcode system works, from initial registration to final rendering.

## Table of Contents
1. [System Architecture](#system-architecture)
2. [Shortcode Registration](#shortcode-registration)
3. [Initial Processing Flow](#initial-processing-flow)
4. [Template System](#template-system)
5. [Search & Filtering](#search--filtering)
6. [Pagination System](#pagination-system)
7. [Profile Rendering](#profile-rendering)
8. [Extension Points](#extension-points)
9. [File Structure](#file-structure)
10. [Usage Examples](#usage-examples)

---

## System Architecture

### Core Components
- **Main Plugin File**: `userlisting.php` - Entry point and class container
- **ShortCode Class**: `includes/ShortCode.php` - Handles shortcode processing
- **Template System**: `templates/` - Contains all rendering templates
- **Admin Builder**: `includes/Admin/Builder.php` - Form builder for directory configuration

### Data Flow
```
WordPress Request → Shortcode Detection → ShortCode::user_lists() → Template Loading → HTML Output
```

---

## Shortcode Registration

### 1. Plugin Initialization
**File**: `userlisting.php` (Lines 100-125)
```php
public function includes_files() {
    if ( is_admin() ) {
        require_once WPUF_UD_INCLUDES . '/Admin.php';
        require_once WPUF_UD_INCLUDES . '/Admin/Builder.php';
        require_once WPUF_UD_INCLUDES . '/Admin/Form.php';
    }

    require_once WPUF_UD_INCLUDES . '/Frontend.php';
    require_once WPUF_UD_INCLUDES . '/ShortCode.php';
    require_once WPUF_UD_INCLUDES . '/Assets.php';
}

public function init_classes() {
    if ( is_admin() ) {
        $this->container['admin'] = new WPUF\UserDirectory\Admin();
    }

    $this->container['frontend']  = new WPUF\UserDirectory\Frontend();
    $this->container['shortcode'] = new WPUF\UserDirectory\ShortCode();
    $this->container['assets']    = new WPUF\UserDirectory\Assets();
}
```

### 2. Shortcode Registration
**File**: `includes/ShortCode.php` (Lines 48-51)
```php
public function init_short_codes() {
    add_shortcode( 'wpuf_user_listing', [ $this, 'user_lists' ] );
}
```

### 3. WordPress Integration
**File**: `includes/ShortCode.php` (Lines 635-642)
```php
public function wpuf_add_user_listing_shortcode( $array ) {
    $array['user-listing'] = array(
        'title'   => __( 'User Listing', 'wpuf-pro' ),
        'content' => '[wpuf_user_listing]',
    );
    return $array;
}
```

---

## Initial Processing Flow

### 1. Shortcode Attributes Processing
**File**: `includes/ShortCode.php` (Lines 54-66)
```php
public function user_lists( $atts ) {
    $atts = shortcode_atts(
        [
            'role'           => 'all',
            'per_page'       => '6',
            'roles_exclude'  => '',
            'roles_include'  => '',
            'status_exclude' => '',
        ], $atts
    );
```

### 2. Profile Detection Logic
**File**: `includes/ShortCode.php` (Lines 78-88)
```php
$profile_base_value = ! empty( $_GET[ $profile_permalink_base ] ) ? wp_unslash( $_GET[ $profile_permalink_base ] ) : '';

// we are viewing a single user
if ( 'username' === $profile_permalink_base ) {
    $single_user = get_user_by( 'login', $profile_base_value );
} else {
    $single_user = get_user_by( 'ID', $profile_base_value );
}

$user_id = ! empty( $single_user->ID ) ? $single_user->ID : 0;
```

### 3. Data Container Setup
**File**: `includes/ShortCode.php` (Lines 89-102)
```php
$this->page_url       = get_permalink();
$this->per_page       = $atts['per_page'];
$this->roles_exclude  = $atts['roles_exclude'];
$this->roles_include  = $atts['roles_include'];
$this->status_exclude = $status_exclude;

$all_data = [];
$all_data['profile_permalink_base'] = $profile_permalink_base;
$all_data['profile_base_value'] = $profile_base_value;
$all_data['user_id'] = $user_id;
$all_data['per_page'] = $atts['per_page'];
$all_data['roles_exclude'] = $atts['roles_exclude'];
$all_data['roles_include'] = $atts['roles_include'];
$all_data['status_exclude'] = $status_exclude;
```

---

## Template System

### 1. Main Template Router
**File**: `templates/user-lists.php` (Lines 1-69)

#### Key Components:
- **Data Validation**: Ensures `$all_data` is available
- **Column Configuration**: Gets display columns from settings
- **Pagination Setup**: Calculates current page, offset, and limits
- **User Query**: Fetches users based on criteria
- **Template Data**: Prepares data for layout templates

```php
// Get columns from $all_data or fallback
$columns = ! empty( $all_data['columns'] ) ? $all_data['columns'] : [ 'username', 'name' ];

// Calculate pagination
$per_page = ! empty( $all_data['max_item_per_page'] ) ? (int) $all_data['max_item_per_page'] : 10;
$current_page = 1;
if ( get_query_var( 'paged' ) ) {
    $current_page = (int) get_query_var( 'paged' );
} elseif ( ! empty( $_GET['page'] ) ) {
    $current_page = (int) $_GET['page'];
}
```

### 2. Layout Selection
**File**: `templates/user-lists.php` (Lines 69+)
```php
switch ( $layout ) {
    case 'list1':
        wp_enqueue_style( 'wpuf-listing-layout-one' );
        wpuf_load_pro_template( 'layout-1.php', $all_data, WPUF_UD_TEMPLATES . '/user-lists/' );
        break;
    case 'list2':
        wp_enqueue_style( 'wpuf-listing-layout-two' );
        wpuf_load_pro_template( 'layout-2.php', $all_data, WPUF_UD_TEMPLATES . '/user-lists/' );
        break;
    // ... more layouts
}
```

### 3. Layout Templates Structure

#### Layout 1 (Table Layout)
**File**: `templates/user-lists/layout-1.php`

**Structure**:
```html
<div class="wpuf-user-listing">
    <!-- Header with Search and Sort -->
    <div class="wpuf-user-directory-header">
        <!-- Search Field Component -->
        <!-- Sort Field Component -->
    </div>
    
    <!-- Main Content Area -->
    <div class="wpuf-ud-list">
        <!-- Table Structure -->
        <table class="wpuf-ud-table">
            <thead>
                <!-- Dynamic Column Headers -->
            </thead>
            <tbody>
                <!-- User Rows via row-1.php template -->
            </tbody>
        </table>
    </div>
    
    <!-- Pagination Component -->
    <div class="wpuf-ud-pagination">
        <!-- Pagination Template -->
    </div>
</div>
```

### 4. Template Parts System

#### Row Templates
**File**: `templates/user-lists/template-parts/row-1.php`

**Features**:
- Dynamic column rendering
- Avatar handling with fallback
- User data display
- Action buttons
- Responsive design classes

```php
<tr class="wpuf-ud-row">
    <?php foreach ( $columns as $column ) : ?>
        <td class="wpuf-p-4 wpuf-text-sm wpuf-text-gray-900 wpuf-text-center">
            <?php
            switch ( $column ) {
                case 'avatar':
                    // Avatar rendering logic
                    break;
                case 'username':
                    // Username display logic
                    break;
                // ... more columns
            }
            ?>
        </td>
    <?php endforeach; ?>
    <td class="wpuf-p-4 wpuf-text-center">
        <!-- Action buttons -->
    </td>
</tr>
```

#### Search Field Component
**File**: `templates/user-lists/template-parts/search-field.php`

**Features**:
- Context-aware rendering (block vs shortcode)
- AJAX-enabled search input
- Responsive design
- Accessibility features

```php
<div class="wpuf-ud-search-wrapper" 
     data-block-id="<?php echo esc_attr( $block_id ); ?>" 
     data-page-id="<?php echo esc_attr( $page_id ); ?>">
    <input type="text" class="wpuf-ud-search-input" placeholder="Search users..." />
    <svg class="search-icon"><!-- Search icon SVG --></svg>
</div>
```

#### Pagination Component
**File**: `templates/user-lists/template-parts/pagination-shortcode.php`

**Features**:
- Shortcode-specific pagination
- Query parameter preservation
- Responsive page navigation
- Custom styling per layout

```php
// Calculate visible page range - show max 7 pages
$max_visible = 7;
$pages = [];

if ( $total <= $max_visible ) {
    // Show all pages
    for ( $i = 1; $i <= $total; $i++ ) {
        $pages[] = $i;
    }
} else {
    // Smart pagination logic
}
```

---

## Search & Filtering

### 1. Search Implementation
**File**: `includes/ShortCode.php` (Lines 323-400)

#### Search Query Logic:
```php
public function user_listing_search() {
    $search_meta       = $this->search_meta_field();
    $search_by         = isset( $_GET['search_by'] ) ? sanitize_text_field( wp_unslash( $_GET['search_by'] ) ) : '';
    $orderby           = isset( $_GET['order_by'] ) ? sanitize_text_field( wp_unslash( $_GET['order_by'] ) ) : 'login';
    $order             = isset( $_GET['order'] ) ? sanitize_text_field( wp_unslash( $_GET['order'] ) ) : 'ASC';
    $search_query      = isset( $_GET['search_field'] ) ? sanitize_text_field( wp_unslash( $_GET['search_field'] ) ) : '';
}
```

### 2. User Query Construction
**File**: `includes/ShortCode.php` (Lines 512-550)

#### Query Arguments:
```php
$args = array(
    'number'        => $per_page,
    'offset'        => $offset,
    'orderby'       => $orderby,
    'order'         => $order,
);

if ( $this->roles_include ) {
    $args['role__in'] = explode( ',', $this->roles_include );
}

if ( ! empty( $this->status_exclude ) ) {
    $args['meta_query'] = [
        [
            'key'     => 'wpuf_user_status',
            'value'   => $this->status_exclude,
            'compare' => 'NOT IN',
        ],
    ];
}
```

### 3. Search Field Handling
**File**: `includes/ShortCode.php` (Lines 550-580)

#### Search Logic:
```php
if ( ! empty( $_GET['search_field'] ) ) {
    $search_query = sanitize_text_field( wp_unslash( $_GET['search_field'] ) );
    $search_by = isset( $_GET['search_by'] ) ? sanitize_text_field( wp_unslash( $_GET['search_by'] ) ) : '';

    if ( 'all' !== $search_by && in_array( $search_by, array( 'ID', 'user_login', 'user_nicename', 'user_email', 'user_url', 'display_name' ), true ) ) {
        $args[ $search_by ] = '*' . $search_query . '*';
    } else {
        $args['search'] = '*' . $search_query . '*';
        $args['search_columns'] = array( 'user_login', 'user_nicename', 'user_email', 'display_name' );
    }
}
```

---

## Pagination System

### 1. Pagination Data Calculation
**File**: `templates/user-lists.php` (Lines 40-55)

```php
$total_pages  = $per_page > 0 ? (int) ceil( $total / $per_page ) : 1;
$pagination   = [
    'total_pages'  => $total_pages,
    'current_page' => $current_page,
    'per_page'     => $per_page,
    'total_items'  => $total,
];
```

### 2. URL Construction
**File**: `templates/user-lists/template-parts/pagination-shortcode.php`

#### Features:
- Preserves existing query parameters
- Supports both GET and pretty permalink styles
- Maintains search and filter states

```php
// Determine base_url and query_args if not provided
if ( ! isset( $base_url ) || ! isset( $query_args ) ) {
    $current_url = ( is_ssl() ? 'https://' : 'http://' ) . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
    $parsed_url = wp_parse_url( $current_url );
    $base_url = $parsed_url['path'] ?? '';

    $query_args = [];
    if ( ! empty( $parsed_url['query'] ) ) {
        parse_str( $parsed_url['query'], $query_args );
    }
    unset( $query_args['page'] ); // We'll set this per link
}
```

---

## Profile Rendering

### 1. Profile Detection
**File**: `includes/ShortCode.php` (Lines 781-800)

```php
public function user_directory_profile() {
    echo '<ul class="wpuf-user-profile">';
    $user_id     = isset( $_GET['user_id'] ) ? intval( wp_unslash( $_GET['user_id'] ) ) : '';
    $user_status = self::is_approved( $user_id );

    if ( ! $user_status ) {
        return;
    }

    $userdata          = get_user_by( 'id', $user_id );
    $current_user      = wp_get_current_user();
    $profile_fields    = $this->get_options();
    $this->settings    = isset( $profile_fields['settings'] ) ? $profile_fields['settings'] : array();
    $profile_role      = isset( $userdata->roles[0] ) ? $userdata->roles[0] : '';
    $current_user_role = is_user_logged_in() ? $current_user->roles[0] : 'guest';
```

### 2. Field Rendering Logic
**File**: `includes/ShortCode.php` (Lines 806-890)

#### Field Types:
- **Meta Fields**: User metadata display
- **Post Fields**: User's posts listing
- **Comment Fields**: User's comments
- **File Fields**: File attachments
- **Social Fields**: Social media links
- **Section Fields**: Content sections

```php
foreach ( $profile_fields['fields'] as $key => $field ) {
    if ( ! self::can_user_see( $profile_role, $field, $current_user_role ) ) {
        continue;
    }

    echo '<li>';

    switch ( $field['type'] ) {
        case 'meta':
            // Meta field rendering
            break;
        case 'post':
            // Post listing rendering
            break;
        case 'social':
            // Social links rendering
            break;
        // ... more field types
    }

    echo '</li>';
}
```

---

## Extension Points

### 1. WordPress Hooks

#### Action Hooks:
```php
// Profile rendering hooks
do_action( 'wpuf_user_profile_before_content' );
do_action( 'wpuf_user_profile_after_content' );
do_action( 'wpuf_user_about_meta', $meta_key );

// Admin hooks
do_action( 'wpuf-admin-profile-builder' );
```

#### Filter Hooks:
```php
// Shortcode registration
add_filter( 'wpuf_page_shortcodes', array( $this, 'wpuf_add_user_listing_shortcode' ) );

// Pagination customization
$pagination_args = apply_filters( 'wpuf_user_directory_pagination_args', $pagination_args, $all_data );
```

### 2. Template Override System

#### Template Hierarchy:
1. **Theme Override**: `theme/wpuf-pro/user-directory/templates/`
2. **Plugin Templates**: `modules/user-directory/templates/`

#### Custom Layout Addition:
```php
// Add new layout option
function custom_user_directory_layout( $layouts ) {
    $layouts['custom-layout'] = 'Custom Layout';
    return $layouts;
}
add_filter( 'wpuf_user_directory_layouts', 'custom_user_directory_layout' );
```

---

## File Structure

```
modules/user-directory/
├── userlisting.php                    # Main plugin file
├── includes/
│   ├── Admin.php                      # Admin functionality
│   ├── Admin/
│   │   ├── Builder.php               # Form builder
│   │   └── Form.php                  # Form handling
│   ├── Frontend.php                  # Frontend functionality
│   ├── ShortCode.php                 # Main shortcode class
│   └── Assets.php                    # Asset management
├── templates/
│   ├── user-lists.php                # Main template router
│   ├── user-lists/
│   │   ├── layout-1.php             # Table layout
│   │   ├── layout-2.php             # Card layout
│   │   ├── layout-3.php             # Grid layout
│   │   ├── layout-4.php             # List layout
│   │   ├── layout-5.php             # Circle grid layout
│   │   ├── layout-6.php             # Square grid layout
│   │   └── template-parts/
│   │       ├── row-1.php            # Table row template
│   │       ├── row-2.php            # Card row template
│   │       ├── row-3.php            # Grid row template
│   │       ├── row-4.php            # List row template
│   │       ├── row-5.php            # Circle row template
│   │       ├── row-6.php            # Square row template
│   │       ├── search-field.php     # Search component
│   │       ├── sort-field.php       # Sort component
│   │       ├── social-icons.php     # Social icons component
│   │       └── pagination-shortcode.php # Pagination component
│   └── shortcodes/
│       ├── directory/               # Shortcode-specific templates
│       └── profile/                 # Profile templates
├── assets/
│   ├── css/                         # Stylesheets
│   ├── js/                          # JavaScript files
│   └── images/                      # Image assets
└── languages/                       # Translation files
```

---

## Usage Examples

### 1. Basic Usage
```php
[wpuf_user_listing]
```

### 2. With Role Filtering
```php
[wpuf_user_listing roles_include="author,contributor" per_page="10"]
```

### 3. With Status Exclusion
```php
[wpuf_user_listing status_exclude="pending,denied" per_page="12"]
```

### 4. Combined Parameters
```php
[wpuf_user_listing 
    per_page="15" 
    roles_include="subscriber,contributor" 
    roles_exclude="administrator" 
    status_exclude="denied"]
```

### 5. PHP Implementation
```php
// In theme files
echo do_shortcode( '[wpuf_user_listing per_page="8"]' );

// Programmatic usage
$shortcode = new WPUF\UserDirectory\ShortCode();
echo $shortcode->user_lists( array(
    'per_page' => 10,
    'role' => 'subscriber'
) );
```

---

## Technical Notes

### 1. Performance Considerations
- User queries are paginated to prevent large dataset loading
- CSS/JS assets are conditionally loaded based on layout
- Caching mechanisms can be implemented via WordPress transients

### 2. Security Measures
- All user inputs are sanitized using WordPress functions
- User permissions are checked before profile access
- SQL injection prevention through prepared statements

### 3. Responsive Design
- All layouts use responsive CSS classes
- Mobile-first design approach
- Tailwind CSS utility classes for consistent styling

### 4. Accessibility Features
- ARIA labels for screen readers
- Keyboard navigation support
- Semantic HTML structure
- Alt text for images and icons

---

## Debugging Tips

### 1. Template Debugging
```php
// Add to wp-config.php
define( 'WPUF_DEBUG', true );

// Check template loading
add_action( 'wpuf_template_debug', function( $template, $args ) {
    error_log( "Loading template: $template" );
    error_log( "Template args: " . print_r( $args, true ) );
}, 10, 2 );
```

### 2. Query Debugging
```php
// Log user queries
add_filter( 'wpuf_user_directory_query_args', function( $args ) {
    error_log( 'User query args: ' . print_r( $args, true ) );
    return $args;
} );
```

### 3. Common Issues
- **No users showing**: Check user roles and status filters
- **Pagination not working**: Verify permalink structure
- **Search not functioning**: Check JavaScript console for errors
- **Styling issues**: Verify CSS file loading and conflicts

---

This documentation provides a complete overview of the WPUF User Directory shortcode system, from registration to rendering, including all major components and extension points.
