<?php

namespace WPUF\UserDirectory\Api;

use WP_Error;
use WP_REST_Controller;
use WP_REST_Server;

/**
 * API for getting user meta fields.
 *
 * @since 4.2.0
 */
class Meta_Fields extends WP_REST_Controller {

    /**
     * The namespace of this controller's route.
     *
     * @var string
     */
    protected $namespace = 'wpuf/v1';

    /**
     * The base of this controller's route.
     *
     * @var string
     */
    protected $rest_base = 'user-meta-keys';

    /**
     * Register the routes for the objects of the controller.
     *
     * @since 4.2.0
     */
    public function register_routes() {
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [ $this, 'get_meta_keys' ],
                    'permission_callback' => [ $this, 'get_meta_keys_permission_check' ],
                    'args'                => $this->get_collection_params(),
                ],
                'schema' => [ $this, 'get_public_item_schema' ],
            ]
        );
    }

    /**
     * Check if a given request has access to get user meta keys.
     *
     * @since 4.2.0
     *
     * @param \WP_REST_Request $request Full data about the request.
     *
     * @return \WP_Error|bool
     */
    public function get_meta_keys_permission_check( $request ) {
        if ( ! current_user_can( 'edit_posts' ) ) {
            return new WP_Error( 'rest_forbidden', __( 'You do not have permission to access this resource.', 'wp-user-frontend' ), [ 'status' => 403 ] );
        }

        return true;
    }

    /**
     * Retrieves a list of user meta keys.
     *
     * @since 4.2.0
     *
     * @param \WP_REST_Request $request Full data about the request.
     *
     * @return \WP_REST_Response|\WP_Error
     */
    public function get_meta_keys( $request ) {
        // Get usermeta keys using existing function
        $meta_fields_data = wpuf_get_user_meta_fields();

        // Extract just the keys for backward compatibility
        $meta_keys = array_keys( $meta_fields_data );

        $profile_fields = [
            'user_login'        => __( 'Username', 'wp-user-frontend' ),
            'first_name'        => __( 'First Name', 'wp-user-frontend' ),
            'last_name'         => __( 'Last Name', 'wp-user-frontend' ),
            'display_name'      => __( 'Display Name', 'wp-user-frontend' ),
            'nickname'          => __( 'Nickname', 'wp-user-frontend' ),
            'user_email'        => __( 'E-mail', 'wp-user-frontend' ),
            'user_url'          => __( 'Website', 'wp-user-frontend' ),
            'description'       => __( 'Biographical Info', 'wp-user-frontend' ),
        ];

        $response_data = [
            'profile_fields' => $profile_fields,
            'meta_keys'      => $meta_keys,
        ];

        return rest_ensure_response( $response_data );
    }
}
