<?php
/**
 * User Tabs Ajax Handler
 *
 * Handles AJAX requests for user tabs functionality
 *
 * @since 4.2.0
 * @package WPUF\UserDirectory
 */

namespace WPUF\UserDirectory;

use WPUF\UserDirectory\FileHelper;

/**
 * User Tabs AJAX Handler
 * Handles dynamic loading of tab content via AJAX
 *
 * @since 4.2.0
 * @package WPUF_Pro
 */
class UserTabsAjaxHandler {

    /**
     * Constructor
     *
     * @since 4.2.0
     */
    public function __construct() {
        add_action( 'wp_ajax_wpuf_load_user_posts', [ $this, 'load_user_posts' ] );
        add_action( 'wp_ajax_nopriv_wpuf_load_user_posts', [ $this, 'load_user_posts' ] );

        add_action( 'wp_ajax_wpuf_load_user_comments', [ $this, 'load_user_comments' ] );
        add_action( 'wp_ajax_nopriv_wpuf_load_user_comments', [ $this, 'load_user_comments' ] );

        add_action( 'wp_ajax_wpuf_load_user_files', [ $this, 'load_user_files' ] );
        add_action( 'wp_ajax_nopriv_wpuf_load_user_files', [ $this, 'load_user_files' ] );

        add_action( 'wp_ajax_wpuf_load_user_activity', [ $this, 'load_user_activity' ] );
        add_action( 'wp_ajax_nopriv_wpuf_load_user_activity', [ $this, 'load_user_activity' ] );
    }

    /**
     * Load user posts for the Posts tab
     *
     * @since 4.2.0
     */
    public function load_user_posts() {
        // Verify nonce
        if ( ! wp_verify_nonce( $_POST['nonce'], 'wpuf_user_tabs_nonce' ) ) {
            wp_die( __( 'Security check failed', 'wpuf-pro' ) );
        }

        $user_id = absint( $_POST['user_id'] );
        $posts_per_page = absint( $_POST['posts_per_page'] ) ?: 5;
        $show_excerpts = $_POST['show_excerpts'] === 'true';

        if ( ! $user_id ) {
            wp_send_json_error( [ 'message' => __( 'Invalid user ID', 'wpuf-pro' ) ] );
        }

        $user = get_user_by( 'ID', $user_id );
        if ( ! $user ) {
            wp_send_json_error( [ 'message' => __( 'User not found', 'wpuf-pro' ) ] );
        }

        $posts = get_posts( [
            'author'      => $user_id,
            'numberposts' => $posts_per_page,
            'post_status' => 'publish',
            'orderby'     => 'date',
            'order'       => 'DESC',
        ] );

        if ( empty( $posts ) ) {
            wp_send_json_error( [ 'message' => __( 'No posts found', 'wpuf-pro' ) ] );
        }

        $html = '<div class="wpuf-overflow-hidden wpuf-shadow wpuf-outline wpuf-outline-1 wpuf-outline-black/5 wpuf-rounded-lg wpuf-dark:shadow-none wpuf-dark:-outline-offset-1 wpuf-dark:outline-white/10">';
        $html .= '<table class="wpuf-relative wpuf-min-w-full wpuf-divide-y wpuf-divide-gray-300 wpuf-dark:divide-white/15">';

        // Table header
        $html .= '<thead class="wpuf-bg-gray-50 wpuf-dark:bg-gray-800/75">';
        $html .= '<tr>';
        $html .= '<th scope="col" class="wpuf-py-3.5 wpuf-pr-3 wpuf-pl-4 wpuf-text-left wpuf-text-sm wpuf-font-semibold wpuf-text-gray-900 wpuf-sm:pl-6 wpuf-dark:text-gray-200">';
        $html .= esc_html__( 'Post Title', 'wpuf-pro' );
        $html .= '</th>';
        $html .= '<th scope="col" class="wpuf-px-3 wpuf-py-3.5 wpuf-text-left wpuf-text-sm wpuf-font-semibold wpuf-text-gray-900 wpuf-dark:text-gray-200">';
        $html .= esc_html__( 'Publish Date', 'wpuf-pro' );
        $html .= '</th>';
        $html .= '<th scope="col" class="wpuf-py-3.5 wpuf-pr-4 wpuf-pl-3 wpuf-sm:pr-6">';
        $html .= '<span class="wpuf-sr-only">' . esc_html__( 'View', 'wpuf-pro' ) . '</span>';
        $html .= '</th>';
        $html .= '</tr>';
        $html .= '</thead>';

        // Table body
        $html .= '<tbody class="wpuf-divide-y wpuf-divide-gray-200 wpuf-bg-white wpuf-dark:divide-white/10 wpuf-dark:bg-gray-800/50">';

        foreach ( $posts as $post ) {
            $html .= '<tr>';
            $html .= '<td class="wpuf-py-4 wpuf-pr-3 wpuf-pl-4 wpuf-text-sm wpuf-font-medium wpuf-whitespace-nowrap wpuf-text-gray-900 wpuf-sm:pl-6 wpuf-dark:text-white">';
            $html .= esc_html( get_the_title( $post->ID ) );
            $html .= '</td>';
            $html .= '<td class="wpuf-px-3 wpuf-py-4 wpuf-text-sm wpuf-whitespace-nowrap wpuf-text-gray-500 wpuf-dark:text-gray-400">';
            $html .= esc_html( get_the_date( 'F j, Y', $post->ID ) );
            $html .= '</td>';
            $html .= '<td class="wpuf-py-4 wpuf-pr-4 wpuf-pl-3 wpuf-text-right wpuf-text-sm wpuf-font-medium wpuf-whitespace-nowrap wpuf-sm:pr-6">';
            $html .= '<a href="' . esc_url( get_permalink( $post->ID ) ) . '" class="wpuf-text-indigo-600 wpuf-hover:text-indigo-900 wpuf-dark:text-indigo-400 wpuf-dark:hover:text-indigo-300">';
            $html .= esc_html__( 'Post Link ->', 'wpuf-pro' );
            $html .= '<span class="wpuf-sr-only">, ' . esc_html( get_the_title( $post->ID ) ) . '</span>';
            $html .= '</a>';
            $html .= '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>';
        $html .= '</table>';
        $html .= '</div>';

        wp_send_json_success( [ 'html' => $html ] );
    }

    /**
     * Load user comments for the Comments tab
     *
     * @since 4.2.0
     */
    public function load_user_comments() {
        // Verify nonce
        if ( ! wp_verify_nonce( $_POST['nonce'], 'wpuf_user_tabs_nonce' ) ) {
            wp_die( __( 'Security check failed', 'wpuf-pro' ) );
        }

        $user_id = absint( $_POST['user_id'] );
        $comment_length = absint( $_POST['comment_length'] ) ?: 100;

        if ( ! $user_id ) {
            wp_send_json_error( [ 'message' => __( 'Invalid user ID', 'wpuf-pro' ) ] );
        }

        $user = get_user_by( 'ID', $user_id );
        if ( ! $user ) {
            wp_send_json_error( [ 'message' => __( 'User not found', 'wpuf-pro' ) ] );
        }

        $comments = get_comments( [
            'user_id' => $user_id,
            'number'  => 10,
            'status'  => 'approve',
            'orderby' => 'comment_date',
            'order'   => 'DESC',
        ] );

        if ( empty( $comments ) ) {
            wp_send_json_error( [ 'message' => __( 'No comments found', 'wpuf-pro' ) ] );
        }

        // Modern minimal design wrapper
        $html = '<div class="wpuf-overflow-hidden wpuf-shadow wpuf-outline wpuf-outline-1 wpuf-outline-black/5 wpuf-rounded-lg wpuf-dark:shadow-none wpuf-dark:-outline-offset-1 wpuf-dark:outline-white/10">';
        $html .= '<div class="wpuf-divide-y wpuf-divide-gray-200 wpuf-bg-white wpuf-dark:divide-white/10 wpuf-dark:bg-gray-800/50">';

        foreach ( $comments as $comment ) {
            $comment_content = wp_strip_all_tags( $comment->comment_content );
            $comment_excerpt = wp_trim_words( $comment_content, 20, '...' );
            $comment_date    = date_i18n( 'M j, Y', strtotime( $comment->comment_date ) );
            $post_title     = get_the_title( $comment->comment_post_ID );
            $post_url       = get_permalink( $comment->comment_post_ID );
            $reply_count    = get_comments( [
                'parent' => $comment->comment_ID,
                'count'  => true,
            ] );

            $html .= sprintf(
                '<div class="wpuf-px-4 wpuf-py-4">
                    <div class="wpuf-flex wpuf-items-start wpuf-gap-3">
                        <div class="wpuf-flex-shrink-0">
                            <div class="wpuf-w-8 wpuf-h-8 wpuf-bg-gray-200 wpuf-dark:bg-gray-700 wpuf-rounded-full wpuf-flex wpuf-items-center wpuf-justify-center">
                                <span class="wpuf-text-xs wpuf-font-medium wpuf-text-gray-600 wpuf-dark:text-gray-300">%s</span>
                            </div>
                        </div>
                        <div class="wpuf-flex-1 wpuf-min-w-0">
                            <div class="wpuf-flex wpuf-items-center wpuf-gap-2 wpuf-mb-1">
                                <span class="wpuf-text-sm wpuf-font-medium wpuf-text-gray-900 wpuf-dark:text-white">%s</span>
                                <span class="wpuf-text-xs wpuf-text-gray-500 wpuf-dark:text-gray-400">%s</span>
                            </div>
                            <p class="wpuf-text-sm wpuf-text-gray-700 wpuf-dark:text-gray-300 wpuf-mb-2">%s</p>
                            <div class="wpuf-flex wpuf-items-center wpuf-gap-4 wpuf-text-xs wpuf-text-gray-500 wpuf-dark:text-gray-400">
                                <span>%s: <a href="%s" class="wpuf-text-indigo-600 wpuf-hover:text-indigo-900 wpuf-dark:text-indigo-400 wpuf-dark:hover:text-indigo-300">%s</a></span>%s
                            </div>
                        </div>
                    </div>
                </div>',
                esc_html( strtoupper( substr( $user->display_name, 0, 1 ) ) ),
                esc_html( $user->display_name ),
                esc_html( $comment_date ),
                esc_html( $comment_excerpt ),
                esc_html__( 'On', 'wpuf-pro' ),
                esc_url( $post_url ),
                esc_html( $post_title ),
                $reply_count > 0 ? sprintf( '<span>%d %s</span>', $reply_count, esc_html__( 'replies', 'wpuf-pro' ) ) : ''
            );
        }

        $html .= '</div>';
        $html .= '</div>';

        wp_send_json_success( [ 'html' => $html ] );
    }

    /**
     * Load user files for the Files tab
     *
     * @since 4.2.0
     */
    public function load_user_files() {
        // Verify nonce
        if ( ! wp_verify_nonce( $_POST['nonce'], 'wpuf_user_tabs_nonce' ) ) {
            wp_die( __( 'Security check failed', 'wpuf-pro' ) );
        }

        $user_id = absint( $_POST['user_id'] );

        if ( ! $user_id ) {
            wp_send_json_error( [ 'message' => __( 'Invalid user', 'wpuf-pro' ) ] );
        }

        $user = get_user_by( 'ID', $user_id );
        if ( ! $user ) {
            wp_send_json_error( [ 'message' => __( 'User not found', 'wpuf-pro' ) ] );
        }

        // Check if user has any files using the existing function
        if ( ! wpuf_ud_user_has_files( $user_id ) ) {
            wp_send_json_error( [ 'message' => __( 'No files found', 'wpuf-pro' ) ] );
        }

        // Prepare field configuration for the grouped files function
        $field_config = [
            'meta_key' => '', // Empty to get all user files
            'label' => __( 'Files', 'wpuf-pro' )
        ];

        // Start output buffering to capture the HTML output
        ob_start();

        // Use the new grouped files function
        wpuf_ud_render_files_grouped_by_type( $field_config, $user_id, [], 'gray' );

        // Get the captured HTML
        $html = ob_get_clean();

        if ( empty( $html ) ) {
            wp_send_json_error( [ 'message' => __( 'No files found', 'wpuf-pro' ) ] );
        }

        wp_send_json_success( [ 'html' => $html ] );
    }

    /**
     * Load user activity for the Activity tab
     *
     * @since 4.2.0
     */
    public function load_user_activity() {
        // Verify nonce
        if ( ! wp_verify_nonce( $_POST['nonce'], 'wpuf_user_tabs_nonce' ) ) {
            wp_die( __( 'Security check failed', 'wpuf-pro' ) );
        }

        $user_id = absint( $_POST['user_id'] );

        if ( ! $user_id ) {
            wp_send_json_error( [ 'message' => __( 'Invalid user ID', 'wpuf-pro' ) ] );
        }

        $user = get_user_by( 'ID', $user_id );
        if ( ! $user ) {
            wp_send_json_error( [ 'message' => __( 'User not found', 'wpuf-pro' ) ] );
        }

        // Activity tab content is handled elsewhere via hooks; nothing to process here
        wp_send_json_success( [ 'html' => '' ] );
    }


}
