import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';

const UserBio = ({ user, attributes, context }) => {
    const {
        contentSource = 'description',
        characterLimit = 150,
        showReadMore = true,
        readMoreText = 'Read More',
        emptyMessage = '',
        allowHTML = false,
        customFieldKey = ''
    } = attributes || {};

    const [isExpanded, setIsExpanded] = useState(false);

    // Get bio content based on source
    const getBioContent = () => {
        if (!user) return __('This is a sample biography text that would be displayed for the user.', 'wpuf-pro');

        switch (contentSource) {
            case 'description':
                // Try different description fields
                return user.description || user.bio || '';
            case 'custom_field':
                // Check both meta object and direct user properties
                if (user.meta && user.meta[customFieldKey]) {
                    return user.meta[customFieldKey];
                }
                return user[customFieldKey] || '';
            default:
                return user.description || user.bio || '';
        }
    };

    // Process bio content with character limits
    const processedBio = () => {
        const bioContent = getBioContent();

        if (!bioContent) {
            return emptyMessage || __('No biography available.', 'wpuf-pro');
        }

        if (characterLimit === 0) {
            return bioContent;
        }

        if (bioContent.length <= characterLimit) {
            return bioContent;
        }

        if (!showReadMore) {
            return bioContent.substring(0, characterLimit) + '...';
        }

        // For toggle, show truncated or full based on state
        if (isExpanded) {
            return bioContent;
        }

        return bioContent.substring(0, characterLimit) + '...';
    };

    // Render read more functionality
    const renderReadMore = () => {
        const bioContent = getBioContent();

        if (!bioContent || bioContent.length <= characterLimit || characterLimit === 0 || !showReadMore) {
            return null;
        }

        return (
            <a
                href="#"
                className="wpuf-read-more-toggle"
                onClick={(e) => {
                    e.preventDefault();
                    setIsExpanded(!isExpanded);
                }}
            >
                {isExpanded
                    ? __('Show Less', 'wpuf-pro')
                    : (readMoreText || __('Read More', 'wpuf-pro'))
                }
            </a>
        );
    };

    return (
        <div className="wpuf-user-bio-container">
            <div className="wpuf-bio-content">
                {allowHTML ? (
                    <div
                        className="wpuf-bio-text"
                        dangerouslySetInnerHTML={{
                            __html: processedBio()
                        }}
                    />
                ) : (
                    <p className="wpuf-bio-text">
                        {processedBio()}
                    </p>
                )}

                {renderReadMore()}
            </div>
        </div>
    );
};

export default UserBio; 