/**
 * WordPress dependencies
 */
import { useCallback } from '@wordpress/element';
import { useSelect } from '@wordpress/data';
import { store as blockEditorStore } from '@wordpress/block-editor';

/**
 * Hook that returns the tools panel dropdown menu props.
 * This is used to ensure consistent dropdown behavior across tools panels.
 *
 * @return {Object} The dropdown menu props.
 */
export const useToolsPanelDropdownMenuProps = () => {
	return useCallback(() => ({}), []);
};

/**
 * Hook that returns whether the current block is in a user directory context.
 *
 * @param {string} clientId The block client ID.
 * @return {boolean} Whether the block is in a user directory context.
 */
export const useIsInUserDirectory = (clientId) => {
	return useSelect(
		(select) => {
			const { getBlockParents } = select(blockEditorStore);
			const parents = getBlockParents(clientId);
			
			return parents.some((parentId) => {
				const parentBlock = select(blockEditorStore).getBlock(parentId);
				return parentBlock?.name === 'wpuf-ud/directory';
			});
		},
		[clientId]
	);
};

/**
 * Hook that returns the user directory context for a block.
 *
 * @param {string} clientId The block client ID.
 * @return {Object} The user directory context.
 */
export const useUserDirectoryContext = (clientId) => {
	return useSelect(
		(select) => {
			const { getBlockParents, getBlock } = select(blockEditorStore);
			const parents = getBlockParents(clientId);
			
			for (const parentId of parents) {
				const parentBlock = getBlock(parentId);
				if (parentBlock?.name === 'wpuf-ud/directory') {
					return {
						query: parentBlock.attributes.query,
						directory_layout: parentBlock.attributes.directory_layout,
								enable_search: parentBlock.attributes.enable_search,
		search_placeholder: parentBlock.attributes.search_placeholder,
					};
				}
			}
			
			return null;
		},
		[clientId]
	);
}; 