import { __ } from '@wordpress/i18n';

// Default tab labels - will be translated
const DEFAULT_TAB_LABELS = {
    about: __('About', 'wpuf-pro'),
    posts: __('Posts', 'wpuf-pro'),
    comments: __('Comments', 'wpuf-pro'),
    files: __('Files', 'wpuf-pro')
};

/**
 * Get available tabs dynamically
 * 
 * @param {Object} customTabs - Additional custom tabs to include
 * @returns {Object} Available tabs with their labels
 */
export const getAvailableTabs = (customTabs = {}) => {
    const tabs = { ...DEFAULT_TAB_LABELS, ...customTabs };
    
    // Add activity tab if user activity module exists
    if (window.wpufUserActivityExists) {
        tabs.activity = __('Activity', 'wpuf-pro');
    }
    
    return tabs;
};

/**
 * Get visible tabs (enabled tabs that are available) in correct order
 * 
 * @param {Array} enabledTabs - Array of enabled tab IDs
 * @param {Array} tabOrder - Array defining the order of tabs
 * @param {Object} availableTabs - Available tabs object
 * @returns {Array} Array of visible tab IDs in correct order
 */
export const getVisibleTabs = (enabledTabs = [], tabOrder = [], availableTabs = {}) => {
    const availableEnabledTabs = (enabledTabs || []).filter(tabId => availableTabs[tabId]);

    // Sort by tabOrder, with any new tabs added at the end
    const sorted = [...availableEnabledTabs].sort((a, b) => {
        const aIndex = (tabOrder || []).indexOf(a);
        const bIndex = (tabOrder || []).indexOf(b);
        if (aIndex === -1 && bIndex === -1) return 0;
        if (aIndex === -1) return 1;
        if (bIndex === -1) return -1;
        return aIndex - bIndex;
    });
    
    return sorted;
};

/**
 * Get tab label (custom or default)
 * 
 * @param {string} tabId - Tab identifier
 * @param {Object} customLabels - Custom labels object
 * @param {Object} availableTabs - Available tabs object
 * @returns {string} Tab label
 */
export const getTabLabel = (tabId, customLabels = {}, availableTabs = {}) => {
    return (customLabels && customLabels[tabId]) || availableTabs[tabId] || tabId;
};

/**
 * Get all tabs in order (union of saved order and available tabs)
 * 
 * @param {Array} tabOrder - Array defining the order of tabs
 * @param {Object} availableTabs - Available tabs object
 * @returns {Array} Array of all tab IDs in order
 */
export const getAllTabsInOrder = (tabOrder = [], availableTabs = {}) => {
    return Array.from(new Set([
        ...tabOrder.filter(tabId => availableTabs[tabId]),
        ...Object.keys(availableTabs)
    ]));
};

/**
 * Reorder tabs by moving an item from one index to another
 * 
 * @param {Array} tabOrder - Current tab order
 * @param {number} fromIndex - Source index
 * @param {number} toIndex - Destination index
 * @param {Object} availableTabs - Available tabs object
 * @returns {Array} New tab order array
 */
export const reorderTabs = (tabOrder = [], fromIndex, toIndex, availableTabs = {}) => {
    // Build union list (saved order first, then any new available tabs)
    const unionOrder = getAllTabsInOrder(tabOrder, availableTabs);
    
    const newOrder = [...unionOrder];
    const [movedTab] = newOrder.splice(fromIndex, 1);
    newOrder.splice(toIndex, 0, movedTab);
    
    return newOrder;
};
