<?php
/**
 * User Data Helper Functions
 *
 * Provides user data formatting functions that can be shared between
 * shortcode and block templates.
 *
 * @since 4.2.0
 */

/**
 * Get formatted user display name
 *
 * @param object $user User object
 * @param string $format Name format (display_name|user_login|first_name|last_name|full_name)
 * @return string Formatted name
 * @since 4.2.0
 */
function wpuf_get_user_display_name($user, $format = 'display_name') {
    if (!$user || !is_object($user) || !isset($user->ID)) {
        return '';
    }

    switch ($format) {
        case 'user_login':
            return esc_html($user->user_login);

        case 'first_name':
            $first_name = get_user_meta($user->ID, 'first_name', true);
            return esc_html($first_name ?: $user->display_name);

        case 'last_name':
            $last_name = get_user_meta($user->ID, 'last_name', true);
            return esc_html($last_name ?: $user->display_name);

        case 'full_name':
            $first_name = get_user_meta($user->ID, 'first_name', true);
            $last_name = get_user_meta($user->ID, 'last_name', true);
            $full_name = trim($first_name . ' ' . $last_name);
            return esc_html($full_name ?: $user->display_name);

        default:
            return esc_html($user->display_name);
    }
}

/**
 * Get user avatar with options
 *
 * @param object $user User object
 * @param array $options Avatar options
 * @return string Avatar HTML
 * @since 4.2.0
 */
function wpuf_get_user_avatar($user, $options = []) {
    if (!$user || !is_object($user) || !isset($user->ID)) {
        return '';
    }

    $defaults = [
        'size' => 96,
        'shape' => 'circle',
        'class' => '',
        'alt' => '',
    ];

    $options = wp_parse_args($options, $defaults);

    // Build classes
    $classes = ['wpuf-user-avatar'];
    if ($options['shape'] === 'square') {
        $classes[] = 'rounded';
    } else {
        $classes[] = 'rounded-full';
    }
    if ($options['class']) {
        $classes[] = $options['class'];
    }

    // Use the common avatar function with fallbacks
    return wpuf_ud_get_user_avatar_html($user, $options['size'], implode(' ', $classes));
}

/**
 * Get user bio with formatting
 *
 * @param object $user User object
 * @param array $options Bio options
 * @return string Formatted bio
 * @since 4.2.0
 */
function wpuf_get_user_bio($user, $options = []) {
    if (!$user || !is_object($user) || !isset($user->ID)) {
        return '';
    }

    $defaults = [
        'character_limit' => 100,
        'read_more_style' => 'none',
        'show_read_more' => false,
    ];

    $options = wp_parse_args($options, $defaults);

    $bio = get_user_meta($user->ID, 'description', true);
    if (empty($bio)) {
        return '';
    }

    $display_bio = $bio;
    if ($options['character_limit'] > 0 && strlen($bio) > $options['character_limit']) {
        $display_bio = substr($bio, 0, $options['character_limit']) . '...';
    }

    $read_more = '';
    if ($options['show_read_more'] && strlen($bio) > $options['character_limit']) {
        $read_more = sprintf(
            '<a href="%s" class="wpuf-read-more">%s</a>',
            esc_url(get_author_posts_url($user->ID)),
            esc_html__('Read More', 'wpuf-pro')
        );
    }

    return sprintf(
        '<div class="wpuf-user-bio">
            <p>%s</p>
            %s
        </div>',
        esc_html($display_bio),
        $read_more
    );
}

/**
 * Get user contact information
 *
 * @param object $user User object
 * @param array $options Contact options
 * @return string Contact info HTML
 * @since 4.2.0
 */
function wpuf_get_user_contact_info($user, $options = []) {
    if (!$user || !is_object($user) || !isset($user->ID)) {
        return '';
    }

    $defaults = [
        'show_email' => true,
        'show_website' => true,
        'show_phone' => false,
        'show_address' => false,
    ];

    $options = wp_parse_args($options, $defaults);

    $contact_info = [];

    // Email
    if ($options['show_email'] && !empty($user->user_email)) {
        $contact_info[] = sprintf(
            '<div class="wpuf-contact-item">
                <span class="wpuf-contact-label">%s:</span>
                <a href="mailto:%s">%s</a>
            </div>',
            esc_html__('Email', 'wpuf-pro'),
            esc_attr($user->user_email),
            esc_html($user->user_email)
        );
    }

    // Website
    if ($options['show_website'] && !empty($user->user_url)) {
        $contact_info[] = sprintf(
            '<div class="wpuf-contact-item">
                <span class="wpuf-contact-label">%s:</span>
                <a href="%s" target="_blank">%s</a>
            </div>',
            esc_html__('Website', 'wpuf-pro'),
            esc_url($user->user_url),
            esc_html($user->user_url)
        );
    }

    // Phone
    if ($options['show_phone']) {
        $phone = get_user_meta($user->ID, 'phone_field', true);
        if (!empty($phone)) {
            $contact_info[] = sprintf(
                '<div class="wpuf-contact-item">
                    <span class="wpuf-contact-label">%s:</span>
                    <a href="tel:%s">%s</a>
                </div>',
                esc_html__('Phone', 'wpuf-pro'),
                esc_attr($phone),
                esc_html($phone)
            );
        }
    }

    // Address
    if ($options['show_address']) {
        $address = get_user_meta($user->ID, 'address', true);
        if (!empty($address)) {
            $contact_info[] = sprintf(
                '<div class="wpuf-contact-item">
                    <span class="wpuf-contact-label">%s:</span>
                    <span>%s</span>
                </div>',
                esc_html__('Address', 'wpuf-pro'),
                esc_html($address)
            );
        }
    }

    if (empty($contact_info)) {
        return '';
    }

    return sprintf(
        '<div class="wpuf-user-contact-info">
            %s
        </div>',
        implode('', $contact_info)
    );
}

/**
 * Get user custom field value
 *
 * @param object $user User object
 * @param string $field_name Field name
 * @param array $options Field options
 * @return string Field HTML
 * @since 4.2.0
 */
function wpuf_get_user_custom_field($user, $field_name, $options = []) {
    if (!$user || !is_object($user) || !isset($user->ID) || empty($field_name)) {
        return '';
    }

    $defaults = [
        'field_label' => '',
        'show_label' => true,
        'format' => 'text',
    ];

    $options = wp_parse_args($options, $defaults);

    $field_value = get_user_meta($user->ID, $field_name, true);
    if (empty($field_value)) {
        return '';
    }

    $field_label = $options['field_label'] ?: ucfirst(str_replace('_', ' ', $field_name));

    $label_html = '';
    if ($options['show_label']) {
        $label_html = sprintf(
            '<span class="wpuf-field-label">%s:</span>',
            esc_html($field_label)
        );
    }

    $value_html = '';
    switch ($options['format']) {
        case 'url':
            $value_html = sprintf(
                '<a href="%s" target="_blank">%s</a>',
                esc_url($field_value),
                esc_html($field_value)
            );
            break;

        case 'email':
            $value_html = sprintf(
                '<a href="mailto:%s">%s</a>',
                esc_attr($field_value),
                esc_html($field_value)
            );
            break;

        default:
            $value_html = sprintf(
                '<span class="wpuf-field-value">%s</span>',
                esc_html($field_value)
            );
            break;
    }

    return sprintf(
        '<div class="wpuf-custom-field">
            %s
            %s
        </div>',
        $label_html,
        $value_html
    );
}

/**
 * Get user registration date
 *
 * @param object $user User object
 * @param string $format Date format
 * @return string Formatted date
 * @since 4.2.0
 */
function wpuf_get_user_registration_date($user, $format = 'F j, Y') {
    if (!$user || !is_object($user) || !isset($user->user_registered)) {
        return '';
    }

    return date_i18n($format, strtotime($user->user_registered));
}

/**
 * Get user role display name
 *
 * @param object $user User object
 * @return string Role display name
 * @since 4.2.0
 */
function wpuf_get_user_role_display($user) {
    if (!$user || !is_object($user) || !isset($user->ID)) {
        return '';
    }

    $user_roles = $user->roles ?? [];
    if (empty($user_roles)) {
        return '';
    }

    $role_names = wp_roles()->get_names();
    $display_roles = [];

    foreach ($user_roles as $role) {
        if (isset($role_names[$role])) {
            $display_roles[] = $role_names[$role];
        }
    }

    return implode(', ', $display_roles);
}
