<?php
/**
 * User Contact Info Block Template
 *
 * @since 4.2.0
 *
 * @var WP_User $user User object
 * @var array $attributes Block attributes
 * @var string $content Block content
 * @var WP_Block $block Block instance
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! $user ) {
    return;
}

// Get block attributes with defaults (matching block.json)
$show_fields = isset( $attributes['showFields'] ) ? $attributes['showFields'] : [ 'all' ];
$layout_style = isset( $attributes['layoutStyle'] ) ? $attributes['layoutStyle'] : 'list';
$show_icons = isset( $attributes['showIcons'] ) ? $attributes['showIcons'] : true;
$icon_size = isset( $attributes['iconSize'] ) ? $attributes['iconSize'] : 'medium';
$show_labels = isset( $attributes['showLabels'] ) ? $attributes['showLabels'] : true;
$email_link = isset( $attributes['emailLink'] ) ? $attributes['emailLink'] : true;
$website_link = isset( $attributes['websiteLink'] ) ? $attributes['websiteLink'] : true;
$icon_color = isset( $attributes['iconColor'] ) ? $attributes['iconColor'] : '#000';

// Ensure show_fields is an array
if ( ! is_array( $show_fields ) ) {
    $show_fields = [ $show_fields ];
}

// Define field icons with dynamic color (matching edit.js)
$field_icons = [
    'email' => '<svg width="19" height="14" viewBox="0 0 19 14" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M17.625 2.625V11.375C17.625 12.4105 16.7855 13.25 15.75 13.25H3.25C2.21447 13.25 1.375 12.4105 1.375 11.375V2.625M17.625 2.625C17.625 1.58947 16.7855 0.75 15.75 0.75H3.25C2.21447 0.75 1.375 1.58947 1.375 2.625M17.625 2.625V2.82726C17.625 3.47837 17.2872 4.08287 16.7327 4.42412L10.4827 8.27027C9.88004 8.64113 9.11996 8.64113 8.51732 8.27027L2.26732 4.42412C1.71279 4.08287 1.375 3.47837 1.375 2.82726V2.625" stroke="' . esc_attr($icon_color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
    </svg>',
    'website' => '<svg width="17" height="18" viewBox="0 0 17 18" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M8.5 16.5C11.9938 16.5 14.9296 14.111 15.7631 10.8775M8.5 16.5C5.00619 16.5 2.07035 14.111 1.23692 10.8775M8.5 16.5C10.5711 16.5 12.25 13.1421 12.25 9C12.25 4.85786 10.5711 1.5 8.5 1.5M8.5 16.5C6.42893 16.5 4.75 13.1421 4.75 9C4.75 4.85786 6.42893 1.5 8.5 1.5M8.5 1.5C11.3043 1.5 13.7492 3.03912 15.0359 5.31871M8.5 1.5C5.69567 1.5 3.25083 3.03912 1.9641 5.31871M15.0359 5.31871C13.2831 6.83366 10.9986 7.75 8.5 7.75C6.00143 7.75 3.71689 6.83366 1.9641 5.31871M15.0359 5.31871C15.6497 6.40619 16 7.66219 16 9C16 9.64832 15.9177 10.2774 15.7631 10.8775M15.7631 10.8775C13.6111 12.0706 11.1349 12.75 8.5 12.75C5.86509 12.75 3.38885 12.0706 1.23692 10.8775M1.23692 10.8775C1.08226 10.2774 1 9.64832 1 9C1 7.66219 1.35027 6.40619 1.9641 5.31871" stroke="' . esc_attr($icon_color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
    </svg>'
];

// Get contact fields based on settings
$contact_fields = [];

// Email field
if ( in_array( 'all', $show_fields ) || in_array( 'email', $show_fields ) ) {
    if ( ! empty( $user->user_email ) ) {
        $contact_fields[] = [
            'type' => 'email',
            'label' => __( 'Email', 'wpuf-pro' ),
            'value' => $user->user_email,
            'icon' => $field_icons['email']
        ];
    }
}

// Website field
if ( in_array( 'all', $show_fields ) || in_array( 'website', $show_fields ) ) {
    if ( ! empty( $user->user_url ) ) {
        $contact_fields[] = [
            'type' => 'website',
            'label' => __( 'Website', 'wpuf-pro' ),
            'value' => $user->user_url,
            'icon' => $field_icons['website']
        ];
    }
}

// Get alignment class
$align_class = '';
if ( isset( $attributes['align'] ) && $attributes['align'] ) {
    $align_class = ' has-text-align-' . esc_attr( $attributes['align'] );
}

// Get block wrapper attributes (includes styling from block editor)
$wrapper_attributes = wpuf_get_block_wrapper_attributes( $attributes, 'contact', [
    'class' => 'wpuf-user-contact-info wpuf-layout-' . esc_attr( $layout_style ) . $align_class,
    'data-user-id' => $user->ID,
] );

?>
<div <?php echo $wrapper_attributes; ?>>
    <?php if ( ! empty( $contact_fields ) ) : ?>
        <?php foreach ( $contact_fields as $field ) : ?>
            <div class="wpuf-contact-field wpuf-field-<?php echo esc_attr( $field['type'] ); ?>">
                <?php if ( $show_icons ) : ?>
                    <span class="wpuf-field-icon wpuf-icon-<?php echo esc_attr( $icon_size ); ?>">
                            <?php echo $field['icon']; ?>
                        </span>
                <?php endif; ?>

                <?php if ( $show_labels ) : ?>
                    <span class="wpuf-field-label">
                            <?php echo esc_html( $field['label'] ); ?>:
                        </span>
                <?php endif; ?>

                <span class="wpuf-field-value">
                        <?php if ( $field['type'] === 'email' && $email_link ) : ?>
                            <a href="mailto:<?php echo esc_attr( $field['value'] ); ?>">
                                <?php echo esc_html( $field['value'] ); ?>
                            </a>
                        <?php elseif ( $field['type'] === 'website' && $website_link ) : ?>
                            <a href="<?php echo esc_url( $field['value'] ); ?>" target="_blank" rel="noopener noreferrer">
                                <?php echo esc_html( $field['value'] ); ?>
                            </a>
                        <?php else : ?>
                            <?php echo esc_html( $field['value'] ); ?>
                        <?php endif; ?>
                    </span>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>
