<?php

namespace WeDevs\Wpuf\Pro\Admin;

use WeDevs\Wpuf\Admin\Forms\Admin_Form_Builder;
use WeDevs\Wpuf\Pro\Admin\Forms\Profile_Form_Template;

class Profile_Form {
    /**
     * Form settings key
     *
     * @var string
     */
    private $form_settings_key = 'wpuf_form_settings';

    /**
     * @var Profile_Form_Template
     */
    public $profile_form_template;

    public function __construct() {
        add_action( 'wpuf_load_profile_forms', [ $this, 'enqueue_assets' ] );
        add_action( 'wpuf_load_profile_forms', [ $this, 'profile_forms_builder_init' ] );

        $this->profile_form_template = new Profile_Form_Template();
    }

    /**
     * Initiate form builder for wpuf_profile post type
     *
     * @since 2.5
     *
     * @return void
     */
    public function profile_forms_builder_init() {
        if ( ! isset( $_GET['action'] ) ) {
            return;
        }

        if ( 'add-new' === $_GET['action'] && empty( $_GET['id'] ) ) {
            $form_id = wpuf_create_sample_form( 'Sample Registration Form', 'wpuf_profile', true );
            $add_new_page_url = add_query_arg( array( 'id' => $form_id ), admin_url( 'admin.php?page=wpuf-profile-forms&action=edit' ) );
            wp_redirect( $add_new_page_url );
        }

        if ( ( 'edit' === $_GET['action'] ) && ! empty( $_GET['id'] ) ) {
            // dynamic hook. hook format: wpuf-form-builder-settings-tabs-{$form_type}
            add_action( 'wpuf_form_builder_settings_tabs_profile', [ $this, 'add_settings_tabs' ] );
            // dynamic hook. hook format: wpuf-form-builder-settings-tab-contents-{$form_type}
            // add_action( 'wpuf-form-builder-settings-tab-contents-profile', [ $this, 'add_settings_tab_contents' ] );
            add_filter( 'wpuf_form_fields_section_before', [ $this, 'add_profile_field_section' ] );
            add_action( 'admin_enqueue_scripts', [ $this, 'admin_enqueue_scripts' ] );
            add_filter( 'wpuf_form_builder_js_deps', [ $this, 'js_dependencies' ] );
            add_filter( 'wpuf_form_builder_js_builder_stage_mixins', [ $this, 'js_builder_stage_mixins' ] );
            add_filter( 'wpuf_form_builder_i18n', [ $this, 'i18n' ] );
            do_action( 'wpuf_form_builder_init_type_wpuf_profile' );
            $settings = [
                'form_type'         => 'profile',
                'post_type'         => 'wpuf_profile',
                'post_id'           => sanitize_key( wp_unslash( $_GET['id'] ) ),
                'form_settings_key' => $this->form_settings_key,
                'shortcodes'        => [
                    [
                        'name' => 'wpuf_profile',
                        'type' => 'registration',
                    ],
                    [
                        'name' => 'wpuf_profile',
                        'type' => 'profile',
                    ],
                ],
            ];
            new Admin_Form_Builder( $settings );
        }
    }


    /**
     * Add settings tabs
     *
     * @since 2.5
     *
     * @return void
     */
    public function add_settings_tabs() {
        $form_settings = WPUF_PRO_ROOT . '/admin/form-builder/views/profile-form-settings.php';

        wpuf_require_once( $form_settings );

        do_action( 'wpuf_profile_form_tab' );
    }

    /**
     * Add settings tabs
     *
     * @since 2.5
     *
     * @return void
     */
    public function add_settings_tab_contents() {
        ?>

        <div id="wpuf-metabox-settings" class="group">
            <?php // $this->form_settings_general(); ?>
        </div>
        <div id="wpuf-metabox-settings-notification" class="group">
            <?php // $this->form_settings_notification(); ?>
        </div>
        <div id="wpuf-metabox-settings-reg-display-settings" class="group">
            <?php // $this->form_settings_reg_display_settings(); ?>
        </div>
        <div id="wpuf-metabox-settings-registration" class="group">
            <?php // $this->form_settings_registration(); ?>
        </div>
        <div id="wpuf-metabox-settings-profile" class="group">
            <?php // $this->form_settings_profile(); ?>
        </div>

        <?php do_action( 'wpuf_profile_form_tab_content' ); ?>

        <?php
    }

    /**
     * Admin script form wpuf_forms form builder
     *
     * @since 2.5
     *
     * @return void
     */
    public function admin_enqueue_scripts() {
        wp_enqueue_script( 'wpuf-vue' );
        wp_enqueue_script( 'wpuf-vuex' );
        wp_enqueue_script( 'wpuf-form-builder-wpuf-profile' );
    }

    /**
     * i18n strings specially for Post Forms
     *
     * @since 2.5
     *
     * @param array $i18n
     *
     * @return array
     */
    public function i18n( $i18n ) {
        return array_merge(
            $i18n, array(
                'email_needed' => __( 'Profile Forms must have Email field', 'wpuf-pro' ),
            )
        );
    }

    /**
     * Add dependencies to form builder script
     *
     * @since 2.5
     *
     * @param array $deps
     *
     * @return array
     */
    public function js_dependencies( $deps ) {
        array_push( $deps, 'wpuf-form-builder-wpuf-profile' );

        return $deps;
    }

    /**
     * Add post fields in form builder
     *
     * @since 2.5
     *
     * @return array
     */
    public function add_profile_field_section() {
        $profile_fields = apply_filters(
            'wpuf-form-builder-wp_profile-fields-section-post-fields', array(
                'user_login',
                'first_name',
                'last_name',
                'display_name',
                'nickname',
                'user_email',
                'user_url',
                'user_bio',
                'password',
                'avatar',
                'profile_photo',
            )
        );

        return [
            [
                'title'  => __( 'Profile Fields', 'wpuf-pro' ),
                'id'     => 'profile-fields',
                'fields' => $profile_fields,
            ],
        ];
    }

    /**
     * Add mixins to form builder builder stage component
     *
     * @since 2.5
     *
     * @param array $mixins
     *
     * @return array
     */
    public function js_builder_stage_mixins( $mixins ) {
        array_push( $mixins, 'wpuf_forms_mixin_builder_stage' );

        return $mixins;
    }

    /**
     * Load CSS and JS needed for the profile forms
     *
     * @since 4.0.0
     *
     * @return void
     */
    public function enqueue_assets() {
        wp_enqueue_style( 'wpuf-frontend-forms' );
        wp_enqueue_style( 'wpuf-registration-forms' );
        wp_enqueue_style( 'wpuf-admin-form-builder-pro' );
        wp_enqueue_style( 'wpuf-admin-form-builder' );
        wp_enqueue_script( 'wpuf-registration-forms' );
        wp_enqueue_script( 'wpuf-form-builder-wpuf-forms-pro' );

        if ( method_exists( 'WeDevs\Wpuf\Admin\Menu', 'load_headway_badge' ) ) {
            add_action( 'admin_footer', [ wpuf()->admin->menu, 'load_headway_badge' ] );
        }
    }

    /**
     * Displays settings on registration form builder
     *
     * @since 2.3.2
     *
     * @return void
     */
    public function form_settings_general() {
        global $post;

        $form_settings = wpuf_get_form_settings( $post->ID );

        $role_selected = isset( $form_settings['role'] ) ? $form_settings['role'] : 'subscriber';

        // Multisteps
        $is_multistep_enabled    = isset( $form_settings['enable_multistep'] ) ? $form_settings['enable_multistep'] : '';
        $multistep_progress_type = isset( $form_settings['multistep_progressbar_type'] ) ? $form_settings['multistep_progressbar_type'] : 'step_by_step';

        $ms_ac_txt_color         = isset( $form_settings['ms_ac_txt_color'] ) ? $form_settings['ms_ac_txt_color'] : '#ffffff';
        $ms_active_bgcolor       = isset( $form_settings['ms_active_bgcolor'] ) ? $form_settings['ms_active_bgcolor'] : '#00a0d2';
        $ms_bgcolor              = isset( $form_settings['ms_bgcolor'] ) ? $form_settings['ms_bgcolor'] : '#E4E4E4';
        ?>
        <table class="form-table">

            <tr class="wpuf-post-type">
                <th><?php _e( 'User Role', 'wpuf-pro' ); ?></th>
                <td>
                    <select name="wpuf_settings[role]">
                        <?php
                        $user_roles = wpuf_get_user_roles();
                        foreach ( $user_roles as $role => $label ) {
                            printf( '<option value="%s"%s>%s</option>', $role, selected( $role_selected, $role, false ), $label );
                        }
                        ?>
                    </select>

                    <p class="description"><?php _e( 'Assign the default role for users upon completing registration.', 'wpuf-pro' ); ?></p>
                </td>
            </tr>
            <tr class="wpuf_enable_multistep_section">
                <th><?php _e( 'Enable Multistep', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="wpuf_settings[enable_multistep]" value="yes" <?php checked( $is_multistep_enabled, 'yes' ); ?> />
                        <?php _e( 'Enable Multistep', 'wpuf-pro' ); ?>
                    </label>

                    <p class="description"><?php echo __( 'If checked, form will be displayed in frontend in multiple steps', 'wpuf-pro' ); ?></p>
                </td>
            </tr>
            <tr class="wpuf_multistep_content">
                <td colspan="2" style="padding: 15px 0;">
                    <h3><?php _e( 'Multistep Form Settings', 'wpuf-pro' ); ?></h3>
                </td>
            </tr>
            <tr class="wpuf_multistep_progress_type wpuf_multistep_content">
                <th><?php _e( 'Multistep Progressbar Type', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <select name="wpuf_settings[multistep_progressbar_type]">
                            <option value="progressive" <?php echo $multistep_progress_type == 'progressive' ? 'selected' : ''; ?>><?php _e( 'Progressbar', 'wpuf-pro' ); ?></option>
                            <option value="step_by_step" <?php echo $multistep_progress_type == 'step_by_step' ? 'selected' : ''; ?>><?php _e( 'Step by Step', 'wpuf-pro' ); ?></option>
                        </select>
                    </label>


                    <p class="description"><?php echo __( 'Choose how you want the progressbar', 'wpuf-pro' ); ?></p>
                </td>
            </tr>

            <tr class="wpuf_multistep_content">
                <th><?php _e( 'Active Text Color', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="text" name="wpuf_settings[ms_ac_txt_color]" class="wpuf-ms-color" value="<?php echo $ms_ac_txt_color; ?>"  />

                    </label>

                    <p class="description"> <?php _e( 'Text color for active step.', 'wpuf-pro' ); ?></p>
                </td>
            </tr>
            <tr class="wpuf_multistep_content">
                <th><?php _e( 'Active Background Color', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="text" name="wpuf_settings[ms_active_bgcolor]" class="wpuf-ms-color" value="<?php echo $ms_active_bgcolor; ?>"  />

                    </label>

                    <p class="description"> <?php _e( 'Background color for progressbar or active step.', 'wpuf-pro' ); ?></p>
                </td>
            </tr>
            <tr class="wpuf_multistep_content">
                <th><?php _e( 'Background Color', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="text" name="wpuf_settings[ms_bgcolor]" class="wpuf-ms-color" value="<?php echo $ms_bgcolor; ?>"  />

                    </label>

                    <p class="description"> <?php _e( 'Background color for normal steps.', 'wpuf-pro' ); ?></p>
                </td>
            </tr>

            <?php do_action( 'wpuf_profile_setting', $form_settings, $post ); ?>
        </table>
        <?php
    }

    /**
     * Displays settings on registration form builder
     *
     * @since 2.3.2
     *
     * @return void
     */
    public function form_settings_notification() {
        global $post;

        $blogname                = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
        $form_settings           = wpuf_get_form_settings( $post->ID );
        $user_notification       = isset( $form_settings['user_notification'] ) ? $form_settings['user_notification'] : 'on';
        $admin_notification      = isset( $form_settings['admin_notification'] ) ? $form_settings['admin_notification'] : 'on';
        $notification_type       = isset( $form_settings['notification_type'] ) ? $form_settings['notification_type'] : 'email_verification';

        $verification_mail_body  = "Congrats! You are Successfully registered to {blogname}\r\n\r\n";
        $verification_mail_body .= "To activate your account, please click the link below\r\n";
        $verification_mail_body .= "{activation_link}\r\n\r\n";
        $verification_mail_body .= 'Thanks!';

        $verification_subject    = isset( $form_settings['notification']['verification_subject'] ) ? $form_settings['notification']['verification_subject'] : __( 'Account Activation', 'wpuf-pro' );
        $verification_body       = isset( $form_settings['notification']['verification_body'] ) ? $form_settings['notification']['verification_body'] : $verification_mail_body;

        $welcome_mail_body       = "Hi {username},\r\n\r\n";
        $welcome_mail_body      .= 'Congrats! You are Successfully registered to ' . $blogname . "\r\n\r\n";
        $welcome_mail_body      .= 'Thanks';

        $welcome_email_subject   = isset( $form_settings['notification']['welcome_email_subject'] ) ? $form_settings['notification']['welcome_email_subject'] : __( 'Thank you for registering', 'wpuf-pro' );
        $welcome_email_body      = isset( $form_settings['notification']['welcome_email_body'] ) ? $form_settings['notification']['welcome_email_body'] : $welcome_mail_body;

        $pending_user_admin_notification       = "Username: {username} ({user_email}) has requested a username.\r\n\r\n";
        $pending_user_admin_notification      .= "To approve or deny this user access go to {pending_users}\r\n\r\n";
        $pending_user_admin_notification      .= 'Thanks';

        $approved_user_admin_notification       = "Username: {username} ({user_email}) has requested a username.\r\n\r\n";
        $approved_user_admin_notification      .= "To pending or deny this user access go to {approved_users}\r\n\r\n";
        $approved_user_admin_notification      .= 'Thanks';

        $admin_email_subject                   = isset( $form_settings['admin_email_subject'] ) ? $form_settings['admin_email_subject'] : __( 'New user registered on your site', 'wpuf-pro' );
        $pending_user_admin_notification       = isset( $form_settings['admin_email_body']['user_status_pending'] ) ? $form_settings['admin_email_body']['user_status_pending'] : $pending_user_admin_notification;
        $approved_user_admin_notification      = isset( $form_settings['notification']['admin_email_body']['user_status_approved'] ) ? $form_settings['notification']['admin_email_body']['user_status_approved'] : $approved_user_admin_notification;
        $editor_settings                       = [
            'textarea_rows' => 10,
            'teeny'         => true,
            'tinymce'       => [
                'autoresize_min_height' => 100,
                'wp_autoresize_on'      => true,
                'plugins'               => 'wpautoresize',
            ],
        ];

        ?>

        <h3><?php _e( 'New User Notification', 'wpuf-pro' ); ?></h3>
        <table class="form-table">
            <tr>
                <th><?php _e( 'Enable user notification', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="hidden" name="wpuf_settings[user_notification]" value="off">
                        <input type="checkbox" name="wpuf_settings[user_notification]" value="on"<?php checked( $user_notification, 'on' ); ?>>
                        <?php _e( 'Enable user notification', 'wpuf-pro' ); ?>
                    </label>
                </td>
            </tr>

            <tr class="wpuf-registration-notification-type">
                <th><?php _e( 'Notification Type', 'wpuf-pro' ); ?></th>
                <td>
                    <label for="wpuf_settings[notification_type]">
                        <input type="hidden" name="wpuf_settings[notification_type]" value="email_verification">

                        <input type="radio" id="notification_type_verification" name="wpuf_settings[notification_type]" value="email_verification" <?php checked( $notification_type, 'email_verification' ); ?>>
                        <?php _e( 'Email Verification', 'wpuf-pro' ); ?>

                        <input type="radio" id="notification_type_welcome_email" name="wpuf_settings[notification_type]" value="welcome_email" <?php checked( $notification_type, 'welcome_email' ); ?>>
                        <?php _e( 'Welcome Email', 'wpuf-pro' ); ?>
                    </label>

                    <p class="description"><?php _e( 'An email will be sent to the user after registration.', 'wpuf-pro' ); ?></p>
                </td>
            </tr>

            <tr class="wpuf-email-verification-settings-fields">
                <th><?php _e( 'Confirmation mail subject', 'wpuf-pro' ); ?></th>
                <td><input type="text" name="wpuf_settings[notification][verification_subject]" class="regular-text" value="<?php echo esc_attr( $verification_subject ); ?>"></td>
            </tr>

            <tr class="wpuf-email-verification-settings-fields">
                <th><?php _e( 'Confirmation mail body', 'wpuf-pro' ); ?></th>
                <td>
                    <?php wp_editor( $verification_body, 'wpuf_verification_body_' . $post->ID, array_merge( [ 'textarea_name' => 'wpuf_settings[notification][verification_body]' ], $editor_settings ) ); ?>
                    <p class="description"><?php _e( 'You may use: {username}, {blogname}, {activation_link}', 'wpuf-pro' ); ?></p>
                    <p class="description"><?php _e( 'You also may use meta key: ', 'wpuf-pro' ); ?> {{meta_field_key}} </p>
                </td>
            </tr>

            <tr class="wpuf-welcome-email-settings-fields">
                <th><?php _e( 'Welcome mail subject', 'wpuf-pro' ); ?></th>
                <td><input type="text" name="wpuf_settings[notification][welcome_email_subject]" class="regular-text" value="<?php echo esc_attr( $welcome_email_subject ); ?>"></td>
            </tr>

            <tr class="wpuf-welcome-email-settings-fields">
                <th><?php _e( 'Welcome mail body', 'wpuf-pro' ); ?></th>
                <td>
                    <?php wp_editor( $welcome_email_body, 'wpuf_welcome_email_body_' . $post->ID, array_merge( [ 'textarea_name' => 'wpuf_settings[notification][welcome_email_body]' ], $editor_settings ) ); ?>
                    <p class="description"><?php _e( 'You may use: {username}, {user_email}, {display_name}, {user_status}, {pending_users}, {approved_users} ,{denied_users}', 'wpuf-pro' ); ?></p>
                    <p class="description"><?php _e( 'You also may use meta key: ', 'wpuf-pro' ); ?> {{meta_field_key}}, </p>
                </td>
            </tr>

            <?php do_action( 'wpuf_profile_setting_notification_user', $form_settings, $post ); ?>
        </table>

        <h3><?php _e( 'Admin Notification', 'wpuf-pro' ); ?></h3>
        <table class="form-table">
            <tr>
                <th><?php _e( 'Enable admin notification', 'wpuf-pro' ); ?></th>
                <td>
                    <label>
                        <input type="hidden" name="wpuf_settings[admin_notification]" value="off">
                        <input type="checkbox" name="wpuf_settings[admin_notification]" value="on"<?php checked( $admin_notification, 'on' ); ?>>
                        <?php _e( 'Enable admin notification', 'wpuf-pro' ); ?>
                    </label>
                </td>
            </tr>

            <tr class="wpuf-new-user-admin-notification subject">
                <th><?php _e( 'Subject', 'wpuf-pro' ); ?></th>
                <td><input type="text" name="wpuf_settings[notification][admin_email_subject]" class="regular-text" value="<?php echo esc_attr( $admin_email_subject ); ?>"></td>
            </tr>

            <tr class="wpuf-new-user-admin-notification content">
                <th><?php esc_html_e( 'Message', 'wpuf-pro' ); ?></th>
                <td>
                    <textarea rows="8" cols="60" id="wpuf_pending_user_admin_notification" name="wpuf_settings[notification][admin_email_body][user_status_pending]"><?php echo esc_textarea( $pending_user_admin_notification ); ?></textarea>
                    <textarea rows="8" cols="60" id="wpuf_approved_user_admin_notification" name="wpuf_settings[notification][admin_email_body][user_status_approved]"><?php echo esc_textarea( $approved_user_admin_notification ); ?></textarea>
                    <p class="description">
                        <?php
                        esc_html_e( 'You may use: {username}, {user_email}, {display_name}, {user_status}, {pending_users}, {approved_users}, {denied_users}', 'wpuf-pro' );
                        ?>
                    </p>
                </td>
            </tr>

            <?php do_action( 'wpuf_profile_setting_notification_admin', $form_settings, $post ); ?>
        </table>
        <?php
    }

    public function form_settings_reg_display_settings() {
        global $post;

        $form_settings  = wpuf_get_form_settings( get_the_ID() );
        $label_position = isset( $form_settings['label_position'] ) ? $form_settings['label_position'] : 'left';
        $theme_css      = isset( $form_settings['use_theme_css'] ) ? $form_settings['use_theme_css'] : 'wpuf-style';
        $form_layout    = isset( $form_settings['profile_form_layout'] ) ? $form_settings['profile_form_layout'] : 'layout1';
        ?>
        <table class="form-table">
            <tr class="wpuf-pro-label-position">
                <th><?php _e( 'Label Position', 'wpuf-pro' ); ?></th>
                <td>
                    <select name="wpuf_settings[label_position]">
                        <?php
                        $positions = array(
                            'above'  => __( 'Above Element', 'wpuf-pro' ),
                            'left'   => __( 'Left of Element', 'wpuf-pro' ),
                            'right'  => __( 'Right of Element', 'wpuf-pro' ),
                            'hidden' => __( 'Hidden', 'wpuf-pro' ),
                        );

                        foreach ( $positions as $to => $label ) {
                            printf( '<option value="%s"%s>%s</option>', $to, selected( $label_position, $to, false ), $label );
                        }
                        ?>
                    </select>

                    <p class="description">
                        <?php _e( 'Customize the position of form labels for improved user navigation and clarity', 'wpuf-pro' ); ?>
                    </p>
                </td>
            </tr>

            <tr class="wpuf-override-theme-css">
                <th><?php _e( 'Use Theme CSS', 'wpuf-pro' ); ?></th>
                <td>
                    <select name="wpuf_settings[use_theme_css]">
                        <?php
                        $options = array(
                            'wpuf-style'         => __( 'No', 'wpuf-pro' ),
                            'wpuf-theme-style'   => __( 'Yes', 'wpuf-pro' ),
                        );

                        foreach ( $options as $to => $label ) {
                            printf( '<option value="%s"%s>%s</option>', $to, selected( $theme_css, $to, false ), $label );
                        }
                        ?>
                    </select>

                    <p class="description">
                        <?php _e( 'Selecting "Yes" will use your theme\'s style for form fields.', 'wpuf-pro' ); ?>
                    </p>
                </td>
            </tr>

            <tr class="wpuf-form-layouts">
                <th><?php _e( 'Form Style', 'wpuf-pro' ); ?></th>
                <td>
                    <ul>
                        <?php
                        $layouts = [
                            'layout1' => WPUF_PRO_ASSET_URI . '/images/forms/layout1.png',
                            'layout2' => WPUF_PRO_ASSET_URI . '/images/forms/layout2.png',
                            'layout3' => WPUF_PRO_ASSET_URI . '/images/forms/layout3.png',
                            'layout4' => WPUF_PRO_ASSET_URI . '/images/forms/layout4.png',
                            'layout5' => WPUF_PRO_ASSET_URI . '/images/forms/layout5.png',
                        ];

                        foreach ( $layouts as $key => $image ) {
                            $active = '';

                            if ( $key == $form_layout ) {
                                $active = 'active';
                            }

                            $output  = '<li class="' . $active . '">';
                            $output .= '<input type="radio" name="wpuf_settings[profile_form_layout]" value="' . $key . '" ' . checked( $form_layout, $key, false ) . '>';
                            $output .= '<img src="' . $image . '" alt="">';
                            $output .= '</li>';

                            echo $output;
                        }
                        ?>
                    </ul>
                </td>
            </tr>
        </table>

        <?php
    }

    /**
     * Adds registration redirect tab content
     *
     * @since 2.3.2
     *
     * @return void
     */
    public function form_settings_registration() {
        global $post;

        $form_settings = wpuf_get_form_settings( $post->ID );

        $redirect_to = isset( $form_settings['reg_redirect_to'] ) ? $form_settings['reg_redirect_to'] : 'post';

        if ( ! isset( $form_settings['reg_redirect_to'] ) ) {
            $redirect_to = isset( $form_settings['reg_redirect_to'] ) ? $form_settings['reg_redirect_to'] : 'post';
        }

        $message                = isset( $form_settings['message'] ) ? $form_settings['message'] : __( 'Registration successful', 'wpuf-pro' );
        $page_id                = isset( $form_settings['reg_page_id'] ) ? $form_settings['reg_page_id'] : 0;
        $url                    = isset( $form_settings['registration_url'] ) ? $form_settings['registration_url'] : '';

        if ( ! isset( $form_settings['registration_url'] ) ) {
            $url = isset( $form_settings['url'] ) ? $form_settings['url'] : '';
        }

        $submit_text            = isset( $form_settings['submit_text'] ) ? $form_settings['submit_text'] : __( 'Register', 'wpuf-pro' );
        $ms_ac_txt_color        = isset( $form_settings['ms_ac_txt_color'] ) ? $form_settings['ms_ac_txt_color'] : '#ffffff';
        $ms_active_bgcolor      = isset( $form_settings['ms_active_bgcolor'] ) ? $form_settings['ms_active_bgcolor'] : '#00a0d2';
        $ms_bgcolor             = isset( $form_settings['ms_bgcolor'] ) ? $form_settings['ms_bgcolor'] : '#E4E4E4';
        ?>
        <table class="form-table">
            <tr class="wpuf-reg-redirect-to">
                <th><?php _e( 'Redirect To', 'wpuf-pro' ); ?></th>
                <td>
                    <select name="wpuf_settings[reg_redirect_to]">
                        <?php
                        $redirect_options = array(
                            'same' => __( 'Same Page', 'wpuf-pro' ),
                            'page' => __( 'To a page', 'wpuf-pro' ),
                            'url' => __( 'To a custom URL', 'wpuf-pro' ),
                        );

                        foreach ( $redirect_options as $to => $label ) {
                            printf( '<option value="%s"%s>%s</option>', $to, selected( $redirect_to, $to, false ), $label );
                        }
                        ?>
                    </select>
                    <div class="description">
                        <?php _e( 'After successfull submit, where the page will redirect to', 'wpuf-pro' ); ?>
                    </div>
                </td>
            </tr>

            <tr class="wpuf-same-page">
                <th><?php _e( 'Registration success message', 'wpuf-pro' ); ?></th>
                <td>
                    <textarea rows="3" cols="40" name="wpuf_settings[message]"><?php echo esc_textarea( $message ); ?></textarea>
                </td>
            </tr>

            <tr class="wpuf-page-id">
                <th><?php _e( 'Page', 'wpuf-pro' ); ?></th>
                <td>
                    <select name="wpuf_settings[reg_page_id]">
                        <?php
                        $pages = get_posts(
                            array(
                                'numberposts' => -1,
                                'post_type' => 'page',
                            )
                        );

                        foreach ( $pages as $page ) {
                            printf( '<option value="%s"%s>%s</option>', $page->ID, selected( $page_id, $page->ID, false ), esc_attr( $page->post_title ) );
                        }
                        ?>
                    </select>
                </td>
            </tr>

            <tr class="wpuf-url">
                <th><?php _e( 'Custom URL', 'wpuf-pro' ); ?></th>
                <td>
                    <input type="url" name="wpuf_settings[registration_url]" value="<?php echo esc_attr( $url ); ?>">
                </td>
            </tr>

            <tr class="wpuf-submit-text">
                <th><?php _e( 'Submit Button text', 'wpuf-pro' ); ?></th>
                <td>
                    <input type="text" name="wpuf_settings[submit_text]" value="<?php echo esc_attr( $submit_text ); ?>">
                </td>
            </tr>

            <?php do_action( 'wpuf_profile_setting_reg', $form_settings, $post ); ?>
        </table>
        <?php
    }

    /**
     * Adds profile update redirect tab content
     *
     * @since 2.3.2
     *
     * @return void
     */

    public function form_settings_profile() {
        global $post;

        $form_settings = wpuf_get_form_settings( $post->ID );

        $redirect_to = isset( $form_settings['profile_redirect_to'] ) ? $form_settings['profile_redirect_to'] : 'post';

        if ( ! isset( $form_settings['reg_redirect_to'] ) ) {
            $redirect_to = isset( $form_settings['profile_redirect_to'] ) ? $form_settings['profile_redirect_to'] : 'post';
        }

        $update_message          = isset( $form_settings['update_message'] ) ? $form_settings['update_message'] : __( 'Profile updated successfully', 'wpuf-pro' );
        $page_id                 = isset( $form_settings['profile_page_id'] ) ? $form_settings['profile_page_id'] : 0;
        $url                     = isset( $form_settings['profile_url'] ) ? $form_settings['profile_url'] : '';

        if ( ! isset( $form_settings['profile_url'] ) ) {
            $url = isset( $form_settings['url'] ) ? $form_settings['url'] : '';
        }

        $update_text = isset( $form_settings['update_text'] ) ? $form_settings['update_text'] : __( 'Update Profile', 'wpuf-pro' );

        $ms_ac_txt_color         = isset( $form_settings['ms_ac_txt_color'] ) ? $form_settings['ms_ac_txt_color'] : '#ffffff';
        $ms_active_bgcolor       = isset( $form_settings['ms_active_bgcolor'] ) ? $form_settings['ms_active_bgcolor'] : '#00a0d2';
        $ms_bgcolor              = isset( $form_settings['ms_bgcolor'] ) ? $form_settings['ms_bgcolor'] : '#E4E4E4';
        ?>
        <table class="form-table">
            <tr class="wpuf-profile-redirect-to">
                <th><?php _e( 'Redirect To', 'wpuf-pro' ); ?></th>
                <td>
                    <select name="wpuf_settings[profile_redirect_to]">
                        <?php
                        $redirect_options = array(
                            'same' => __( 'Same Page', 'wpuf-pro' ),
                            'page' => __( 'To a page', 'wpuf-pro' ),
                            'url'  => __( 'To a custom URL', 'wpuf-pro' ),
                        );

                        foreach ( $redirect_options as $to => $label ) {
                            printf( '<option value="%s"%s>%s</option>', $to, selected( $redirect_to, $to, false ), $label );
                        }
                        ?>
                    </select>
                    <div class="description">
                        <?php _e( 'After successfull submit, where the page will redirect to', 'wpuf-pro' ); ?>
                    </div>
                </td>
            </tr>

            <tr class="wpuf-same-page">
                <th><?php _e( 'Update profile message', 'wpuf-pro' ); ?></th>
                <td>
                    <textarea rows="3" cols="40" name="wpuf_settings[update_message]"><?php echo esc_textarea( $update_message ); ?></textarea>
                </td>
            </tr>

            <tr class="wpuf-page-id">
                <th><?php _e( 'Page', 'wpuf-pro' ); ?></th>
                <td>
                    <select name="wpuf_settings[profile_page_id]">
                        <?php
                        $pages = get_posts(
                            array(
                                'numberposts' => -1,
                                'post_type' => 'page',
                            )
                        );

                        foreach ( $pages as $page ) {
                            printf( '<option value="%s"%s>%s</option>', $page->ID, selected( $page_id, $page->ID, false ), esc_attr( $page->post_title ) );
                        }
                        ?>
                    </select>
                </td>
            </tr>

            <tr class="wpuf-url">
                <th><?php _e( 'Custom URL', 'wpuf-pro' ); ?></th>
                <td>
                    <input type="url" name="wpuf_settings[profile_url]" value="<?php echo esc_attr( $url ); ?>">
                </td>
            </tr>

            <tr class="wpuf-update-text">
                <th><?php _e( 'Update Button text', 'wpuf-pro' ); ?></th>
                <td>
                    <input type="text" name="wpuf_settings[update_text]" value="<?php echo esc_attr( $update_text ); ?>">
                </td>
            </tr>

            <?php do_action( 'wpuf_profile_setting_profile', $form_settings, $post ); ?>
        </table>
        <?php
    }
}
