<?php

namespace WeDevs\Wpuf\Pro\Fields;

use WeDevs\Wpuf\Pro\Fields\Traits\Pricing_Field_Trait;

/**
 * Cart Total Field Class
 *
 * Displays calculated total from pricing fields
 *
 * @since 4.2.2
 */
class Field_Cart_Total extends Pro_Field_Contract {

    use Pricing_Field_Trait;

    /**
     * Constructor
     *
     * @since 4.2.2
     */
    public function __construct() {
        $this->name       = __( 'Cart Total', 'wpuf-pro' );
        $this->input_type = 'cart_total';
        $this->icon       = 'receipt-percent';
    }

    /**
     * Render the Cart Total field
     *
     * @since 4.2.2
     *
     * @param array  $field_settings Field settings.
     * @param int    $form_id        Form ID.
     * @param string $type           Form type.
     * @param int    $post_id        Post ID.
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        $show_summary = isset( $field_settings['show_summary'] ) && $field_settings['show_summary'] === 'yes';

        $this->field_print_label( $field_settings, $form_id ); ?>

        <div class="wpuf-fields">
            <?php if ( $show_summary ) : ?>
                <div class="wpuf-cart-summary" style="margin-bottom: 15px;">
                    <table class="wpuf-cart-table" style="width: 100%; border-collapse: collapse;">
                        <thead>
                            <tr style="border-bottom: 2px solid #ddd;">
                                <th style="text-align: left; padding: 8px;"><?php _e( 'Item', 'wpuf-pro' ); ?></th>
                                <th style="text-align: center; padding: 8px;"><?php _e( 'Quantity', 'wpuf-pro' ); ?></th>
                                <th style="text-align: right; padding: 8px;"><?php _e( 'Price', 'wpuf-pro' ); ?></th>
                            </tr>
                        </thead>
                        <tbody class="wpuf-cart-items">
                            <tr class="wpuf-cart-empty">
                                <td colspan="3" style="text-align: center; padding: 15px; color: #999;">
                                    <?php _e( 'No items selected', 'wpuf-pro' ); ?>
                                </td>
                            </tr>
                        </tbody>
                        <tfoot>
                            <tr style="border-top: 2px solid #ddd; font-weight: bold;">
                                <td colspan="2" style="text-align: right; padding: 10px;"><?php _e( 'TOTAL', 'wpuf-pro' ); ?></td>
                                <td style="text-align: right; padding: 10px;">
                                    <span class="wpuf-cart-total-amount">0.00</span>
                                </td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            <?php else : ?>
                <div class="wpuf-cart-total-display" style="font-size: 18px; font-weight: bold;">
                    <span class="wpuf-cart-total-label"><?php echo esc_html( $field_settings['label'] ); ?>:</span>
                    <span class="wpuf-cart-total-amount">0.00</span>
                </div>
            <?php endif; ?>

            <input type="hidden" name="<?php echo esc_attr( $field_settings['name'] ); ?>" class="wpuf-cart-total-hidden" value="0" />

            <?php $this->help_text( $field_settings ); ?>
        </div>

        <?php $this->after_field_print_label();
    }

    /**
     * Get field options setting
     *
     * @since 4.2.2
     *
     * @return array Field options.
     */
    public function get_options_settings() {
        $default_options = $this->get_default_option_settings( false, [ 'width' ] );

        $cart_options = [
            [
                'name'          => 'show_summary',
                'title'         => __( 'Show Summary Table', 'wpuf-pro' ),
                'type'          => 'radio',
                'options'       => [
                    'yes'   => __( 'Yes', 'wpuf-pro' ),
                    'no'    => __( 'No', 'wpuf-pro' ),
                ],
                'default'       => 'yes',
                'inline'        => true,
                'section'       => 'advanced',
                'priority'      => 10,
                'help_text'     => __( 'Show detailed item summary table', 'wpuf-pro' ),
            ],
        ];

        return array_merge( $default_options, $cart_options );
    }

    /**
     * Get the field props
     *
     * @since 4.2.2
     *
     * @return array Field properties.
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'        => 'cart_total',
            'label'             => __( 'Total', 'wpuf-pro' ),
            'is_meta'           => 'yes',
            'show_summary'      => 'yes',
            'id'                => 0,
            'is_new'            => true,
            'show_in_post'      => 'yes',
            'hide_field_label'  => 'no',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Prepare entry
     *
     * @since 4.2.2
     *
     * @param array $field Field settings.
     *
     * @return string Formatted total amount.
     */
    public function prepare_entry( $field ) {
        check_ajax_referer( 'wpuf_form_add' );

        $total = isset( $_POST[$field['name']] ) ? floatval( wp_unslash( $_POST[$field['name']] ) ) : 0;

        return number_format( $total, 2, '.', '' );
    }

    /**
     * Check if field is full width
     *
     * @since 4.2.2
     *
     * @return bool True if field is full width.
     */
    public function is_full_width() {
        return true;
    }
}
