<?php

namespace WeDevs\Wpuf\Pro\Fields;

use WeDevs\Wpuf\Fields\Form_Field_Dropdown;
use WeDevs\Wpuf\Pro\Fields\Traits\Pricing_Field_Trait;

/**
 * Pricing Dropdown Field Class
 *
 * Dropdown with price values for each option
 *
 * @since 4.2.2
 */
class Field_Pricing_Dropdown extends Form_Field_Dropdown {

    use Pricing_Field_Trait;

    public function __construct() {
        $this->name       = __( 'Pricing Dropdown', 'wpuf-pro' );
        $this->input_type = 'pricing_dropdown';
        $this->icon       = 'currency-dollar';
    }

    /**
     * Render the Pricing Dropdown field
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        if ( isset( $post_id ) && '0' !== $post_id ) {
            $selected = $this->get_meta( $post_id, $field_settings['name'], $type );
        } else {
            $selected = isset( $field_settings['selected'] ) ? $field_settings['selected'] : '';
        }

        $enable_quantity = isset( $field_settings['enable_quantity'] ) && 'yes' === $field_settings['enable_quantity'];
        $name            = $field_settings['name'];

        $this->field_print_label( $field_settings, $form_id ); ?>

        <div class="wpuf-fields">
            <select
                class="<?php echo 'wpuf_' . esc_attr( $field_settings['name'] ) . '_' . esc_attr( $form_id ); ?>"
                id="<?php echo esc_attr( $field_settings['name'] ) . '_' . esc_attr( $form_id ); ?>"
                name="<?php echo esc_attr( $name ); ?>"
                data-required="<?php echo esc_attr( $field_settings['required'] ); ?>"
                data-type="pricing-select">

                <?php if ( ! empty( $field_settings['first'] ) ) { ?>
                    <option value="-1" data-price="0"><?php echo esc_html( $field_settings['first'] ); ?></option>
                <?php } ?>

                <?php
                if ( $field_settings['options'] && count( $field_settings['options'] ) > 0 ) {
                    foreach ( $field_settings['options'] as $value => $option ) {
                        $price = isset( $field_settings['prices'][$value] ) ? floatval( $field_settings['prices'][$value] ) : 0;
                        $current_select = selected( $selected, $value, false );
                        ?>
                        <option
                            value="<?php echo esc_attr( $value ); ?>"
                            data-price="<?php echo esc_attr( $price ); ?>"
                            <?php echo esc_attr( $current_select ); ?>>
                            <?php echo esc_html( $option . ' - ' . number_format( $price, 2 ) ); ?>
                        </option>
                        <?php
                    }
                } ?>
            </select>

            <?php if ( $enable_quantity ) : ?>
                <input
                    type="number"
                    class="wpuf-pricing-quantity"
                    name="<?php echo esc_attr( $field_settings['name'] ); ?>_qty"
                    min="1"
                    value="1"
                    style="width: 60px; margin-left: 10px;"
                />
            <?php endif; ?>

            <?php $this->help_text( $field_settings ); ?>
        </div>

        <?php $this->after_field_print_label();
    }

    /**
     * Get field options setting
     */
    public function get_options_settings() {
        $default_options = $this->get_default_option_settings();
        $pricing_options = $this->get_pricing_field_options( false, false, false );

        // Add the 'first' field for dropdown-specific "Select Text" option
        $pricing_options[] = [
            'name'          => 'first',
            'title'         => __( 'Select Text', 'wpuf-pro' ),
            'type'          => 'text',
            'section'       => 'basic',
            'priority'      => 14,
            'help_text'     => __( "First element of the select dropdown. Leave empty if not needed", 'wpuf-pro' ),
        ];

        return array_merge( $default_options, $pricing_options );
    }

    /**
     * Get the field props
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'        => 'pricing_dropdown',
            'label'             => __( 'Pricing Dropdown', 'wpuf-pro' ),
            'name'              => 'wpuf_pricing_dropdown',
            'is_meta'           => 'yes',
            'enable_quantity'   => 'no',
            'options'           => [
                'first_item'  => __( 'First Item', 'wpuf-pro' ),
                'second_item' => __( 'Second Item', 'wpuf-pro' ),
                'third_item'  => __( 'Third Item', 'wpuf-pro' ),
            ],
            'prices'            => [
                'first_item'  => '10',
                'second_item' => '25',
                'third_item'  => '50',
            ],
            'first'             => __( '- Select -', 'wpuf-pro' ),
            'id'                => 0,
            'is_new'            => true,
            'show_in_post'      => 'yes',
            'hide_field_label'  => 'no',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Prepare entry
     */
    public function prepare_entry( $field ) {
        check_ajax_referer( 'wpuf_form_add' );

        $val = isset( $_POST[$field['name']] ) ? sanitize_text_field( wp_unslash( $_POST[$field['name']] ) ) : '';

        if ( $val && isset( $field['options'][$val] ) ) {
            $label = $field['options'][$val];
            $price = isset( $field['prices'][$val] ) ? floatval( $field['prices'][$val] ) : 0;

            $qty = 1;
            if ( isset( $_POST[$field['name'] . '_qty'] ) ) {
                $qty = intval( $_POST[$field['name'] . '_qty'] );
            }

            return $label . ' (' . ( $price * $qty ) . ')';
        }

        return '';
    }

    public function is_pro() {
        return true;
    }
}
