// User Bio Edit Component
// Displays and configures user biography with various display options

import { __ } from '@wordpress/i18n';
import {
    InspectorControls,
    useBlockProps
} from '@wordpress/block-editor';
import {
    PanelBody,
    SelectControl,
    ToggleControl,
    RangeControl,
    TextControl
} from '@wordpress/components';
import { Fragment } from '@wordpress/element';

const Edit = ({ attributes, setAttributes, context }) => {
    const {
        characterLimit,
        showReadMore,
        readMoreText
    } = attributes;

    // Get user data from context prop (Query Loop style)
    	const user = context?.['wpuf-ud/userObject'] || {
        id: 0,
        display_name: __('Sample User', 'wpuf-pro'),
        user_login: 'sampleuser',
        user_email: 'sample@example.com',
        description: __('This is a sample user biography that demonstrates how the user bio block will display user information. This text shows how the biography will appear with various formatting options and character limits applied.', 'wpuf-pro'),
        first_name: 'Sample',
        last_name: 'User',
        nickname: 'sample',
        user_url: 'https://example.com',
        roles: ['subscriber']
    };

    const blockProps = useBlockProps({
        className: 'wpuf-user-bio'
    });

    // Get bio text
    const getBioText = () => {
        const bio = user.description || user.bio || __('This user has not provided a biography yet.', 'wpuf-pro');

        if (characterLimit && bio.length > characterLimit) {
            return bio.substring(0, characterLimit) + '...';
        }

        return bio;
    };

    const bioText = getBioText();
    const hasMoreText = (user.description || user.bio || '').length > characterLimit;

    return (
        <Fragment>
            <InspectorControls>
                <PanelBody title={__('Bio Settings', 'wpuf-pro')} initialOpen={true}>
                    <RangeControl
                        label={__('Character Limit', 'wpuf-pro')}
                        value={characterLimit}
                        onChange={(value) => setAttributes({ characterLimit: value })}
                        min={50}
                        max={500}
                        step={10}
                        help={__('Set to 0 for no limit', 'wpuf-pro')}
                        __next40pxDefaultSize={true}
                        __nextHasNoMarginBottom={true}
                    />

                    <ToggleControl
                        label={__('Show Read More', 'wpuf-pro')}
                        checked={showReadMore}
                        onChange={(value) => setAttributes({ showReadMore: value })}
                        __nextHasNoMarginBottom={true}
                    />

                    {showReadMore && (
                        <Fragment>
                            <TextControl
                                label={__('Read More Text', 'wpuf-pro')}
                                value={readMoreText}
                                onChange={(value) => setAttributes({ readMoreText: value })}
                                placeholder={__('Read more...', 'wpuf-pro')}
                                __next40pxDefaultSize={true}
                                __nextHasNoMarginBottom={true}
                            />
                        </Fragment>
                    )}
                </PanelBody>
            </InspectorControls>

            <div {...blockProps}>
                <div className="wpuf-bio-content">
                    {showReadMore && hasMoreText ? (
                        <>
                            {/* Truncated content (initially visible) */}
                            <div className="wpuf-bio-truncated" data-bio-state="truncated">
                                <p className="wpuf-bio-text">{bioText}</p>
                            </div>

                            {/* Full content (initially hidden) */}
                            <div className="wpuf-bio-full" data-bio-state="full" style={{display: 'none'}}>
                                <p className="wpuf-bio-text">{(user.description || user.bio || __('This user has not provided a biography yet.', 'wpuf-pro'))}</p>
                            </div>

                            {/* Read More Toggle Link */}
                            <a href="#" 
                               className="wpuf-read-more-toggle"
                               data-target={`wpuf-bio-${Math.random().toString(36).substr(2, 9)}`}
                               data-read-more-text={readMoreText || __('Read more...', 'wpuf-pro')}
                               data-show-less-text={__('Show Less', 'wpuf-pro')}
                            >
                                {readMoreText || __('Read more...', 'wpuf-pro')}
                            </a>
                        </>
                    ) : (
                        /* Display content without read more functionality */
                        <p className="wpuf-bio-text">{bioText}</p>
                    )}
                </div>
            </div>
        </Fragment>
    );
};

export default Edit;
