/**
 * WordPress dependencies
 */
import {
	PanelBody,
	SelectControl,
	RangeControl,
	ToggleControl,
	TextControl,
	CheckboxControl,
	__experimentalToolsPanel as ToolsPanel,
	__experimentalToolsPanelItem as ToolsPanelItem,
} from '@wordpress/components';
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies
 */
import { getBlockPanels, getFieldOptions, getFieldLabel, getFieldDescription } from '../../utils/schema-utils';

/**
 * Schema-driven inspector controls for User Directory block
 */
export default function SchemaInspectorControls({ attributes, setAttributes }) {
	const panels = getBlockPanels('user-directory');
	
	// Helper function to render a field based on its control type
	const renderField = (fieldName, field) => {
		const value = attributes[fieldName];
		const label = getFieldLabel(field);
		const description = getFieldDescription(field);
		const options = getFieldOptions(field);
		
		const onChange = (newValue) => {
			setAttributes({ [fieldName]: newValue });
		};
		
		switch (field.control) {
			case 'select':
				return (
					<SelectControl
						key={fieldName}
						label={label}
						help={description}
						value={value}
						options={options}
						onChange={onChange}
						__next40pxDefaultSize={true}
						__nextHasNoMarginBottom={true}
					/>
				);
				
			case 'multiselect':
				return (
					<SelectControl
						key={fieldName}
						label={label}
						help={description}
						value={value}
						options={options}
						multiple={true}
						onChange={onChange}
						__next40pxDefaultSize={true}
						__nextHasNoMarginBottom={true}
					/>
				);
				
			case 'number':
				return (
					<RangeControl
						key={fieldName}
						label={label}
						help={description}
						value={value}
						min={field.min || 1}
						max={field.max || 100}
						onChange={onChange}
						__next40pxDefaultSize={true}
						__nextHasNoMarginBottom={true}
					/>
				);
				
			case 'toggle':
				return (
					<ToggleControl
						key={fieldName}
						label={label}
						help={description}
						checked={value}
						onChange={onChange}
						__nextHasNoMarginBottom={true}
					/>
				);
				
			case 'text':
				return (
					<TextControl
						key={fieldName}
						label={label}
						help={description}
						value={value}
						onChange={onChange}
						__next40pxDefaultSize={true}
						__nextHasNoMarginBottom={true}
					/>
				);
				
			default:
				return null;
		}
	};
	
	// Helper function to render a panel
	const renderPanel = (panelName, panel) => {
		const fields = Object.entries(panel.fields || {});
		
		if (fields.length === 0) {
			return null;
		}
		
		return (
			<PanelBody
				key={panelName}
				title={__(panel.panel_label, 'wpuf-pro')}
				initialOpen={panel.initial_open || false}
			>
				{fields.map(([fieldName, field]) => renderField(fieldName, field))}
			</PanelBody>
		);
	};
	
	// If no panels, return null to avoid errors
	if (!panels || Object.keys(panels).length === 0) {
		return null;
	}
	
	return (
		<>
			{Object.entries(panels).map(([panelName, panel]) => 
				renderPanel(panelName, panel)
			)}
		</>
	);
} 