/**
 * WordPress dependencies
 */
import { Modal, Button } from '@wordpress/components';
import { useDispatch } from '@wordpress/data';
import { store as blockEditorStore } from '@wordpress/block-editor';
import { createBlocksFromInnerBlocksTemplate } from '@wordpress/blocks';
import { __ } from '@wordpress/i18n';

/**
 * Simplified pattern selection for MVP
 */
export function SimplePatternSelectionModal({
	clientId,
	attributes,
	setIsPatternSelectionModalOpen,
}) {
	const { replaceInnerBlocks } = useDispatch(blockEditorStore);

	const patterns = [
		{
			id: 'layout-2-basic',
			title: __('Layout 2 - Basic Cards', 'wpuf-pro'),
			description: __('Display users in a card layout with avatar, name, and bio.', 'wpuf-pro'),
			template: [
				['wpuf-ud/directory-item', {}, [
					['core/group', { className: 'wpuf-user-card wpuf-p-4 wpuf-border wpuf-rounded-lg wpuf-shadow-sm' }, [
						['wpuf-ud/avatar', { avatarSize: 'medium', avatarShape: 'circle' }],
						['wpuf-ud/name', { nameFormat: 'display_name', headingLevel: 'p' }],
						['wpuf-ud/bio', { characterLimit: 100, readMoreStyle: 'none' }]
					]]
				]]
			]
		},
		{
			id: 'layout-1-basic',
			title: __('Layout 1 - Basic List', 'wpuf-pro'),
			description: __('Display users in a simple list format.', 'wpuf-pro'),
			template: [
				['wpuf-ud/directory-item', {}, [
					['core/group', { className: 'wpuf-user-item wpuf-flex wpuf-items-center wpuf-gap-4 wpuf-p-4 wpuf-border-b' }, [
						['wpuf-ud/avatar', { avatarSize: 'small', avatarShape: 'circle' }],
						['wpuf-ud/name', { nameFormat: 'display_name', headingLevel: 'p' }],
						['wpuf-ud/bio', { characterLimit: 150, readMoreStyle: 'none' }]
					]]
				]]
			]
		}
	];

	const handlePatternSelect = (pattern) => {
		try {
			// Convert the template to blocks using createBlocksFromInnerBlocksTemplate
			const blocks = createBlocksFromInnerBlocksTemplate(pattern.template);
			replaceInnerBlocks(clientId, blocks, false);
			setIsPatternSelectionModalOpen(false);
		} catch (error) {
			// Error handling
		}
	};

	return (
		<Modal
			title={__('Choose a pattern', 'wpuf-pro')}
			onRequestClose={() => setIsPatternSelectionModalOpen(false)}
			className="wpuf-pattern-selection-modal"
		>
			<div className="wpuf-pattern-grid wpuf-grid wpuf-grid-cols-1 md:wpuf-grid-cols-2 wpuf-gap-4 wpuf-p-4">
				{patterns.map((pattern) => (
					<div key={pattern.id} className="wpuf-pattern-card wpuf-border wpuf-rounded-lg wpuf-p-4 wpuf-cursor-pointer hover:wpuf-shadow-md">
						<h3 className="wpuf-text-lg wpuf-font-medium wpuf-mb-2">{pattern.title}</h3>
						<p className="wpuf-text-sm wpuf-text-gray-600 wpuf-mb-4">{pattern.description}</p>
						<Button
							variant="primary"
							onClick={() => handlePatternSelect(pattern)}
							className="wpuf-w-full"
						>
							{__('Choose Pattern', 'wpuf-pro')}
						</Button>
					</div>
				))}
			</div>
		</Modal>
	);
} 