/**
 * WordPress dependencies
 */
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies
 */
import variations from '../variations';

/**
 * Extract template from variation by layout name
 * 
 * @param {string} layoutName - The layout name to extract template for
 * @returns {Array} The template array for the layout
 */
export function getTemplate(layoutName) {
	const variation = variations.find(v => v.name === layoutName);
	return variation?.innerBlocks || [];
}

/**
 * Extract default attributes from variation by layout name
 * 
 * @param {string} layoutName - The layout name to extract attributes for
 * @returns {Object} The default attributes for the layout
 */
export function getTemplateAttributes(layoutName) {
	const variation = variations.find(v => v.name === layoutName);
	return variation?.attributes || {};
}

/**
 * Get all available layout options for the directory block
 * 
 * @returns {Array} Array of layout options with name, title, icon, and template
 */
export function getLayoutOptions() {
	return variations.map(v => ({
		name: v.name,
		title: v.title,
		icon: v.icon,
		template: v.innerBlocks || []
	}));
}

/**
 * Get layout option by name
 * 
 * @param {string} layoutName - The layout name to find
 * @returns {Object|null} The layout option or null if not found
 */
export function getLayoutOption(layoutName) {
	return variations.find(v => v.name === layoutName) || null;
}

/**
 * Check if a layout name is valid
 * 
 * @param {string} layoutName - The layout name to validate
 * @returns {boolean} True if the layout exists
 */
export function isValidLayout(layoutName) {
	return variations.some(v => v.name === layoutName);
}

/**
 * Get all available layout names
 * 
 * @returns {Array} Array of layout names
 */
export function getAvailableLayoutNames() {
	return variations.map(v => v.name);
}

/**
 * Get template metadata including name, title, icon, and attributes
 * 
 * @param {string} layoutName - The layout name to get metadata for
 * @returns {Object|null} Template metadata or null if not found
 */
export function getTemplateMetadata(layoutName) {
	const variation = variations.find(v => v.name === layoutName);
	if (!variation) return null;
	
	return {
		name: variation.name,
		title: variation.title,
		icon: variation.icon,
		attributes: variation.attributes,
		scope: variation.scope
	};
}

/**
 * Get all template metadata for all layouts
 * 
 * @returns {Array} Array of template metadata objects
 */
export function getAllTemplateMetadata() {
	return variations.map(v => ({
		name: v.name,
		title: v.title,
		icon: v.icon,
		attributes: v.attributes,
		scope: v.scope
	}));
}
