import { __ } from '@wordpress/i18n';

const UserName = ({ user, attributes, context }) => {
    const {
        nameFormat = 'display_name',
        headingLevel = 'h3',
        showRoles = false,
        linkToProfile = false
    } = attributes || {};

    // Get formatted name based on format setting
    const getFormattedName = () => {
        if (!user) return __('User Name', 'wpuf-pro');

        switch (nameFormat) {
            case 'first_name':
                return user.first_name || user.display_name || user.name || user.user_login;
            case 'last_name':
                return user.last_name || user.display_name || user.name || user.user_login;
            case 'first_last':
                const firstName = user.first_name || '';
                const lastName = user.last_name || '';
                return firstName && lastName ? `${firstName} ${lastName}` : user.display_name || user.name || user.user_login;
            case 'last_first':
                const first = user.first_name || '';
                const last = user.last_name || '';
                return first && last ? `${last}, ${first}` : user.display_name || user.name || user.user_login;
            case 'username':
                return user.user_login || user.username;
            case 'nickname':
                return user.nickname || user.display_name || user.name || user.user_login;
            case 'display_name':
            default:
                return user.display_name || user.name || user.user_login;
        }
    };

    // Get user roles
    const getUserRoles = () => {
        if (!user || !user.roles) return [];
        
        const roleLabels = {
            administrator: __('Administrator', 'wpuf-pro'),
            editor: __('Editor', 'wpuf-pro'),
            author: __('Author', 'wpuf-pro'),
            contributor: __('Contributor', 'wpuf-pro'),
            subscriber: __('Subscriber', 'wpuf-pro'),
            customer: __('Customer', 'wpuf-pro'),
            shop_manager: __('Shop Manager', 'wpuf-pro')
        };

        return user.roles.map(role => roleLabels[role] || role);
    };

    // Render name with optional link
    const renderNameWithLink = () => {
        const name = getFormattedName();
        const Tag = headingLevel;

        if (linkToProfile && user) {
            const profileUrl = `?username=${user.user_login || user.username}`;
            return (
                <Tag className="wpuf-user-name">
                    <a href={profileUrl} className="wpuf-user-name-link">
                        {name}
                    </a>
                </Tag>
            );
        }

        return (
            <Tag className="wpuf-user-name">
                {name}
            </Tag>
        );
    };

    // Render roles if enabled
    const renderRoles = () => {
        if (!showRoles) return null;

        const roles = getUserRoles();
        if (roles.length === 0) return null;

        return (
            <div className="wpuf-user-roles">
                <span className="wpuf-roles-label">{__('Roles:', 'wpuf-pro')}</span>
                <span className="wpuf-roles-list">
                    {roles.join(', ')}
                </span>
            </div>
        );
    };

    return (
        <div className="wpuf-user-name-container">
            {renderNameWithLink()}
            {renderRoles()}
        </div>
    );
};

export default UserName; 