/**
 * WordPress dependencies
 */
import { __ } from '@wordpress/i18n';

/**
 * Search patterns by title, description, or keywords.
 *
 * @param {Array} patterns The patterns to search.
 * @param {string} searchValue The search value.
 * @return {Array} The filtered patterns.
 */
export const searchPatterns = (patterns, searchValue) => {
	if (!searchValue) {
		return patterns;
	}

	const searchLower = searchValue.toLowerCase();

	return patterns.filter((pattern) => {
		const title = pattern.title?.toLowerCase() || '';
		const description = pattern.description?.toLowerCase() || '';
		const keywords = pattern.keywords?.join(' ').toLowerCase() || '';

		return (
			title.includes(searchLower) ||
			description.includes(searchLower) ||
			keywords.includes(searchLower)
		);
	});
};

/**
 * Get default patterns for User Directory block.
 *
 * @return {Array} The default patterns.
 */
export const getDefaultUserDirectoryPatterns = () => {
	return [
		{
			name: 'wpuf-ud/directory/layout-1',
			title: __('User List Layout', 'wpuf-pro'),
			description: __('A simple list layout for displaying users.', 'wpuf-pro'),
			keywords: ['users', 'list', 'directory'],
			categories: ['user-directory'],
			content: '<!-- wp:wpuf-ud/directory-item -->\n<div class="wp-block-wpuf-ud-directory-item">\n<!-- wp:wpuf-ud/avatar {"avatarSize":"medium"} /-->\n<!-- wp:wpuf-ud/name {"nameFormat":"display_name","headingLevel":"h3"} /-->\n<!-- wp:wpuf-ud/bio {"characterLimit":100} /-->\n</div>\n<!-- /wp:wpuf-ud/directory-item -->',
		},
		{
			name: 'wpuf-ud/directory/layout-2',
			title: __('User Grid Layout', 'wpuf-pro'),
			description: __('A grid layout for displaying users in cards.', 'wpuf-pro'),
			keywords: ['users', 'grid', 'cards', 'directory'],
			categories: ['user-directory'],
			content: '<!-- wp:wpuf-ud/directory-item -->\n<div class="wp-block-wpuf-ud-directory-item">\n<!-- wp:core/group {"className":"wpuf-user-card"} -->\n<div class="wp-block-group wpuf-user-card">\n<!-- wp:wpuf-ud/avatar {"avatarSize":"large","avatarShape":"circle"} /-->\n<!-- wp:wpuf-ud/name {"nameFormat":"display_name","headingLevel":"h3"} /-->\n<!-- wp:wpuf-ud/bio {"characterLimit":150} /-->\n<!-- wp:core/button {"text":"View Profile"} /-->\n</div>\n<!-- /wp:core/group -->\n</div>\n<!-- /wp:wpuf-ud/directory-item -->',
		},
		{
			name: 'wpuf-ud/directory/layout-3',
			title: __('User Profile Layout', 'wpuf-pro'),
			description: __('A detailed profile layout for displaying user information.', 'wpuf-pro'),
			keywords: ['users', 'profile', 'detailed', 'directory'],
			categories: ['user-directory'],
			content: '<!-- wp:wpuf-ud/directory-item -->\n<div class="wp-block-wpuf-ud-directory-item">\n<!-- wp:core/group {"className":"wpuf-user-profile"} -->\n<div class="wp-block-group wpuf-user-profile">\n<!-- wp:wpuf-ud/avatar {"avatarSize":"large","avatarShape":"circle"} /-->\n<!-- wp:wpuf-ud/name {"nameFormat":"display_name","headingLevel":"h2"} /-->\n<!-- wp:wpuf-ud/bio {"characterLimit":200} /-->\n<!-- wp:wpuf-ud/contact /-->\n<!-- wp:wpuf-ud/posts {"postsPerPage":5} /-->\n</div>\n<!-- /wp:core/group -->\n</div>\n<!-- /wp:wpuf-ud/directory-item -->',
		},
	];
}; 