<?php
/**
 * User Files Block Template
 *
 * @since 4.2.0
 *
 * @var WP_User $user User object
 * @var array $attributes Block attributes
 * @var string $content Block content
 * @var WP_Block $block Block instance
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! $user ) {
    return;
}

$meta_field_key = ! empty( $attributes['metaFieldKey'] ) ? $attributes['metaFieldKey'] : '';
$field_label = ! empty( $attributes['fieldLabel'] ) ? $attributes['fieldLabel'] : '';
$show_label = isset( $attributes['showLabel'] ) ? (bool) $attributes['showLabel'] : true;

// Don't show anything if no field key
if ( empty( $meta_field_key ) ) {
    return;
}

// Get the custom field value
$field_value = get_user_meta( $user->ID, $meta_field_key, true );

// Don't show anything if no value
if ( empty( $field_value ) ) {
    return;
}

// Use field key as label if no label provided
if ( empty( $field_label ) ) {
    $field_label = ucwords( str_replace( [ '_', '-' ], ' ', $meta_field_key ) );
}

// Use the FileHelper to convert file IDs to file data
$files = \WPUF\UserDirectory\FileHelper::convert_file_ids_to_data( $field_value );

// Don't show anything if no files
if ( empty( $files ) ) {
    return;
}

// Role-based visibility: show only if the profile user has one of the selected roles
$selected_roles = $attributes['profileUserRoles'] ?? [ 'all' ];

if ( ! is_array( $selected_roles ) || empty( $selected_roles ) ) {
    $selected_roles = [ 'all' ];
}
if ( ! in_array( 'all', $selected_roles, true ) ) {
    $user_roles = is_array( $user->roles ?? null ) ? $user->roles : [];
    if ( empty( array_intersect( $user_roles, $selected_roles ) ) ) {
        return;
    }
}

// Viewer role-based visibility: check current viewer (logged-in user or guest)
$viewer_roles = $attributes['viewerRoles'] ?? [ 'all' ];

if ( ! is_array( $viewer_roles ) || empty( $viewer_roles ) ) {
    $viewer_roles = [ 'all' ];
}

if ( ! in_array( 'all', $viewer_roles, true ) ) {
    $current_user     = wp_get_current_user();
    // Administrators can always view
    if ( is_user_logged_in() && user_can( $current_user, 'administrator' ) ) {
        // allow
    } else {
        $is_guest_allowed = in_array( 'guest', $viewer_roles, true );
        $viewer_has_role  = false;
        if ( is_user_logged_in() ) {
            $current_roles   = is_array( $current_user->roles ?? null ) ? $current_user->roles : [];
            $viewer_has_role = ! empty( array_intersect( $current_roles, $viewer_roles ) );
        } else {
            $viewer_has_role = $is_guest_allowed; // guest
        }
        if ( ! $viewer_has_role ) {
            return;
        }
    }
}

// Get block wrapper attributes (includes styling from block editor)
$wrapper_attributes = wpuf_get_block_wrapper_attributes( $attributes, 'files', [
    'class' => 'wpuf-user-files',
    'data-user-id' => $user->ID,
    'data-field-key' => $meta_field_key,
] );


?>
<div <?php echo $wrapper_attributes; ?>>
    <div class="wpuf-files-content">
        <?php if ( $show_label && ! empty( $field_label ) ) : ?>
            <div class="wpuf-files-label">
                <?php echo esc_html( $field_label ); ?>
            </div>
        <?php endif; ?>

        <?php echo \WPUF\UserDirectory\FileHelper::render_file_grid( $files ); ?>
    </div>
</div>
