/**
 * WP User Frontend Pro - Login Form Frontend Logic
 *
 * Handles placeholder conversion, help text, and form element manipulation
 * for all login form layouts.
 *
 * @package WP User Frontend Pro
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * Initialize login form enhancements
     */
    function initializeLoginForm() {
        var loginForm = document.getElementById('wpuf-login-form');

        if (!loginForm) {
            return;
        }

        // Get current layout from localized data
        var layout = wpufLoginForm.layout || 'layout1';

        // Add layout class to form
        loginForm.classList.add('wpuf-layout-' + layout.replace('layout', ''));

        // Convert labels to placeholders
        convertLabelsToPlaceholders(loginForm);

        // Add help text for layout 7
        if (layout === 'layout7') {
            addHelpText(loginForm);
        }

        // Wrap action links
        wrapActionLinks(loginForm);
    }

    /**
     * Convert label text to input placeholders
     *
     * @param {HTMLElement} loginForm The login form element
     */
    function convertLabelsToPlaceholders(loginForm) {
        var labels = loginForm.querySelectorAll('label');

        labels.forEach(function(label) {
            // Skip "Remember Me" checkbox label
            if (label.closest('.forgetmenot')) {
                return;
            }

            var forAttr = label.getAttribute('for');

            if (forAttr) {
                var input = document.getElementById(forAttr);

                if (input) {
                    var labelText = label.textContent.trim();
                    input.setAttribute('placeholder', 'Enter ' + labelText);
                }
            }
        });
    }

    /**
     * Add help text to form fields (Layout 7 only)
     *
     * @param {HTMLElement} loginForm The login form element
     */
    function addHelpText(loginForm) {
        var form = loginForm.querySelector('form');

        if (!form) {
            return;
        }

        // Help text from localized data
        var usernameHelp = wpufLoginForm.usernameHelpText || 'Please enter your username or email address';
        var passwordHelp = wpufLoginForm.passwordHelpText || 'Please enter your password';

        // Add help text to username/email field
        var usernameField = form.querySelector('input[name="log"]');
        if (usernameField && !usernameField.parentElement.querySelector('.wpuf-help')) {
            var usernameHelpSpan = document.createElement('span');
            usernameHelpSpan.className = 'wpuf-help';
            usernameHelpSpan.textContent = usernameHelp;
            usernameField.parentElement.appendChild(usernameHelpSpan);
        }

        // Add help text to password field
        var passwordField = form.querySelector('input[type="password"]');
        if (passwordField && !passwordField.parentElement.querySelector('.wpuf-help')) {
            var passwordHelpSpan = document.createElement('span');
            passwordHelpSpan.className = 'wpuf-help';
            passwordHelpSpan.textContent = passwordHelp;
            passwordField.parentElement.appendChild(passwordHelpSpan);
        }
    }

    /**
     * Wrap action links in proper containers
     *
     * @param {HTMLElement} loginForm The login form element
     */
    function wrapActionLinks(loginForm) {
        var links = loginForm.querySelectorAll('a');

        links.forEach(function(link) {
            // Skip if already wrapped
            if (link.parentElement.classList.contains('wpuf-lost-password-wrapper')) {
                return;
            }

            // Skip if inside form
            if (link.closest('form')) {
                return;
            }

            // Check if it's an action link (lost password, register, etc.)
            var isActionLink = /lostpassword|action=login|\/login\/|register/.test(link.href);

            if (isActionLink || link.parentElement.id === 'wpuf-login-form') {
                var wrapper = document.createElement('div');
                wrapper.className = 'wpuf-lost-password-wrapper';
                link.parentNode.insertBefore(wrapper, link);
                wrapper.appendChild(link);
            }
        });
    }

    /**
     * Initialize on DOM ready
     */
    $(document).ready(function() {
        initializeLoginForm();
    });

})(jQuery);
