/**
 * AI Review Column JavaScript
 *
 * Handles tooltip functionality and interactions for the AI Review column
 *
 * @since 4.2.5
 */
(function($) {
    'use strict';

    /**
     * Initialize AI Review column functionality
     */
    function initAIReviewColumn() {
        // Only run on post list screen
        if (!$('body').hasClass('edit-php') || !$('body').hasClass('post-type-post')) {
            return;
        }

        initTooltips();
        initAccessibility();
    }

    /**
     * Initialize tooltip functionality
     */
    function initTooltips() {
        // Wrap AI circles with tooltip containers
        $('.wpuf-ai-circle').each(function() {
            var $circle = $(this);
            var tooltipText = $circle.attr('title') || $circle.data('tooltip');

            if (!tooltipText) {
                return;
            }

            // Remove title attribute to prevent default tooltip
            $circle.removeAttr('title');

            // Wrap with tooltip container
            $circle.wrap('<div class="wpuf-ai-tooltip"></div>');

            // Add tooltip text
            $circle.after('<div class="wpuf-ai-tooltip-text">' + escapeHtml(tooltipText) + '</div>');
        });

        // Handle tooltip positioning on scroll
        $(window).on('scroll resize', function() {
            $('.wpuf-ai-tooltip:hover .wpuf-ai-tooltip-text').each(function() {
                positionTooltip($(this));
            });
        });
    }

    /**
     * Position tooltip to stay within viewport
     */
    function positionTooltip($tooltip) {
        var $trigger = $tooltip.siblings('.wpuf-ai-circle');
        var triggerOffset = $trigger.offset();
        var tooltipWidth = $tooltip.outerWidth();
        var viewportWidth = $(window).width();

        if (!triggerOffset) {
            return;
        }

        var leftPosition = triggerOffset.left - (tooltipWidth / 2) + ($trigger.outerWidth() / 2);

        // Adjust if tooltip would go off screen
        if (leftPosition < 10) {
            leftPosition = 10;
        } else if (leftPosition + tooltipWidth > viewportWidth - 10) {
            leftPosition = viewportWidth - tooltipWidth - 10;
        }

        $tooltip.css('left', leftPosition + 'px');
    }


    /**
     * Initialize accessibility features
     */
    function initAccessibility() {
        // Add ARIA labels for screen readers
        $('.wpuf-ai-circle').each(function() {
            var $circle = $(this);
            var tooltipText = $circle.data('tooltip');

            if (tooltipText) {
                $circle.attr('aria-label', tooltipText);
                $circle.attr('role', 'button');
                $circle.attr('tabindex', '0');
            }
        });

        // Handle keyboard navigation
        $('.wpuf-ai-circle').on('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                $(this).trigger('mouseenter');
            }
        });

        // Handle focus events for keyboard users
        $('.wpuf-ai-circle').on('focus', function() {
            $(this).trigger('mouseenter');
        }).on('blur', function() {
            $(this).trigger('mouseleave');
        });
    }

    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };

        return text.replace(/[&<>"']/g, function(m) {
            return map[m];
        });
    }


    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        initAIReviewColumn();
    });


})(jQuery);
