<?php

namespace WeDevs\Wpuf\Pro\Admin;

use WeDevs\Wpuf\Free\Free_Loader;

class Feature_Lock {

    /**
     * @var Free_Loader
     */
    public $free_loader;

    public function __construct() {
        $this->free_loader = new Free_Loader();
        $this->deactive_modules();

        // form builder
        add_filter( 'wpuf_post_form_builder_setting_menu_titles', [ $this, 'remove_post_expiration' ], 11 );
        add_filter( 'wpuf_form_builder_settings_general', [ $this, 'remove_multistep_settings' ], 11 );
        add_filter( 'wpuf_form_fields', [ $this, 'remove_form_fields' ], 11 );
        add_filter( 'wpuf_form_fields_groups', [ $this, 'remove_form_groups' ], 11 );
        add_filter( 'wpuf_get_post_form_templates', [ $this, 'remove_form_template' ], 11 );
        add_filter( 'wpuf_form_builder_settings_notification', [ $this, 'remove_update_post_notification' ], 11 );
        add_filter( 'wpuf_form_builder_settings_display', [ $this, 'remove_form_settings_display' ], 11 );

        // subscription
        add_filter( 'wpuf_subscription_payment_fields', [ $this, 'remove_recurring_fields' ], 12 );
        add_filter( 'wpuf_subscription_section_advanced', [ $this, 'remove_taxonomy_restriction_section' ], 11 );

        // settings
        add_filter( 'wpuf_options_others', [ $this, 'remove_general_settings' ], 11 );
        add_filter( 'wpuf_options_wpuf_my_account', [ $this, 'remove_my_account_settings' ], 11 );
        add_filter( 'wpuf_options_payment', [ $this, 'remove_payment_options' ], 11 );
        add_filter( 'wpuf_mail_options', [ $this, 'remove_subs_notification_mail_options' ], 11 );
        add_filter( 'wpuf_settings_sections', [ $this, 'remove_setting_sections' ], 11 );
        add_filter( 'wpuf_settings_fields', [ $this, 'remove_global_settings' ], 11 );
        add_filter( 'wpuf_settings_user_roles', '__return_empty_array', 11 );
        add_filter( 'wpuf_field_get_js_settings', [ $this, 'remove_conditional_logic' ], 11 );
    }

    /**
     * Remove form settings display from form builder
     *
     * @since 4.1.3
     *
     * @param array $settings Form settings.
     *
     * @return array
     */
    public function remove_form_settings_display( $settings ) {
        if ( ! empty( $settings['section']['custom_form_style']['fields']['form_layout'] ) ) {
            unset( $settings['section']['custom_form_style']['fields']['form_layout'] );
        }

        return $settings;
    }

    /**
     * Remove update post notification from form builder
     *
     * @since 4.1.3
     *
     * @param array $settings Form settings.
     *
     * @return array
     */
    public function remove_update_post_notification( $settings ) {
        if ( ! empty( $settings['section']['update_post'] ) ) {
            unset( $settings['section']['update_post'] );
        }

        return $settings;
    }

    /**
     * Remove conditional logic from form fields
     *
     * @since 4.1.3
     *
     * @param array $field Form field.
     *
     * @return array
     */
    public function remove_conditional_logic( $field ) {
        if ( empty( $field['settings'] ) ) {
            return $field;
        }

        $remove_key = 'wpuf_cond';

        foreach ( $field['settings'] as $key => $value ) {
            if ( ! empty( $value['name'] ) && $remove_key === $value['name'] ) {
                unset( $field['settings'][ $key ] );
            }
        }

        return $field;
    }

    /**
     * Remove form template from form builder
     *
     * @since 4.1.3
     *
     * @param array $templates Form templates.
     *
     * @return array
     */
    public function remove_form_template( $templates ) {
        $key = 'post_form_template_edd';

        if ( ! empty( $templates[ $key ] ) ) {
            unset( $templates[ $key ] );
        }

        return $templates;
    }

    /**
     * Remove pro settings from global settings from WPUF settings
     *
     * @since 4.1.3
     *
     * @param array $settings Global settings fields.
     *
     * @return array
     */
    public function remove_global_settings( $settings ) {
        $fields_to_remove = [
            'avatar_size',
            'pending_user_message',
            'denied_user_message',
        ];

        if ( empty( $settings['wpuf_profile'] ) ) {
            return $settings;
        }

        foreach ( $settings['wpuf_profile'] as $key => $field ) {
            if ( ! empty( $field['name'] ) && in_array( $field['name'], $fields_to_remove ) ) {
                unset( $settings['wpuf_profile'][ $key ] );
            }
        }

        return $settings;
    }

    /**
     * Remove pro sections from WPUF settings
     *
     * @since 4.1.3
     *
     * @param array $sections Settings sections.
     *
     * @return array
     */
    public function remove_setting_sections( $sections ) {
        $sections_to_remove = [
            'wpuf_payment_tax',
            'wpuf_payment_invoices',
            'wpuf_content_restriction',
        ];

        foreach ( $sections as $setting ) {
            // remove the index containing id => `wpuf_payment_tax`
            if ( ! empty( $setting['id'] ) && in_array( $setting['id'], $sections_to_remove ) ) {
                $key = array_search( $setting, $sections, true );
                if ( false !== $key ) {
                    unset( $sections[ $key ] );
                }
            }
        }

        return $sections;
    }

    /**
     * Remove pro settings from subscription notification mail options from WPUF settings
     *
     * @since 4.1.3
     *
     * @param array $mail_options Mail options fields.
     *
     * @return array
     */
    public function remove_subs_notification_mail_options( $mail_options ) {
        $fields_to_remove = [
            'subscription_setting',
            'enable_subs_notification',
            'pre_sub_notification_date',
            'post_sub_notification_date',
            'pre_sub_exp_subject',
            'pre_sub_exp_body',
            'post_sub_exp_subject',
            'post_sub_exp_body',
            'pending_user_email',
            'enable_pending_user_email_notification',
            'pending_user_email_subject',
            'pending_user_email_body',
            'denied_user_email',
            'enable_denied_user_email_notification',
            'denied_user_email_subject',
            'denied_user_email_body',
            'approved_user_email',
            'enable_approved_user_email_notification',
            'approved_user_email_subject',
            'approved_user_email_body',
            'account_activated_user_email',
            'account_activated_user_email_notification',
            'account_activated_user_email_subject',
            'account_activated_user_email_body',
            'approved_post_email',
            'enable_post_approval_notification',
            'approved_post_email_subject',
            'approved_post_email_body',
        ];

        foreach ( $mail_options as $setting ) {
            if ( ! empty( $setting['name'] ) && in_array( $setting['name'], $fields_to_remove ) ) {
                $key = array_search( $setting, $mail_options, true );
                if ( false !== $key ) {
                    unset( $mail_options[ $key ] );
                }
            }
        }

        return $mail_options;
    }

    /**
     * Remove pro settings from payment settings from WPUF settings
     *
     * @since 4.1.3
     *
     * @param array $options Payment settings fields.
     *
     * @return array
     */
    public function remove_payment_options( $options ) {
        $fields_to_remove = [
            'gate_instruct_stripe',
            'use_legacy_stripe_api',
            'use_stripe_3ds',
            'stripe_api_secret',
            'stripe_api_key',
            'stripe_endpoint',
            'stripe_signing_secret',
        ];

        foreach ( $options as $setting ) {
            if ( ! empty( $setting['name'] ) && in_array( $setting['name'], $fields_to_remove ) ) {
                $key = array_search( $setting, $options, true );
                if ( false !== $key ) {
                    unset( $options[ $key ] );
                }
            }
        }

        return $options;
    }

    /**
     * Remove pro settings from my account settings from WPUF settings
     *
     * @since 4.1.3
     *
     * @param array $options My account settings fields.
     *
     * @return array
     */
    public function remove_my_account_settings( $options ) {
        $fields_to_remove = [
            'show_edit_profile_menu',
            'edit_profile_form',
        ];

        foreach ( $options as $setting ) {
            if ( ! empty( $setting['name'] ) && in_array( $setting['name'], $fields_to_remove ) ) {
                $key = array_search( $setting, $options, true );
                if ( false !== $key ) {
                    unset( $options[ $key ] );
                }
            }
        }

        return $options;
    }

    /**
     * Remove pro settings from general settings from WPUF settings
     *
     * @since 4.1.3
     *
     * @param array $wpuf_general_fields General settings fields.
     *
     * @return array
     */
    public function remove_general_settings( $wpuf_general_fields ) {
        foreach ( $wpuf_general_fields as $setting ) {
            if ( ! empty( $setting['name'] ) && 'gmap_api_key' === $setting['name'] ) {
                $key = array_search( $setting, $wpuf_general_fields, true );
                if ( false !== $key ) {
                    unset( $wpuf_general_fields[ $key ] );
                }
            }
        }

        return $wpuf_general_fields;
    }

    /**
     * Deactivate all modules
     *
     * @since 4.1.3
     *
     * @return void
     */
    public function deactive_modules() {
        $modules = wpuf_pro_get_modules();

        foreach ( $modules as $key => $value ) {
            wpuf_pro_deactivate_module( $key );
        }
    }

    /**
     * Remove taxonomy restriction section from subscription settings
     *
     * @since 4.1.3
     *
     * @param array $sections Subscription settings sections.
     *
     * @return array
     */
    public function remove_taxonomy_restriction_section( $sections ) {
        foreach ( $sections['advanced_configuration'] as $section ) {
            if ( ! empty( $section['id'] ) && 'taxonomy_restriction' === $section['id'] ) {
                $key = array_search( $section, $sections['advanced_configuration'], true );
                if ( false !== $key ) {
                    unset( $sections['advanced_configuration'][ $key ] );
                }
            }
        }

        return $sections;
    }

    /**
     * Remove recurring settings from subscriptions
     *
     * @since 4.1.3
     *
     * @param array $fields Form fields.
     *
     * @return array
     */
    public function remove_recurring_fields( $fields ) {
        $fields_to_remove = [
            'enable_recurring',
            'billing_cycle',
            'stop_cycle',
            'billing_limit',
            'trial',
            'trial_period',
        ];

        foreach ( $fields_to_remove as $field ) {
            if ( isset( $fields['payment_details'][ $field ] ) ) {
                unset( $fields['payment_details'][ $field ] );
            }
        }

        return $fields;
    }

    /**
     * Remove form groups from form builder
     *
     * @since 4.1.3
     *
     * @param array $groups Form groups.
     *
     * @return array
     */
    public function remove_form_groups( $groups ) {
        $fields_to_remove = [
            'user_login',
            'first_name',
            'last_name',
            'display_name',
            'nickname',
            'user_email',
            'user_url',
            'user_bio',
            'password',
            'avatar',
            'profile_photo',
            'profile_phone',
            'cover_photo',
            'repeat_field',
            'date_field',
            'file_upload',
            'country_list_field',
            'numeric_text_field',
            'address_field',
            'google_map',
            'shortcode',
            'action_hook',
            'toc',
            'ratings',
            'step_start',
            'embed',
            'really_simple_captcha',
            'math_captcha',
            'phone_field',
            'time_field',
        ];

        $modified_groups = $groups;

        foreach ( $groups as $key => $group ) {
            if ( isset( $group['fields'] ) && is_array( $group['fields'] ) ) {
                foreach ( $fields_to_remove as $field ) {
                    if ( in_array( $field, $group['fields'], true ) ) {
                        // remove the field from the $modified_groups[$key]['fields'] array
                        $modified_groups[ $key ]['fields'] = array_diff( $modified_groups[ $key ]['fields'], [ $field ] );
                        // remove the group if it has no fields left
                        if ( empty( $modified_groups[ $key ]['fields'] ) ) {
                            unset( $modified_groups[ $key ] );
                        }
                    }
                }
            }
        }

        return $modified_groups;
    }

    /**
     * Remove form fields from form builder
     *
     * @since 4.1.3
     *
     * @param array $fields Form fields.
     *
     * @return array
     */
    public function remove_form_fields( $fields ) {
        $fields_to_remove = [
            'user_login',
            'first_name',
            'last_name',
            'display_name',
            'nickname',
            'user_email',
            'user_url',
            'user_bio',
            'password',
            'avatar',
            'repeat_field',
            'date_field',
            'file_upload',
            'country_list_field',
            'numeric_text_field',
            'address_field',
            'google_map',
            'shortcode',
            'action_hook',
            'toc',
            'ratings',
            'step_start',
            'embed',
            'really_simple_captcha',
            'math_captcha',
            'phone_field',
            'time_field',
        ];

        foreach ( $fields_to_remove as $field ) {
            if ( isset( $fields[ $field ] ) ) {
                unset( $fields[ $field ] );
            }
        }

        return $fields;
    }

    /**
     * Remove post expiration settings from form builder
     *
     * @since 4.1.3
     *
     * @param array $settings Form settings.
     *
     * @return array
     */
    public function remove_post_expiration( $settings ) {
        if ( isset( $settings['post_expiration'] ) ) {
            unset( $settings['post_expiration'] );
        }

        return $settings;
    }

    /**
     * Remove multistep settings from form builder
     *
     * @since 4.1.3
     *
     * @param array $settings Form settings.
     *
     * @return array
     */
    public function remove_multistep_settings( $settings ) {
        $fields_to_remove = [
            'enable_multistep',
            'multistep_progressbar_type',
            'ms_ac_txt_color',
            'ms_active_bgcolor',
            'ms_bgcolor',
        ];

        foreach ( $fields_to_remove as $field ) {
            if ( isset( $settings['section']['before_post_settings']['fields'][ $field ] ) ) {
                unset( $settings['section']['before_post_settings']['fields'][ $field ] );
            }
        }

        return $settings;
    }
}
