<?php

namespace WeDevs\Wpuf\Pro\Fields;

/**
 * Gender Field Class
 *
 * Predefined gender field with fixed options for user profiles/registration
 *
 * @since 4.2.2
 */
class Field_Gender extends Profile_Field {

    /**
     * Constructor
     *
     * @since 4.2.2
     */
    public function __construct() {
        $this->name       = __( 'Gender', 'wpuf-pro' );
        $this->input_type = 'gender_field';
        $this->icon       = 'gender';
    }

    /**
     * Render the Gender field
     *
     * @since 4.2.2
     *
     * @param array  $field_settings
     * @param int    $form_id
     * @param string $type
     * @param int    $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        if ( isset( $post_id ) && $post_id != '0' ) {
            $selected = $this->get_meta( $post_id, $field_settings['name'], $type );
        } else {
            $selected = isset( $field_settings['selected'] ) ? $field_settings['selected'] : '';
        }

        $name = $field_settings['name'];

        $this->field_print_label( $field_settings, $form_id ); ?>

        <div class="wpuf-fields">
            <select
                class="<?php echo 'wpuf_' . esc_attr( $field_settings['name'] ) . '_' . esc_attr( $form_id ); ?>"
                id="<?php echo esc_attr( $field_settings['name'] ) . '_' . esc_attr( $form_id ); ?>"
                name="<?php echo esc_attr( $name ); ?>"
                data-required="<?php echo esc_attr( $field_settings['required'] ); ?>"
                data-type="select">

                <?php if ( !empty( $field_settings['first'] ) ) { ?>
                    <option value="-1"><?php echo esc_html( $field_settings['first'] ); ?></option>
                <?php } ?>

                <?php
                // Use custom options if provided, otherwise use default gender options
                $gender_options = !empty( $field_settings['options'] ) ? $field_settings['options'] : $this->get_gender_options();
                foreach ( $gender_options as $value => $label ) {
                    $current_select = selected( $selected, $value, false ); ?>
                    <option value="<?php echo esc_attr( $value ); ?>"<?php echo esc_attr( $current_select ); ?>><?php echo esc_html( $label ); ?></option>
                    <?php
                } ?>
            </select>
            <?php $this->help_text( $field_settings ); ?>
        </div>

        <?php $this->after_field_print_label();
    }

    /**
     * Get predefined gender options
     *
     * @since 4.2.2
     *
     * @return array
     */
    private function get_gender_options() {
        return [
            'male'           => __( 'Male', 'wpuf-pro' ),
            'female'         => __( 'Female', 'wpuf-pro' ),
            'non_binary'     => __( 'Non-binary', 'wpuf-pro' ),
            'prefer_not_say' => __( 'Prefer not to say', 'wpuf-pro' ),
        ];
    }

    /**
     * Get field options setting
     *
     * @since 4.2.2
     *
     * @return array
     */
    public function get_options_settings() {
        // Get default options but exclude the name field since we'll provide our own meta_key field
        $filtered_options = $this->get_default_option_settings( true, array( 'name' ) );

        // Get dropdown settings with default gender options
        $dropdown_settings = $this->get_default_option_dropdown_settings();
        $dropdown_settings['default'] = $this->get_gender_options();
        $dropdown_settings['help_text'] = __( 'Add custom gender options or use the default ones', 'wpuf-pro' );

        $gender_options = [
            [
                'name'      => 'meta_key',
                'title'     => __( 'Meta Key', 'wpuf-pro' ),
                'type'      => 'text',
                'default'   => 'wpuf_gender',
                'section'   => 'basic',
                'priority'  => 11.5,
                'readonly'  => true,
                'help_text' => __( 'The meta key used to store this field data in the database. This is fixed to "wpuf_gender" and cannot be changed.', 'wpuf-pro' ),
            ],

            $dropdown_settings,

            [
                'name'      => 'first',
                'title'     => __( 'Select Text', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'basic',
                'priority'  => 13,
                'default'   => __( 'Select your gender', 'wpuf-pro' ),
                'help_text' => __( 'First element of the select dropdown. Leave this empty if you don\'t want to show this field', 'wpuf-pro' ),
            ],
        ];

        return array_merge( $filtered_options, $gender_options );
    }

    /**
     * Get the field props
     *
     * @since 4.2.2
     *
     * @return array
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();
        $props    = [
            'input_type'       => 'select',
            'label'            => __( 'Gender', 'wpuf-pro' ),
            'meta_key'         => 'wpuf_gender', // Fixed meta key for form builder
            'is_meta'          => 'yes',
            'options'          => $this->get_gender_options(),
            'first'            => __( 'Select your gender', 'wpuf-pro' ),
            'id'               => 0,
            'is_new'           => true,
            'show_in_post'     => 'yes',
            'hide_field_label' => 'no',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Prepare entry
     *
     * @since 4.2.2
     *
     * @param $field
     *
     * @return mixed
     */
    public function prepare_entry( $field ) {
        check_ajax_referer( 'wpuf_form_add' );

        $val = isset( $_POST[$field['name']] ) ? sanitize_text_field( wp_unslash( $_POST[$field['name']] ) ) : '';

        // Use custom options if provided, otherwise use default gender options
        $gender_options = !empty( $field['options'] ) ? $field['options'] : $this->get_gender_options();

        // Return the readable label instead of value for display
        return isset( $gender_options[$val] ) ? $gender_options[$val] : '';
    }

    /**
     * Render field data for display
     *
     * @since 4.2.2
     *
     * @param mixed $data
     * @param array $field
     *
     * @return string
     */
    public function render_field_data( $data, $field ) {
        $data       = is_array( $data ) ? array_pop( $data ) : $data;
        $hide_label = isset( $field['hide_field_label'] )
            ? wpuf_validate_boolean( $field['hide_field_label'] )
            : false;

        $container_classnames = [ 'wpuf-field-data', 'wpuf-field-data-' . $this->input_type ];

        // Use custom options if provided, otherwise use default gender options
        $gender_options = !empty( $field['options'] ) ? $field['options'] : $this->get_gender_options();

        // If we have a value key, convert it to label
        if ( isset( $gender_options[$data] ) ) {
            $data = $gender_options[$data];
        }

        if ( empty( $data ) ) {
            return '';
        }

        ob_start();
        ?>
            <li class="<?php echo esc_attr( implode( ' ' , $container_classnames ) );  ?>">
                <?php if ( ! $hide_label ): ?>
                    <label><?php echo esc_html( $field['label'] ); ?>:</label>
                <?php endif; ?>
                <?php echo esc_html( $data ); ?>
            </li>
        <?php
        return ob_get_clean();
    }
}
