<?php

namespace WeDevs\Wpuf\Pro\Fields;

use WeDevs\Wpuf\Fields\Form_Field_Radio;
use WeDevs\Wpuf\Pro\Fields\Traits\Pricing_Field_Trait;

/**
 * Pricing Radio Field Class
 *
 * Radio with price values for each option
 *
 * @since 4.2.2
 */
class Field_Pricing_Radio extends Form_Field_Radio {

    use Pricing_Field_Trait;

    public function __construct() {
        $this->name       = __( 'Pricing Radio', 'wpuf-pro' );
        $this->input_type = 'pricing_radio';
        $this->icon       = 'currency';
    }

    /**
     * Render the Pricing Radio field
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        if ( isset( $post_id ) && '0' !== $post_id ) {
            if ( $this->is_meta( $field_settings ) ) {
                $selected = $this->get_meta( $post_id, $field_settings['name'], $type );
            }
        } else {
            $selected = isset( $field_settings['selected'] ) ? $field_settings['selected'] : '';
        }

        $show_price      = isset( $field_settings['show_price_label'] ) && 'yes' === $field_settings['show_price_label'];
        $enable_quantity = isset( $field_settings['enable_quantity'] ) && 'yes' === $field_settings['enable_quantity'];

        $this->field_print_label( $field_settings, $form_id ); ?>

        <div class="wpuf-fields">
            <?php
            if ( $field_settings['options'] && count( $field_settings['options'] ) > 0 ) {
                foreach ( $field_settings['options'] as $value => $option ) {
                    $selected = is_array( $selected ) ? '' : $selected;
                    $price = isset( $field_settings['prices'][$value] ) ? floatval( $field_settings['prices'][$value] ) : 0;
                    ?>
                    <div class="wpuf-pricing-radio-item <?php echo 'yes' === $field_settings['inline'] ? 'wpuf-radio-inline' : 'wpuf-radio-block'; ?>">
                        <label>
                            <input
                                name="<?php echo esc_attr( $field_settings['name'] ); ?>"
                                id="<?php echo esc_attr( $field_settings['name'] ); ?>"
                                class="<?php echo esc_attr( 'wpuf_' . $field_settings['name'] . '_' . $form_id ); ?>"
                                type="radio"
                                data-type="pricing-radio"
                                data-required="<?php echo esc_attr( $field_settings['required'] ); ?>"
                                data-price="<?php echo esc_attr( $price ); ?>"
                                value="<?php echo esc_attr( $value ); ?>"
                                <?php checked( $selected, $value ); ?>
                                ondblclick="WP_User_Frontend.doUncheckRadioBtn(this)"
                            />
                            <?php echo esc_html( $option ); ?>
                            <?php if ( $show_price ) : ?>
                                <span class="wpuf-price-label"> - <?php echo esc_html( number_format( $price, 2 ) ); ?></span>
                            <?php endif; ?>
                        </label>
                        <?php if ( $enable_quantity ) : ?>
                            <input
                                type="number"
                                class="wpuf-pricing-quantity"
                                name="<?php echo esc_attr( $field_settings['name'] ); ?>_qty"
                                data-for="<?php echo esc_attr( $value ); ?>"
                                min="1"
                                value="1"
                                style="width: 60px; margin-left: 10px;"
                                disabled
                            />
                        <?php endif; ?>
                    </div>
                    <?php
                }
            } ?>

            <?php $this->help_text( $field_settings ); ?>
        </div>

        <?php $this->after_field_print_label();
    }

    /**
     * Get field options setting
     */
    public function get_options_settings() {
        $default_options = $this->get_default_option_settings( true, [ 'width' ] );
        $pricing_options = $this->get_pricing_field_options( true, true, false );

        return array_merge( $default_options, $pricing_options );
    }

    /**
     * Get the field props
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'        => 'pricing_radio',
            'label'             => __( 'Pricing Radio', 'wpuf-pro' ),
            'name'              => 'wpuf_pricing_radio',
            'selected'          => '',
            'inline'            => 'no',
            'show_price_label'  => 'yes',
            'enable_quantity'   => 'no',
            'options'           => [
                'first_item'  => __( 'First Item', 'wpuf-pro' ),
                'second_item' => __( 'Second Item', 'wpuf-pro' ),
                'third_item'  => __( 'Third Item', 'wpuf-pro' ),
            ],
            'prices'            => [
                'first_item'  => '10',
                'second_item' => '25',
                'third_item'  => '50',
            ],
            'id'                => 0,
            'is_new'            => true,
            'show_in_post'      => 'yes',
            'hide_field_label'  => 'no',
            'is_meta'           => 'yes',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Prepare entry
     */
    public function prepare_entry( $field ) {
        check_ajax_referer( 'wpuf_form_add' );

        $val = isset( $_POST[$field['name']] ) ? sanitize_text_field( wp_unslash( $_POST[$field['name']] ) ) : '';

        if ( $val && isset( $field['options'][$val] ) ) {
            $label = $field['options'][$val];
            $price = isset( $field['prices'][$val] ) ? floatval( $field['prices'][$val] ) : 0;

            $qty = 1;
            if ( isset( $_POST[$field['name'] . '_qty'] ) ) {
                $qty = intval( $_POST[$field['name'] . '_qty'] );
            }

            return $label . ' (' . ( $price * $qty ) . ')';
        }

        return '';
    }

    public function is_pro() {
        return true;
    }
}
