<?php

namespace WeDevs\Wpuf\Pro\Fields;

/**
 * Profile Phone Field Class
 *
 * Phone field specifically designed for user profile forms
 *
 * @since 4.2.3
 */
class Field_Profile_Phone extends Profile_Field {

    /**
     * Constructor
     *
     * @since 4.2.3
     */
    public function __construct() {
        $this->name       = __( 'Phone', 'wpuf-pro' );
        $this->input_type = 'profile_phone';
        $this->icon       = 'telephone';
    }

    /**
     * Render the Profile Phone field in the frontend
     *
     * @since 4.2.3
     *
     * @param array   $field_settings
     * @param integer $form_id
     * @param string  $type
     * @param integer $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        $value = $this->get_profile_field_value( $field_settings, $post_id );
        ?>

        <style>
            .iti--allow-dropdown.has-error .iti__selected-flag {
                height: 70%;
                padding-left: 16px;
            }

            /* field specific style to override our default padding */
            .iti--allow-dropdown input, .iti--allow-dropdown input[type=text], .iti--allow-dropdown input[type=tel], .iti--separate-dial-code input, .iti--separate-dial-code input[type=text], .iti--separate-dial-code input[type=tel] {
                padding-right: 6px !important;
                padding-left: 52px !important;
            }

        </style>
        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <?php $this->print_label( $field_settings, $form_id ); ?>
            <div class="wpuf-fields">
                <input
                    class="wpuf_telephone text <?php echo 'wpuf_' . esc_attr( $field_settings['name'] ) . '_' . esc_attr( $form_id ); ?>"
                    id="<?php echo esc_attr( $field_settings['name'] . '_' . $form_id ); ?>"
                    type="text"
                    data-required="<?php echo esc_attr( $field_settings['required'] ); ?>"
                    data-label="<?php echo esc_attr( $field_settings['label'] ); ?>"
                    data-show-list="<?php echo esc_attr( $field_settings['show_country_list'] ); ?>"
                    data-type="text"
                    name="<?php echo esc_attr( $field_settings['name'] ); ?>"
                    placeholder="<?php echo esc_attr( $field_settings['placeholder'] ); ?>"
                    value="<?php echo esc_attr( $value ); ?>"
                    size="<?php echo esc_attr( $field_settings['size'] ); ?>" />
                <?php $this->help_text( $field_settings ); ?>
            </div>
        </li>

        <?php
        $show_list = 'no';

        if ( ! empty( $field_settings['show_country_list'] ) && 'yes' === $field_settings['show_country_list'] ) {
            $show_list = $field_settings['show_country_list'];
        }
        ?>
        <script>
            window.addEventListener('DOMContentLoaded', (event) => {
                ;(function ($) {
                    let showList = "<?php echo esc_js( $show_list ); ?>";
                    let defaultCountry = "<?php echo ! empty( $field_settings['country_list']['name'] ) ? esc_js( $field_settings['country_list']['name'] ) : ''; ?>";
                    let onlyCountries = <?php echo ! empty( $field_settings['country_list']['country_select_show_list'] ) ? wp_json_encode( $field_settings['country_list']['country_select_show_list'] ) : "''"; ?>;
                    let excludeCountries = <?php echo ! empty( $field_settings['country_list']['country_select_hide_list'] ) ? wp_json_encode( $field_settings['country_list']['country_select_hide_list'] ) : "''"; ?>;
                    let autoPlaceholder = "<?php echo ! empty( $field_settings['auto_placeholder'] ) ? esc_js( $field_settings['auto_placeholder'] ) : ''; ?>";

                    let utilsScript = "<?php echo esc_js( WPUF_PRO_ROOT_URI . '/assets/vendor/intl-tel-input/js/utils.js' ); ?>";

                    if ( 'yes' === showList ) {
                        let tempTelObj = {
                            utilsScript: utilsScript
                        };

                        if ( '' !== defaultCountry ) {
                            tempTelObj.initialCountry = defaultCountry;
                        }

                        if ( '' !== onlyCountries ) {
                            tempTelObj.onlyCountries = onlyCountries;
                        }

                        if ( '' !== excludeCountries ) {
                            tempTelObj.excludeCountries = excludeCountries;
                        }

                        if ( 'no' === autoPlaceholder ) {
                            tempTelObj.autoPlaceholder = 'off';
                        }

                        let fieldId = "<?php echo esc_js( $field_settings['name'] . '_' . $form_id ); ?>";
                        let input = document.getElementById(fieldId); // intlTelInput not works properly if we use jQuery selector
                        window.intlTelInput(input, tempTelObj);
                    }
                })(jQuery);
            });
        </script>

        <?php
    }

    /**
     * Get field options setting
     *
     * @since 4.2.3
     *
     * @return array
     */
    public function get_options_settings() {
        $default_options = $this->get_default_option_settings( false, [ 'dynamic', 'width' ] );
        $text_options    = $this->get_default_text_option_settings();

        $settings = [
            $this->get_meta_key_setting( 'wpuf_profile_phone', 11 ),
            [
                'name'      => 'show_country_list',
                'title'     => __( 'Show Country List', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                    'no'  => __( 'No', 'wpuf-pro' ),
                ],
                'inline'    => true,
                'default'   => 'yes',
                'section'   => 'advanced',
                'priority'  => 23,
                'help_text' => __( 'Select yes to show the country selection dropdown.', 'wpuf-pro' ),
            ],
            [
                'name'      => 'auto_placeholder',
                'title'     => __( 'Auto Placeholder', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                    'no'  => __( 'No', 'wpuf-pro' ),
                ],
                'inline'    => true,
                'default'   => 'yes',
                'section'   => 'advanced',
                'priority'  => 23.1,
                'help_text' => __( 'Set the input\'s placeholder to an example number for the selected country, and update it if the country changes.', 'wpuf-pro' ),
                'dependencies'  => [
                    'show_country_list' => 'yes',
                ],
            ],
            [
                'name'          => 'country_list',
                'title'         => '',
                'type'          => 'country-list',
                'section'       => 'advanced',
                'priority'      => 23.2,
                'dependencies'  => [
                    'show_country_list' => 'yes',
                ],
                'help_text' => __( 'You must include the Default Country if you choose "Only Show These" option', 'wpuf-pro' ),
            ],
        ];

        return array_merge( $default_options, $text_options, $settings );
    }

    /**
     * Get the field props for this Vue Component
     *
     * @since 4.2.3
     *
     * @return array
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'        => 'profile_phone',
            'show_country_list' => 'yes',
            'auto_placeholder'  => 'yes',
            'country_list'  => [
                'name'                              => '',
                'country_list_visibility_opt_name'  => 'all', // all, hide, show
                'country_select_show_list'          => [],
                'country_select_hide_list'          => [],
            ],
            'name'              => 'wpuf_profile_phone',
            'is_meta'           => 'yes',
            'show_in_post'      => 'yes',
            'hide_field_label'  => 'no',
            'id'                => 0,
            'is_new'            => true,
        ];

        return array_merge( $defaults, $props );
    }
}

