<?php

namespace WeDevs\Wpuf\Pro;

use WeDevs\Wpuf\Pro\Admin\FormBuilder;
use WeDevs\Wpuf\Pro\Admin\FormBuilder\Form_Element;
use WeDevs\WpUtils\ContainerTrait;

/**
 * Pro features for wpuf_forms builder
 *
 * @since 2.5
 */
class Post_Form {

    use ContainerTrait;

    /**
     * Class constructor
     *
     * @since 2.5
     *
     * @return void
     */
    public function __construct() {
        $this->render_form_element = new FormBuilder\Render_Form_Element();
        $this->event               = new FormBuilder\Events_Plugins_Integration();

        $this->gmap_api_key = wpuf_get_option( 'gmap_api_key', 'wpuf_general' );

        $this->init_post_form_items();

        add_action( 'wpuf_load_post_forms', [ $this, 'enqueue_scripts' ] );
        // add_action( 'wpuf-form-builder-enqueue-after-components', array( $this, 'admin_enqueue_scripts_components' ) );
    }

    public function init_post_form_items() {
        // $this->enqueue_scripts();
        add_filter( 'wpuf_form_builder_localize_script', [ $this, 'localize_script' ] );
        add_filter( 'wpuf_form_builder_wpuf_forms_js_deps', [ $this, 'wpuf_forms_pro_scripts' ] );
        add_filter( 'wpuf_form_builder_js_builder_stage_mixins', [ $this, 'add_builder_stage_mixins' ] );
        add_filter( 'wpuf_form_builder_js_form_fields_mixins', [ $this, 'add_form_field_mixins' ] );
        add_filter( 'wpuf_form_builder_i18n', [ $this, 'i18n' ] );
        add_filter( 'wpuf-get-form-fields', [ $this, 'get_form_fields' ] );

        add_action( 'wpuf_form_builder_enqueue_after_mixins', [ $this, 'admin_enqueue_mixins' ] );
        add_action( 'wpuf_form_builder_enqueue_after_components', [ $this, 'admin_enqueue_components' ] );
        add_action( 'wpuf_form_builder_add_js_templates', [ $this, 'add_js_templates' ] );
        // render element form in backend form builder
        add_action( 'wpuf_admin_field_custom_repeater', [ $this, 'custom_repeater' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_repeat_field', [ $this, 'repeat_field' ], 10, 5 );
        add_action( 'wpuf_admin_field_custom_date', [ $this, 'custom_date' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_date_field', [ $this, 'date_field' ], 10, 5 );
        add_action( 'wpuf_admin_field_custom_file', [ $this, 'custom_file' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_file_upload', [ $this, 'file_upload' ], 10, 5 );
        add_action( 'wpuf_admin_field_custom_map', [ $this, 'custom_map' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_google_map', [ $this, 'google_map' ], 10, 5 );
        add_action( 'wpuf_admin_field_country_select', [ $this, 'country_select' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_country_list_field', [ $this, 'country_list' ], 10, 5 );
        add_action( 'wpuf_admin_field_numeric_field', [ $this, 'wpuf_admin_field_numeric_field_runner' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_numeric_text_field', [ $this, 'wpuf_admin_template_post_numeric_text_field_runner' ], 10, 5 );
        add_action( 'wpuf_admin_field_address_field', [ $this, 'wpuf_admin_field_address_field_runner' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_address_field', [ $this, 'wpuf_admin_template_post_address_field_runner' ], 10, 5
        );
        add_action( 'wpuf_admin_field_step_start', [ $this, 'wpuf_admin_field_step_start_runner' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_step_start', [ $this, 'wpuf_admin_template_post_step_start_runner' ], 10, 5 );
        add_action( 'wpuf_admin_field_really_simple_captcha', [ $this, 'wpuf_admin_field_really_simple_captcha_runner' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_really_simple_captcha', [ $this, 'wpuf_admin_template_post_really_simple_captcha_runner' ], 10, 5 );
        add_action( 'wpuf_admin_field_action_hook', [ $this, 'wpuf_admin_field_action_hook_runner' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_action_hook', [ $this, 'wpuf_admin_template_post_action_hook_runner' ], 10, 5 );
        add_action( 'wpuf_admin_field_toc', [ $this, 'wpuf_admin_field_toc_runner' ], 10, 4 );
        add_action( 'wpuf_admin_field_ratings', [ $this, 'wpuf_admin_field_ratings_runner' ], 10, 4 );
        add_action( 'wpuf_admin_template_post_toc', [ $this, 'wpuf_admin_template_post_toc_runner' ], 10, 5 );
        add_action( 'wpuf_admin_template_post_ratings', [ $this, 'wpuf_admin_template_post_ratings' ], 10, 5 );
        // others
        add_action( 'wpuf_form_buttons_custom', [ $this, 'wpuf_form_buttons_custom_runner' ] );
        add_action( 'wpuf_form_buttons_other', [ $this, 'wpuf_form_buttons_other_runner' ] );
        add_action( 'wpuf_form_post_expiration', [ $this, 'wpuf_form_post_expiration_runner' ] );
        add_action( 'wpuf_form_setting', [ $this, 'form_setting_runner' ], 10, 2 );
        add_action( 'wpuf_form_settings_post_notification', [ $this, 'post_notification_hook_runner' ] );
        add_action( 'wpuf_after_post_form_settings_field_limit_message', [ $this, 'add_submit_button_conditional_field' ], 10, 3 );
        add_action( 'wpuf_edit_form_area_profile', [ $this, 'wpuf_edit_form_area_profile_runner' ] );
        add_action( 'wpuf_add_profile_form_top', [ $this, 'wpuf_add_profile_form_top_runner' ], 10, 2 );
        add_action( 'registration_setting', [ $this, 'registration_setting_runner' ] );
        add_action( 'wpuf_check_post_type', [ $this, 'wpuf_check_post_type_runner' ], 10, 2 );
        add_action( 'wpuf_form_custom_taxonomies', [ $this, 'wpuf_form_custom_taxonomies_runner' ] );
        add_action( 'wpuf_conditional_field_render_hook', [ $this, 'wpuf_conditional_field_render_hook_runner' ], 10, 3 );
        add_action( 'wpuf_submit_btn', [ $this->render_form_element, 'conditional_logic_on_submit_button' ], 10, 2 );
        add_filter( 'wpuf_get_post_types', [ $this, 'add_custom_post_types' ] );

        // post form settings
        add_filter( 'wpuf_form_builder_settings_general', [ $this, 'form_settings_general' ] );
        add_filter( 'wpuf_form_builder_settings_notification', [ $this, 'form_settings_notification' ] );
        add_filter( 'wpuf_form_builder_settings_display', [ $this, 'form_settings_display' ] );
        add_filter( 'wpuf_form_builder_settings_payment', [ $this, 'form_settings_payment' ] );
        add_filter( 'wpuf_form_builder_post_settings_menu_items', [ $this, 'form_settings_modules' ] );
        add_filter( 'wpuf_form_builder_settings_post_expiration', [ $this, 'form_settings_post_expiration' ] );
        add_filter( 'wpuf_form_builder_settings_advanced', [ $this, 'form_settings_advanced' ] );
    }

    /**
     * Load the pro settings for the post form
     *
     *
     * @return void
     */
    public function form_settings_advanced( $settings ) {
        return $settings;
    }

    /**
     * Add payment settings pro fields
     *
     * @since 4.2.2
     *
     * @param array $payment_settings Payment settings array.
     *
     * @return array
     */
    public function form_settings_payment( $payment_settings ) {
        $payment_settings['enable_pricing_payment'] = [
            'label'     => __( 'Enable Pricing Fields Payment', 'wpuf-pro' ),
            'type'      => 'toggle',
            'help_text' => __(
                'Charge users based on pricing field selections (checkbox, radio, dropdown with prices). The total will be calculated automatically from selected options.',
                'wpuf-pro'
            ),
        ];

        return $payment_settings;
    }

    /**
     * Add post expiration settings form styles
     *
     * @since 4.1.0
     *
     * @param array $post_expiration
     *
     * @return array
     */
    public function form_settings_post_expiration( $post_expiration ) {
        $post_expiration['enable_post_expiration']    = [
            'id'=> 'wpuf-enable_post_expiration',
            'name'=> 'post_expiration_settings[enable_post_expiration]',
            'label' => __( 'Enable Post Expiration', 'wpuf-pro' ),
            'type'  => 'toggle',
            'db_key'  => '_enable_post_expiration',
            'db_type'  => 'meta',
            'value' => 'on',
            'help_text' => __( 'Set an expiration date for posts to automatically unpublish them after a certain period', 'wpuf-pro' ),
        ];
        $post_expiration['inline_fields']['fields']   = [
            'expiration_time_value' => [
                'id'      => 'wpuf-expiration_time_value',
                'name'    => 'post_expiration_settings[expiration_time_value]',
                'label'   => __( 'Post Expiration Time', 'wpuf-pro' ),
                'type'    => 'number',
                'db_key'  => '_post_expiration_time',
                'db_type' => 'meta',
                'value'   => '7',
            ],
            'expiration_time_type'  => [
                'label'   => __( 'Duration Type', 'wpuf-pro' ),
                'type'    => 'select',
                'id'      => 'wpuf-expiration_time_type',
                'name'    => 'post_expiration_settings[expiration_time_type]',
                'db_key'  => '_post_expiration_time',
                'db_type' => 'meta',
                'options' => [
                    'day'   => __( 'Day(s)', 'wpuf-pro' ),
                    'week'  => __( 'Week(s)', 'wpuf-pro' ),
                    'month' => __( 'Month(s)', 'wpuf-pro' ),
                ],
            ],
        ];
        $post_expiration['expired_post_status']       = [
            'label'     => __( 'Post Status', 'wpuf-pro' ),
            'type'      => 'select',
            'help_text' => __( 'Status of post after post expiration time is over', 'wpuf-pro' ),
            'options'   => [
                'draft' => __( 'Draft', 'wpuf-pro' ),
                'pending' => __( 'Pending Review', 'wpuf-pro' ),
                'private' => __( 'Private', 'wpuf-pro' ),
                'publish' => __( 'Published', 'wpuf-pro' ),
            ],
        ];
        $post_expiration['enable_mail_after_expired'] = [
            'label'     => __( 'Send post expiration email to author', 'wpuf-pro' ),
            'type'      => 'checkbox',
            'help_text' => __( 'Send email to author after exceeding post expiration time', 'wpuf-pro' ),
        ];
        $post_expiration['post_expiration_message']   = [
            'label'     => __( 'Post Expired Message', 'wpuf-pro' ),
            'type'      => 'textarea',
            'help_text' => __(
                'Display a custom message to notify users about the post\'s expiration status.', 'wpuf-pro'
            ),
        ];

        return $post_expiration;
    }

    /**
     * Add module settings
     *
     * @since 4.1.0
     *
     * @param array $settings
     *
     * @return array
     */
    public function form_settings_modules( $settings ) {
        $settings['modules'] = [
            'label'     => __( 'Modules', 'wpuf-pro' ),
            'icon'      => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M6.42857 9.75L2.25 12L6.42857 14.25M6.42857 9.75L12 12.75L17.5714 9.75M6.42857 9.75L2.25 7.5L12 2.25L21.75 7.5L17.5714 9.75M17.5714 9.75L21.75 12L17.5714 14.25M17.5714 14.25L21.75 16.5L12 21.75L2.25 16.5L6.42857 14.25M17.5714 14.25L12 17.25L6.42857 14.25" stroke="#9CA3AF" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>',
            'sub_items' => apply_filters( 'wpuf_post_form_module_settings_menu_titles', [] ),
        ];

        return $settings;
    }

    /**
     * Add display settings form styles
     *
     * @since 4.1.0
     *
     * @param array $display_settings
     *
     * @return array
     */
    public function form_settings_display( $display_settings ) {
        $display_settings['section']['custom_form_style']['fields']['form_layout'] = [
            'label'     => __( 'Choose Form Style', 'wpuf-pro' ),
            'type'      => 'pic-radio',
            'help_text' => __(
                'If selected a form template, it will try to execute that integration options when new post created and updated.',
                'wpuf-pro'
            ),
            'options'   => [
                'layout1' => [
                    'label' => __( 'Post Form', 'wpuf-pro' ),
                    'image' => WPUF_ASSET_URI . '/images/forms/layout1.png',
                ],
                'layout2' => [
                    'label' => __( 'Post Form', 'wpuf-pro' ),
                    'image' => WPUF_ASSET_URI . '/images/forms/layout2.png',
                ],
                'layout3' => [
                    'label' => __( 'Post Form', 'wpuf-pro' ),
                    'image' => WPUF_ASSET_URI . '/images/forms/layout3.png',
                ],
                'layout4' => [
                    'label' => __( 'Post Form', 'wpuf-pro' ),
                    'image' => WPUF_ASSET_URI . '/images/forms/layout4.png',
                ],
                'layout5' => [
                    'label' => __( 'Post Form', 'wpuf-pro' ),
                    'image' => WPUF_ASSET_URI . '/images/forms/layout5.png',
                ],
            ],
        ];

        return $display_settings;
    }

    /**
     * Add notification settings pro fields preview
     *
     * @since WPUF_SINCE
     *
     * @param array $notification_settings
     *
     * @return array
     */
    public function form_settings_notification( $notification_settings ) {
        // Check if this is an Event Calendar form
        global $post;
        $form_settings = $post ? wpuf_get_form_settings( $post->ID ) : [];
        $is_event_form = ! empty( $form_settings['form_template'] ) && 'post_form_template_events_calendar' === $form_settings['form_template'];

        // Check if values exist in database, if not use appropriate defaults
        if ( $is_event_form ) {
            // Override NEW post notification for Event Calendar forms if not already set
            if ( isset( $notification_settings['section']['new_post']['fields']['new_subject'] ) ) {
                // Check if database has a value
                if ( empty( $form_settings['notification']['new_subject'] ) ) {
                    $notification_settings['section']['new_post']['fields']['new_subject']['value'] = __( 'New event has been created', 'wpuf-pro' );
                }
            }
            if ( isset( $notification_settings['section']['new_post']['fields']['new_body'] ) ) {
                // Check if database has a value
                if ( empty( $form_settings['notification']['new_body'] ) ) {
                    $notification_settings['section']['new_post']['fields']['new_body']['value'] = "Hi,\r\nA new event has been created in your site {sitename} ({siteurl}).\r\n\r\nHere are the details:\r\nEvent Title: {post_title}\r\nDescription: {post_content}\r\nShort Description: {post_excerpt}\r\nAuthor: {author}\r\nPost URL: {permalink}\r\nEdit URL: {editlink}";
                }
            }

            // Set edit notification defaults if not in database
            if ( empty( $form_settings['notification']['edit_subject'] ) ) {
                $edit_subject = __( 'An event has been updated', 'wpuf-pro' );
            } else {
                $edit_subject = $form_settings['notification']['edit_subject'];
            }

            if ( empty( $form_settings['notification']['edit_body'] ) ) {
                $edit_body = "Hi Admin,\r\n\r\nThe event \"{post_title}\" has been updated.\r\n\r\nHere are the event details:\r\nEvent Title: {post_title}\r\nEvent Start: {custom__EventStartDate}\r\nEvent End: {custom__EventEndDate}\r\nEvent Website: {custom__EventURL}\r\nEvent Cost: {custom__EventCost}\r\nDescription: {post_content}\r\nShort Description: {post_excerpt}\r\nAuthor: {author}\r\nEvent URL: {permalink}\r\nEdit URL: {editlink}";
            } else {
                $edit_body = $form_settings['notification']['edit_body'];
            }
        } else {
            // For regular posts, use database values if they exist, otherwise use defaults
            if ( ! empty( $form_settings['notification']['edit_subject'] ) ) {
                $edit_subject = $form_settings['notification']['edit_subject'];
            } else {
                $edit_subject = __( 'A post has been edited', 'wpuf-pro' );
            }

            if ( ! empty( $form_settings['notification']['edit_body'] ) ) {
                $edit_body = $form_settings['notification']['edit_body'];
            } else {
                $edit_body = "Hi Admin, \r\n\r\nThe post \"{post_title}\" has been updated. \r\n\r\nHere is the details: \r\nPost Title: {post_title} \r\nContent: {post_content} \r\nAuthor: {author} \r\nPost URL: {permalink} \r\nEdit URL: {editlink}";
            }
        }

        $notification_settings['section']['update_post']['fields']['notification_edit']         = [
            'label' => __( 'Enable Update Post Notification', 'wpuf-pro' ),
            'type'  => 'toggle',
            'help_text' => __(
                'Enable email alerts when a post is updated through the frontend form', 'wpuf-pro'
            ),
        ];
        $notification_settings['section']['update_post']['fields']['notification_edit_to']      = [
            'label' => __( 'To', 'wpuf-pro' ),
            'type'  => 'text',
            'value' => get_option( 'admin_email' ),
        ];
        $notification_settings['section']['update_post']['fields']['notification_edit_subject'] = [
            'label' => __( 'Subject', 'wpuf-pro' ),
            'type'  => 'text',
            'value' => $edit_subject,
        ];
        $notification_settings['section']['update_post']['fields']['notification_edit_body']    = [
            'label'     => __( 'Email Body', 'wpuf-pro' ),
            'type'      => 'textarea',
            'value'     => $edit_body,
            'long_help' => '<h4 class="wpuf-m-0">You may use in to, subject & message:</h4>
                                         <p class="wpuf-leading-8">
                                         <span data-clipboard-text="{post_title}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{post_title}</span>
                                         <span data-clipboard-text="{post_content}" class="wpuf-post-content wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{post_content}</span>
                                         <span data-clipboard-text="{post_excerpt}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{post_excerpt}</span>
                                         <span data-clipboard-text="{tags}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{tags}</span>
                                         <span data-clipboard-text="{category}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{category}</span>
                                         <span data-clipboard-text="{author}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{author}</span>
                                         <span data-clipboard-text="{author_email}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{author_email}</span>
                                         <span data-clipboard-text="{author_bio}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{author_bio}</span>
                                         <span data-clipboard-text="{sitename}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{sitename}</span>
                                         <span data-clipboard-text="{siteurl}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{siteurl}</span>
                                         <span data-clipboard-text="{permalink}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{permalink}</span>
                                         <span data-clipboard-text="{editlink}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{editlink}</span>
                                         <span class="wpuf-pill-green">{custom_{NAME_OF_CUSTOM_FIELD}}</span>
                                         e.g: <span class="wpuf-pill-green">{custom_website_url}</span> for <i>website_url meta</i> field</p>',
        ];

        return $notification_settings;
    }

    /**
     * Add general settings pro fields preview
     *
     * @since WPUF_SINCE
     *
     * @param array $general_settings
     *
     * @return array
     */
    public function form_settings_general( $general_settings ) {
        $general_settings['section']['before_post_settings']['fields']['enable_multistep']           = [
            'label'     => __( 'Enable Multi-Step', 'wpuf-pro' ),
            'type'      => 'toggle',
            'help_text' => __(
                'Split the form into multiple steps for a better user experience', 'wpuf-pro'
            ),
            'link'      => esc_url_raw(
                'https://wedevs.com/docs/wp-user-frontend-pro/posting-forms/how-to-add-multi-step-form/'
            ),
        ];
        $general_settings['section']['before_post_settings']['fields']['multistep_progressbar_type'] = [
            'label'     => __( 'Multistep Progressbar Type', 'wpuf-pro' ),
            'type'      => 'select',
            'help_text' => __( 'Select how to display progress in a multi-step form', 'wpuf-pro' ),
            'options'   => [
                'progressive'  => __( 'Progressbar', 'wpuf-pro' ),
                'step_by_step' => __( 'Step by Step', 'wpuf-pro' ),
            ],
        ];
        $general_settings['section']['before_post_settings']['fields']['ms_ac_txt_color']            = [
            'label'     => __( 'Active Text Color', 'wpuf-pro' ),
            'type'      => 'color-picker',
            'help_text' => __( 'Set the color of the text for the active step in the progress bar', 'wpuf-pro' ),
            'default'   => '#fff',
        ];
        $general_settings['section']['before_post_settings']['fields']['ms_active_bgcolor']          = [
            'label'     => __( 'Active Background Color', 'wpuf-pro' ),
            'type'      => 'color-picker',
            'help_text' => __( 'Set the background color for the active step in the progress bar', 'wpuf-pro' ),
            'default'   => '#00a0d2',
        ];
        $general_settings['section']['before_post_settings']['fields']['ms_bgcolor']                 = [
            'label'     => __( 'Background Color', 'wpuf-pro' ),
            'type'      => 'color-picker',
            'help_text' => __( 'Set the background color for inactive steps in the progress bar', 'wpuf-pro' ),
            'default'   => '#E4E4E4',
        ];

         return $general_settings;
    }

    public function custom_map( $type, $field_id, $classname, $obj ) {
        Form_Element::google_map( $field_id, 'Custom Field: Google Map', $classname );
    }

    public function google_map( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::google_map( $count, $name, $classname, $input_field );
    }

    public function country_select( $type, $field_id, $classname, $obj ) {
        Form_Element::country_list_field( $field_id, 'Custom field: Select', $classname );
    }

    public function country_list( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::country_list_field( $count, $name, $classname, $input_field );
    }

    public function wpuf_admin_field_numeric_field_runner( $type, $field_id, $classname, $obj ) {
        Form_Element::numeric_text_field( $field_id, 'Custom field: Numeric Text', $classname );
    }

    public function wpuf_admin_template_post_numeric_text_field_runner( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::numeric_text_field( $count, $name, $classname, $input_field );
    }

    public function wpuf_admin_field_address_field_runner( $type, $field_id, $classname, $obj ) {
        Form_Element::address_field( $field_id, 'Custom field: Address', $classname );
    }

    public function wpuf_admin_template_post_address_field_runner( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::address_field( $count, $name, $classname, $input_field );
    }

    public function wpuf_admin_field_step_start_runner( $type, $field_id, $classname, $obj ) {
        Form_Element::step_start( $field_id, 'Step Starts', $classname );
    }

    public function wpuf_admin_template_post_step_start_runner( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::step_start( $count, $name, $classname, $input_field );
    }

    public function wpuf_admin_field_really_simple_captcha_runner( $type, $field_id, $classname, $obj ) {
        Form_Element::really_simple_captcha( $field_id, 'Really Simple Captcha', $classname );
    }

    public function wpuf_admin_template_post_really_simple_captcha_runner( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::really_simple_captcha( $count, $name, $classname, $input_field );
    }

    public function wpuf_admin_field_action_hook_runner( $type, $field_id, $classname, $obj ) {
        Form_Element::action_hook( $field_id, 'Action Hook', $classname );
    }

    public function wpuf_admin_template_post_action_hook_runner( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::action_hook( $count, $name, $classname, $input_field );
    }

    public function wpuf_admin_field_toc_runner( $type, $field_id, $classname, $obj ) {
        Form_Element::toc( $field_id, 'TOC', $classname );
    }

    public function wpuf_admin_field_ratings_runner( $type, $field_id, $classname, $obj ) {
        Form_Element::ratings( $field_id, 'Ratings', $classname );
    }

    public function wpuf_admin_template_post_toc_runner( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::toc( $count, $name, $classname, $input_field );
    }

    public function wpuf_admin_template_post_ratings( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::ratings( $count, $name, $classname, $input_field );
    }

    public function wpuf_add_profile_form_top_runner( $form_id, $form_settings ) {
        if ( isset( $form_settings['multistep_progressbar_type'] ) && $form_settings['multistep_progressbar_type'] == 'progressive' ) {
            wp_enqueue_script( 'jquery-ui-progressbar' );
        }
    }

    public function custom_date( $type, $field_id, $classname, $obj ) {
        Form_Element::date_field( $field_id, 'Custom Field: Date', $classname );
    }

    public function file_upload( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::file_upload( $count, $name, $classname, $input_field );
    }

    public function custom_file( $type, $field_id, $classname, $obj ) {
        Form_Element::file_upload( $field_id, 'Custom field: File Upload', $classname );
    }

    public function repeat_field( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::repeat_field( $count, $name, $classname, $input_field );
    }

    //form element's rendering form in backend form builder
    public function custom_repeater( $type, $field_id, $classname, $obj ) {
        Form_Element::repeat_field( $field_id, 'Custom field: Repeat Field', $classname );
    }

    public function date_field( $name, $count, $input_field, $classname, $obj ) {
        Form_Element::date_field( $count, $name, $classname, $input_field );
    }

    /**
     * Enqueue pro scripts in post forms editor page
     *
     * @since 2.5.3
     *
     * @param array $deps
     *
     * @return array
     */
    public function wpuf_forms_pro_scripts( $deps ) {
        $deps[] = 'wpuf-form-builder-wpuf-forms-pro';

        return $deps;
    }

    /**
     * Enqueue Vue components
     *
     * @since 2.5
     *
     * @return void
     */
    public function admin_enqueue_components() {
        wp_enqueue_script( 'wpuf-form-builder-components-pro' );
        wp_enqueue_script( 'wpuf-google-maps' );
    }

    /**
     * i18n translatable strings
     *
     * @since 2.5
     *
     * @param array $i18n
     *
     * @return array
     */
    public function i18n( $i18n ) {
        return array_merge( $i18n, array(
            'street_address'    => __( 'Address Line 1', 'wpuf-pro' ),
            'street_address2'   => __( 'Address Line 2', 'wpuf-pro' ),
            'city_name'         => __( 'City', 'wpuf-pro' ),
            'state'             => __( 'State', 'wpuf-pro' ),
            'zip'               => __( 'Zip Code', 'wpuf-pro' ),
            'country_select'    => __( 'Country', 'wpuf-pro' ),
            'show_all'          => __( 'Show all', 'wpuf-pro' ),
            'hide_these'        => __( 'Hide these', 'wpuf-pro' ),
            'only_show_these'   => __( 'Only show', 'wpuf-pro' ),
            'select_countries'  => __( 'Select Countries', 'wpuf-pro' ),
        ) );
    }

    /**
     * Add mixin_form_field_pro mixin
     *
     * @since 2.5
     *
     * @param array $mixins
     *
     * @return array
     */
    public function add_builder_stage_mixins( $mixins ) {
        return array_merge( $mixins, array( 'mixin_form_field_pro', 'mixin_builder_stage_pro' ) );
    }

    public function form_setting_runner( $form_settings, $post ) {
        Form_Element::add_form_settings_content( $form_settings, $post );
    }

    /**
     * Add submit button conditional logic field using Vue component
     *
     * @since 4.1.0
     *
     * @param $field
     * @param $value
     * @param $form_settings
     *
     * @return void
     */
    public function add_submit_button_conditional_field( $field, $value, $form_settings ) {
        $conditional_logic_settings = isset( $form_settings['submit_button_cond'] )
            ? $form_settings['submit_button_cond']
            : [
                'condition_status' => 'no',
                'cond_logic' => 'any',
                'conditions' => []
            ];
        ?>
        <submit-button-conditional-logics
            :current-settings="<?php echo esc_attr( wp_json_encode( $conditional_logic_settings ) ); ?>"
            label="<?php esc_attr_e( 'Conditional Logic on Submit Button', 'wpuf-pro' ); ?>">
        </submit-button-conditional-logics>
        <?php
    }

    public function wpuf_form_post_expiration_runner() {
        Form_Element::render_form_expiration_tab();
    }

    public function post_notification_hook_runner() {
        Form_Element::add_post_notification_content();
    }


    /**
     * Add Vue templates
     *
     * @since 2.5
     *
     * @return void
     */
    public function add_js_templates() {
        wpuf_include_once( WPUF_PRO_ROOT . '/assets/js-templates/form-components.php' );
    }

    /**
     * Add mixins to form_fields
     *
     * @since 2.5
     *
     * @param array $mixins
     *
     * @return array
     */
    public function add_form_field_mixins( $mixins ) {
        return array_merge( $mixins, array( 'mixin_form_field_pro' ) );
    }

    /**
     * Filter form fields
     *
     * @since 2.5
     *
     * @param array $field
     *
     * @return array
     */
    public function get_form_fields( $field ) {
        // make sure that country_list has all its properties
        if ( 'country_list' === $field['input_type'] ) {
            if ( ! isset( $field['country_list']['country_select_hide_list'] ) ) {
                $field['country_list']['country_select_hide_list'] = [];
            }
            if ( ! isset( $field['country_list']['country_select_show_list'] ) ) {
                $field['country_list']['country_select_show_list'] = [];
            }
        }
        if ( 'address' === $field['input_type'] ) {
            if ( ! isset( $field['address']['country_select']['country_select_hide_list'] ) ) {
                $field['address']['country_select']['country_select_hide_list'] = [];
            }
            if ( ! isset( $field['address']['country_select']['country_select_show_list'] ) ) {
                $field['address']['country_select']['country_select_show_list'] = [];
            }
        }
        if ( 'google_map' === $field['input_type'] && ! isset( $field['directions'] ) ) {
            $field['show_checkbox'] = false;
        }
        if ( 'toc' === $field['input_type'] && ! isset( $field['show_checkbox'] ) ) {
            $field['show_checkbox'] = false;
        }
        if ( 'ratings' === $field['input_type'] && ! isset( $field['selected'] ) ) {
            $field['selected'] = [];
        }

        return $field;
    }

    /**
     * Add data to localize script data array
     *
     * @since 2.5
     *
     * @param array $data
     *
     * @return array
     */
    public function localize_script( $data ) {
        if ( ! array_key_exists( 'form_settings', $data ) || ! array_key_exists( 'submit_button_cond', $data['form_settings'] ) ) {
            $data['form_settings']['submit_button_cond'] = [
                'condition_status' => 'no',
                'cond_logic'       => 'any',
                'conditions'       => [
                    [
                        'name'     => '',
                        'operator' => '=',
                        'option'   => '',
                    ],
                ],
            ];
        }

        if ( ! empty( $data['form_type'] ) && 'wpuf_profile' === $data['form_type'] ) {
            $data['settings_titles'] = wpuf_get_registration_form_builder_setting_menu_titles();
            $data['settings_items']  = wpuf_get_registration_form_builder_setting_menu_contents();
        }

        return array_merge(
            $data, [
                'gmap_api_key'               => $this->gmap_api_key,
                'is_rs_captcha_active'       => class_exists( 'ReallySimpleCaptcha' ) ? true : false,
                'countries'                  => wpuf_get_countries(),
                'wpuf_cond_supported_fields' => [
                    'radio_field',
                    'checkbox_field',
                    'dropdown_field',
                    'text_field',
                    'numeric_text_field',
                    'textarea_field',
                    'email_address',
                    'website_url',
                    'password',
                    'google_map',
                    'user_email',
                ],
            ]
        );
    }

    /**
     * Enqueue form builder related CSS and JS
     *
     * @since 4.0.0
     *
     * @return void
     */
    public function enqueue_scripts() {
        wp_enqueue_style( 'wpuf-css-stars' );
        // wp_enqueue_style( 'wpuf-math-captcha' );
        wp_enqueue_style( 'wpuf-tax' );
        // Load International Telephone Input CSS - https://github.com/jackocnr/intl-tel-input.
        wp_enqueue_style( 'wpuf-intlTelInput' );
        wp_enqueue_style( 'wpuf-admin-form-builder-pro' );
        wp_enqueue_style( 'wpuf-css-stars' );

        wp_enqueue_script( 'wpuf-barrating' );
        wp_enqueue_script( 'wpuf-conditional-logic' );
        // Load International Telephone Input JS - https://github.com/jackocnr/intl-tel-input.s
        wp_enqueue_script( 'wpuf-intlTelInput' );
        wp_enqueue_script( 'wpuf-tax' );
        wp_enqueue_script( 'wpuf-field-initialization' );
        // wp_enqueue_script( 'wpuf-form-builder-wpuf-forms-pro' );
    }

    /**
     * Enqueue Vue mixins
     *
     * @since 2.5
     *
     * @return void
     */
    public function admin_enqueue_mixins() {
        wp_enqueue_script( 'wpuf-form-builder-mixins-pro' );
    }

    /**
     * Filter to add custom post types to wpuf_get_post_types
     *
     * @since 2.5
     *
     * @param array $post_types
     *
     * @return array
     */
    public function add_custom_post_types( $post_types ) {
        $args = array( '_builtin' => false );

        $custom_post_types = get_post_types( $args );

        $ignore_post_types = array(
            'wpuf_forms', 'wpuf_profile', 'wpuf_input'
        );

        foreach ( $custom_post_types as $key => $val ) {
            if ( in_array( $val, $ignore_post_types ) ) {
                unset( $custom_post_types[$key] );
            }
        }

        return array_merge( $post_types, $custom_post_types );
    }
}
