import { __ } from '@wordpress/i18n';

const UserPosts = ({ user, attributes, context }) => {
    const {
        postCount = 5,
        postStatus = ['publish'],
        excludeCategories = [],
        order = 'DESC',
        showPostTitles = true,
        showPostDate = true,
        postLength = 100,
        showThumbnails = false,
        thumbnailSize = 'thumbnail',
        linkToPosts = true,
        dateFormat = 'relative',
        customDateFormat = 'F j, Y',
        emptyMessage = '',
        showLabel = true,
        customLabel = 'Recent Posts',
        layoutStyle = 'list'
    } = attributes || {};

    // Mock posts data for display
    const getMockPosts = () => {
        if (!user) {
            return [
                {
                    id: 1,
                    title: __('Sample Post Title', 'wpuf-pro'),
                    excerpt: __('This is a sample post excerpt that shows how user posts will be displayed in the profile.', 'wpuf-pro'),
                    date: '2024-01-12',
                    link: '#',
                    thumbnail: 'https://via.placeholder.com/150'
                },
                {
                    id: 2,
                    title: __('Another Sample Post', 'wpuf-pro'),
                    excerpt: __('Another sample post to demonstrate the layout and functionality.', 'wpuf-pro'),
                    date: '2024-01-10',
                    link: '#',
                    thumbnail: 'https://via.placeholder.com/150'
                }
            ];
        }

        // In production, this would fetch real posts from the API
        return [
            {
                id: 1,
                title: __('User Post Title', 'wpuf-pro'),
                excerpt: __('This is a user post excerpt that shows how posts will be displayed.', 'wpuf-pro'),
                date: '2024-01-12',
                link: '#',
                thumbnail: 'https://via.placeholder.com/150'
            }
        ];
    };

    const posts = getMockPosts();

    // Format date
    const formatDate = (dateString) => {
        if (dateFormat === 'relative') {
            const date = new Date(dateString);
            const now = new Date();
            const diffTime = Math.abs(now - date);
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
            
            if (diffDays === 1) return __('1 day ago', 'wpuf-pro');
            if (diffDays < 7) return `${diffDays} ${__('days ago', 'wpuf-pro')}`;
            if (diffDays < 30) return `${Math.floor(diffDays / 7)} ${__('weeks ago', 'wpuf-pro')}`;
            return date.toLocaleDateString('en-US', { 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric' 
            });
        }
        
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', { 
            year: 'numeric', 
            month: 'long', 
            day: 'numeric' 
        });
    };

    // Render post item
    const renderPostItem = (post) => {
        const displayExcerpt = post.excerpt.length > postLength 
            ? post.excerpt.substring(0, postLength) + '...' 
            : post.excerpt;

        return (
            <div key={post.id} className="wpuf-post-item">
                {showThumbnails && post.thumbnail && (
                    <div className="wpuf-post-thumbnail">
                        <img 
                            src={post.thumbnail} 
                            alt={post.title}
                            className="wpuf-post-image"
                        />
                    </div>
                )}
                <div className="wpuf-post-content">
                    {showPostTitles && (
                        <h4 className="wpuf-post-title">
                            {linkToPosts ? (
                                <a href={post.link} className="wpuf-post-link">
                                    {post.title}
                                </a>
                            ) : (
                                post.title
                            )}
                        </h4>
                    )}
                    <div className="wpuf-post-excerpt">
                        {displayExcerpt}
                    </div>
                    {showPostDate && (
                        <div className="wpuf-post-meta">
                            <span className="wpuf-post-date">
                                {formatDate(post.date)}
                            </span>
                        </div>
                    )}
                </div>
            </div>
        );
    };

    return (
        <div className={`wpuf-user-posts wpuf-layout-${layoutStyle}`}>
            {showLabel && customLabel && (
                <h3 className="wpuf-posts-label">{customLabel}</h3>
            )}

            <div className="wpuf-posts-container">
                {posts.length === 0 ? (
                    <div className="wpuf-posts-empty">
                        <p>{emptyMessage || __('No posts found.', 'wpuf-pro')}</p>
                    </div>
                ) : (
                    posts.slice(0, postCount).map(post => renderPostItem(post))
                )}
            </div>
        </div>
    );
};

export default UserPosts; 