<?php
/**
 * User Social Fields Block Template
 *
 * @since 4.2.0
 *
 * @var WP_User $user User object
 * @var array $attributes Block attributes
 * @var string $content Block content
 * @var WP_Block $block Block instance
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! $user ) {
    return;
}

// Get block attributes with defaults
$show_facebook = isset( $attributes['showFacebook'] ) ? $attributes['showFacebook'] : true;
$show_twitter = isset( $attributes['showTwitter'] ) ? $attributes['showTwitter'] : true;
$show_linkedin = isset( $attributes['showLinkedIn'] ) ? $attributes['showLinkedIn'] : true;
$show_instagram = isset( $attributes['showInstagram'] ) ? $attributes['showInstagram'] : true;
$icon_size = isset( $attributes['iconSize'] ) ? $attributes['iconSize'] : 'medium';
$layout_style = isset( $attributes['layoutStyle'] ) ? $attributes['layoutStyle'] : 'layout-1';
$background_color = isset( $attributes['backgroundColor'] ) ? $attributes['backgroundColor'] : '#000000';
$icon_color = isset( $attributes['iconColor'] ) ? $attributes['iconColor'] : '#3B82F6';

// Get social media usernames
$facebook_username = get_user_meta( $user->ID, 'wpuf_social_facebook', true );
$twitter_username = get_user_meta( $user->ID, 'wpuf_social_twitter', true );
$linkedin_username = get_user_meta( $user->ID, 'wpuf_social_linkedin', true );
$instagram_username = get_user_meta( $user->ID, 'wpuf_social_instagram', true );

// Check if any social fields should be displayed
$has_social_data = ( $show_facebook && ! empty( $facebook_username ) )
    || ( $show_twitter && ! empty( $twitter_username ) )
    || ( $show_linkedin && ! empty( $linkedin_username ) )
    || ( $show_instagram && ! empty( $instagram_username ) );

if ( ! $has_social_data ) {
    return;
}

// Social media icons for Layout 1 (original)
$social_icons_layout1 = [
    'facebook' => '<svg width="25" height="24" viewBox="0 0 25 24" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M12.5 24C19.1274 24 24.5 18.6274 24.5 12C24.5 5.37258 19.1274 0 12.5 0C5.87258 0 0.5 5.37258 0.5 12C0.5 18.6274 5.87258 24 12.5 24Z" fill="#3B82F6"/>
        <path d="M13.5463 18.749V13.274H15.3837L15.659 11.1395H13.5463V9.77829C13.5463 9.15879 13.7158 8.74031 14.6053 8.74031H15.7303V6.83377C15.1842 6.77539 14.6354 6.74712 14.0862 6.74904C13.7126 6.7217 13.3374 6.77643 12.9872 6.90939C12.6369 7.04235 12.3199 7.25032 12.0585 7.51872C11.7971 7.78712 11.5976 8.10947 11.474 8.46313C11.3503 8.8168 11.3056 9.19321 11.3428 9.56602V11.141H9.5V13.274H11.3428V18.749H13.5463Z" fill="white"/>
    </svg>',
    'twitter' => '<svg width="25" height="25" viewBox="0 0 25 25" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M24.5 12.4531C24.5 19.0531 19.1 24.4531 12.5 24.4531C5.9 24.4531 0.5 19.0531 0.5 12.4531C0.5 5.85313 5.9 0.453125 12.5 0.453125C19.1 0.453125 24.5 5.85313 24.5 12.4531Z" fill="#111827"/>
        <path d="M11.4647 12.5013L8.72622 8.45312H10.8932L12.56 11.1678L14.2864 8.45312H16.3343L13.6078 12.5132L16.5724 16.9542H14.346L12.4886 14.0133L10.5955 16.9542H8.5L11.4647 12.5013Z" fill="white"/>
    </svg>',
    'linkedin' => '<svg width="24" height="25" viewBox="0 0 24 25" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M24 12.4531C24 19.0531 18.6 24.4531 12 24.4531C5.4 24.4531 0 19.0531 0 12.4531C0 5.85313 5.4 0.453125 12 0.453125C18.6 0.453125 24 5.85313 24 12.4531Z" fill="#0284C7"/>
        <path d="M8.7498 7.65078C8.7498 8.23488 8.3593 8.73758 7.64925 8.73758C6.981 8.73758 6.5498 8.26348 6.5498 7.67938C6.5498 7.07988 6.9678 6.55078 7.6498 6.55078C8.3318 6.55078 8.7366 7.05128 8.7498 7.65078ZM6.5498 16.4508V9.30078H8.7498V16.4508H6.5498ZM9.8498 11.745C9.8498 10.8952 9.82175 10.1852 9.7937 9.57193H11.7704L11.8683 10.5196H11.9101C12.1884 10.0735 12.8704 9.41958 14.0122 9.41958C15.4048 9.41958 16.4498 10.3518 16.4498 12.3577V16.4508H14.2498V12.6772C14.2498 11.8005 13.916 11.1317 13.1498 11.1317C12.5657 11.1317 12.2731 11.6047 12.1197 11.9947C12.0636 12.1344 12.0498 12.3291 12.0498 12.5243V16.4508H9.8498V11.745Z" fill="white"/>
    </svg>',
    'instagram' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M24 12C24 18.6 18.6 24 12 24C5.4 24 0 18.6 0 12C0 5.4 5.4 0 12 0C18.6 0 24 5.4 24 12Z" fill="url(#paint0_linear_5793_2712)"/>
<path d="M12 7.08093C13.6021 7.08093 13.7919 7.08693 14.4246 7.11583C15.0096 7.14252 15.3273 7.24016 15.5387 7.32239C15.8189 7.43131 16.0187 7.56116 16.2287 7.77112C16.4386 7.98109 16.5687 8.18092 16.6774 8.46107C16.7596 8.67248 16.8573 8.99016 16.884 9.57522C16.9128 10.2079 16.9189 10.3977 16.9189 11.9998C16.9189 13.602 16.9129 13.7916 16.884 14.4243C16.8573 15.0094 16.7596 15.3271 16.6774 15.5385C16.5685 15.8187 16.4387 16.0185 16.2287 16.2284C16.0187 16.4384 15.8189 16.5685 15.5387 16.6772C15.3273 16.7594 15.0096 16.857 14.4246 16.8837C13.7919 16.9126 13.6022 16.9186 12 16.9186C10.3978 16.9186 10.2082 16.9126 9.57546 16.8837C8.9904 16.857 8.67271 16.7594 8.46131 16.6772C8.18112 16.5682 7.98133 16.4384 7.77136 16.2284C7.56144 16.0185 7.43131 15.8186 7.32263 15.5385C7.24035 15.3271 7.14276 15.0094 7.11607 14.4243C7.08722 13.7917 7.08117 13.6019 7.08117 11.9998C7.08117 10.3977 7.08717 10.208 7.11607 9.57522C7.14276 8.99016 7.2404 8.67248 7.32263 8.46107C7.43155 8.18088 7.5614 7.98109 7.77136 7.77112C7.98133 7.5612 8.18116 7.43107 8.46131 7.32239C8.67271 7.24016 8.9904 7.14252 9.57546 7.11583C10.2081 7.08722 10.3979 7.08093 12 7.08093ZM12 6C10.3705 6 10.1661 6.00696 9.52616 6.03605C8.88743 6.06514 8.45142 6.16667 8.0698 6.315C7.67516 6.46841 7.34068 6.67358 7.0071 7.0071C6.67358 7.34068 6.46822 7.67516 6.315 8.0698C6.16662 8.45147 6.06514 8.88767 6.03605 9.52616C6.00696 10.1661 6 10.3705 6 12C6 13.6295 6.00696 13.8339 6.03605 14.4738C6.06514 15.1126 6.16667 15.5486 6.315 15.9302C6.46841 16.3248 6.67358 16.6593 7.0071 16.9929C7.34068 17.3265 7.6754 17.5316 8.0698 17.685C8.45147 17.8334 8.88767 17.9349 9.52616 17.964C10.1661 17.993 10.3705 18 12 18C13.6295 18 13.8339 17.993 14.4738 17.964C15.1126 17.9349 15.5486 17.8333 15.9302 17.685C16.3248 17.5315 16.6593 17.3264 16.9929 16.9929C17.3265 16.6593 17.5316 16.3246 17.685 15.9302C17.8334 15.5486 17.9349 15.1123 17.964 14.4738C17.993 13.8339 18 13.6295 18 12C18 10.3705 17.993 10.1661 17.964 9.52616C17.9349 8.88743 17.8333 8.45142 17.685 8.0698C17.5315 7.67516 17.3264 7.34068 16.9929 7.0071C16.6593 6.67358 16.3246 6.46841 15.9302 6.315C15.5486 6.16662 15.1123 6.06514 14.4738 6.03605C13.834 6.00696 13.6295 6 12 6ZM12 8.91897C10.2983 8.91897 8.91897 10.2983 8.91897 12C8.91897 13.7017 10.2983 15.0811 12 15.0811C13.7017 15.0811 15.0811 13.7017 15.0811 12C15.0811 10.2983 13.7017 8.91897 12 8.91897ZM12 14.0001C10.8955 14.0001 9.9999 13.1048 9.9999 12C9.9999 10.8953 10.8955 9.99995 12 9.99995C13.1045 9.99995 14.0001 10.8953 14.0001 12C14.0001 13.1048 13.1045 14.0001 12 14.0001ZM15.2028 8.07728C14.805 8.07728 14.4828 8.39953 14.4828 8.79728C14.4828 9.19504 14.805 9.51733 15.2028 9.51733C15.6005 9.51733 15.9228 9.19504 15.9228 8.79728C15.9228 8.39953 15.6006 8.07728 15.2028 8.07728Z" fill="white"/>
<defs>
<linearGradient id="paint0_linear_5793_2712" x1="23.6394" y1="-4.87584e-07" x2="5.6394" y2="22" gradientUnits="userSpaceOnUse">
<stop stop-color="#B900B3"/>
<stop offset="1" stop-color="#F40002"/>
</linearGradient>
</defs>
</svg>'
];

// Social media icons for Layout 2 (new design) - with dynamic background color and icon size
$get_icon_size = function($icon_size) {
    switch ($icon_size) {
        case 'small':
            return ['width' => 14, 'height' => 15];
        case 'large':
            return ['width' => 24, 'height' => 25];
        case 'xlarge':
            return ['width' => 32, 'height' => 33];
        default: // medium
            return ['width' => 18, 'height' => 19];
    }
};

$icon_dimensions = $get_icon_size($icon_size);
$icon_width = $icon_dimensions['width'];
$icon_height = $icon_dimensions['height'];

$social_icons_layout2 = [
    'facebook' => '<svg width="' . $icon_width . '" height="' . $icon_height . '" viewBox="0 0 18 19" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M9 18.4102C13.9706 18.4102 18 14.3807 18 9.41016C18 4.43959 13.9706 0.410156 9 0.410156C4.02944 0.410156 0 4.43959 0 9.41016C0 14.3807 4.02944 18.4102 9 18.4102Z" fill="' . esc_attr($background_color) . '"/>
        <path d="M9.7847 14.4719V10.3657H11.1628L11.3693 8.76481H9.7847V7.74387C9.7847 7.27925 9.91182 6.96539 10.5789 6.96539H11.4227V5.53548C11.0132 5.4917 10.6015 5.47049 10.1897 5.47193C9.90944 5.45143 9.62807 5.49248 9.36536 5.5922C9.10266 5.69192 8.86495 5.84789 8.6689 6.0492C8.47285 6.2505 8.32322 6.49226 8.23048 6.75751C8.13773 7.02275 8.10417 7.30507 8.13208 7.58467V8.76594H6.75V10.3657H8.13208V14.4719H9.7847Z" fill="white"/>
    </svg>',
    'twitter' => '<svg width="' . $icon_width . '" height="' . $icon_height . '" viewBox="0 0 18 19" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M18 9C18 13.95 13.95 18 9 18C4.05 18 0 13.95 0 9C0 4.05 4.05 0 9 0C13.95 0 18 4.05 18 9Z" fill="' . esc_attr($background_color) . '"/>
        <path d="M8.22349 9.03609L6.16966 6H7.79487L9.04502 8.03597L10.3398 6H11.8757L9.83084 9.04502L12.0543 12.3758H10.3845L8.99145 10.1702L7.57162 12.3758H6L8.22349 9.03609Z" fill="white"/>
    </svg>',
    'linkedin' => '<svg width="' . $icon_width . '" height="' . $icon_height . '" viewBox="0 0 18 19" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M9.00003 18C7.81767 18.0033 6.64638 17.7719 5.55402 17.3194C4.46166 16.8669 3.46998 16.2023 2.63625 15.3639C1.79793 14.5301 1.13328 13.5383 0.680785 12.4459C0.228287 11.3536 -0.00311327 10.1824 3.25664e-05 9.00003C-0.00315845 7.81767 0.228185 6.64642 0.680686 5.55407C1.13319 4.46172 1.7979 3.46995 2.63625 2.6362C3.47001 1.79785 4.46167 1.13319 5.55402 0.680685C6.64637 0.228185 7.81767 -0.00315845 9.00003 3.25664e-05C10.1824 -0.00315845 11.3537 0.228185 12.446 0.680685C13.5384 1.13319 14.5301 1.79785 15.3638 2.6362C16.2022 3.46995 16.8669 4.46172 17.3194 5.55407C17.7719 6.64642 18.0032 7.81767 18 9.00003C18.0032 10.1824 17.7718 11.3536 17.3193 12.4459C16.8668 13.5383 16.2021 14.5301 15.3638 15.3639C14.5301 16.2023 13.5384 16.8669 12.446 17.3194C11.3537 17.7719 10.1824 18.0033 9.00003 18Z" fill="' . esc_attr($background_color) . '"/>
        <path d="M13 9.9044V13H11.285V10.1119C11.285 9.38668 11.0373 8.89145 10.4164 8.89145C9.94251 8.89145 9.661 9.22486 9.53668 9.5477C9.49153 9.66307 9.47989 9.82329 9.47989 9.98511V12.9999H7.76472C7.76472 12.9999 7.78775 8.10832 7.76472 7.60197H9.48002V8.36691C9.47656 8.37293 9.4717 8.37882 9.46863 8.38457H9.48002V8.36691C9.70794 7.9999 10.1144 7.47522 11.0257 7.47522C12.1541 7.47522 13 8.24671 13 9.9044ZM5.97052 5C5.38383 5 5 5.40301 5 5.93251C5 6.45076 5.37271 6.86542 5.94801 6.86542H5.95914C6.55733 6.86542 6.92927 6.45076 6.92927 5.93251C6.91789 5.40301 6.55733 5 5.97052 5ZM5.10194 13H6.81646V7.60197H5.10194V13Z" fill="white"/>
    </svg>',
    'instagram' => '<svg width="' . $icon_width . '" height="' . $icon_width . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M9 18C13.9706 18 18 13.9706 18 9C18 4.02944 13.9706 0 9 0C4.02944 0 0 4.02944 0 9C0 13.9706 4.02944 18 9 18Z" fill="' . esc_attr($background_color) . '"/>
        <path fill-rule="evenodd" clip-rule="evenodd" d="M9.1575 4C10.3725 4 10.5495 4.0065 11.0615 4.03C11.5935 4.0545 11.957 4.139 12.275 4.2625C12.6087 4.38794 12.9111 4.58468 13.161 4.839C13.4153 5.08889 13.6121 5.39125 13.7375 5.725C13.861 6.043 13.9455 6.4065 13.97 6.9385C13.994 7.472 14 7.642 14 9V9.04C14 10.3615 13.994 10.5335 13.97 11.0615C13.9455 11.5935 13.861 11.957 13.7375 12.275C13.6121 12.6087 13.4153 12.9111 13.161 13.161C12.9111 13.4153 12.6087 13.6121 12.275 13.7375C11.957 13.861 11.5935 13.9455 11.0615 13.97C10.528 13.994 10.358 14 9 14H8.96C7.6385 14 7.4665 13.994 6.9385 13.97C6.4065 13.9455 6.043 13.861 5.725 13.7375C5.39125 13.6121 5.08889 13.4153 4.839 13.161C4.58468 12.9111 4.38794 12.6087 4.2625 12.275C4.139 11.957 4.0545 11.5935 4.03 11.0615C4.0065 10.5495 4 10.372 4 9.1575V8.8425C4 7.6275 4.0065 7.4505 4.03 6.9385C4.0545 6.4065 4.139 6.043 4.2625 5.725C4.38794 5.39125 4.58468 5.08889 4.839 4.839C5.08889 4.58468 5.39125 4.38794 5.725 4.2625C6.043 4.139 6.4065 4.0545 6.9385 4.03C7.4505 4.0065 7.628 4 8.8425 4H9.1575ZM9.117 4.901H8.883C7.655 4.901 7.491 4.9065 6.9795 4.93C6.492 4.9525 6.2275 5.0335 6.051 5.102C5.8175 5.193 5.651 5.301 5.476 5.476C5.301 5.651 5.193 5.8175 5.102 6.051C5.0335 6.2275 4.952 6.492 4.93 6.9795C4.9065 7.491 4.901 7.655 4.901 8.883V9.117C4.901 10.345 4.9065 10.509 4.93 11.0205C4.9525 11.508 5.0335 11.7725 5.102 11.949C5.193 12.182 5.3015 12.349 5.476 12.524C5.651 12.699 5.8175 12.807 6.051 12.898C6.2275 12.9665 6.492 13.048 6.9795 13.07C7.5065 13.094 7.6645 13.099 9 13.099H9.04C10.3385 13.099 10.4985 13.094 11.02 13.07C11.508 13.0475 11.7725 12.9665 11.949 12.898C12.182 12.807 12.349 12.699 12.524 12.524C12.699 12.349 12.807 12.1825 12.898 11.949C12.9665 11.7725 13.048 11.508 13.07 11.0205C13.094 10.493 13.099 10.3355 13.099 9V8.96C13.099 7.6615 13.094 7.5015 13.07 6.98C13.0475 6.492 12.9665 6.2275 12.898 6.051C12.8179 5.83378 12.6901 5.6373 12.524 5.476C12.3627 5.30993 12.1662 5.18213 11.949 5.102C11.7725 5.0335 11.508 4.952 11.0205 4.93C10.509 4.9065 10.345 4.901 9.117 4.901ZM9 6.4325C9.33717 6.4325 9.67104 6.49891 9.98254 6.62794C10.294 6.75697 10.5771 6.94609 10.8155 7.1845C11.0539 7.42292 11.243 7.70596 11.3721 8.01746C11.5011 8.32896 11.5675 8.66283 11.5675 9C11.5675 9.33717 11.5011 9.67104 11.3721 9.98254C11.243 10.294 11.0539 10.5771 10.8155 10.8155C10.5771 11.0539 10.294 11.243 9.98254 11.3721C9.67104 11.5011 9.33717 11.5675 9 11.5675C8.31906 11.5675 7.666 11.297 7.1845 10.8155C6.703 10.334 6.4325 9.68094 6.4325 9C6.4325 8.31906 6.703 7.666 7.1845 7.1845C7.666 6.703 8.31906 6.4325 9 6.4325ZM9 7.3335C8.55802 7.3335 8.13414 7.50908 7.82161 7.82161C7.50908 8.13414 7.3335 8.55802 7.3335 9C7.3335 9.44198 7.50908 9.86586 7.82161 10.1784C8.13414 10.4909 8.55802 10.6665 9 10.6665C9.44198 10.6665 9.86586 10.4909 10.1784 10.1784C10.4909 9.86586 10.6665 9.44198 10.6665 9C10.6665 8.55802 10.4909 8.13414 10.1784 7.82161C9.86586 7.50908 9.44198 7.3335 9 7.3335ZM11.669 5.731C11.8281 5.731 11.9807 5.79421 12.0933 5.90674C12.2058 6.01926 12.269 6.17187 12.269 6.331C12.269 6.49013 12.2058 6.64274 12.0933 6.75526C11.9807 6.86779 11.8281 6.931 11.669 6.931C11.5099 6.931 11.3573 6.86779 11.2447 6.75526C11.1322 6.64274 11.069 6.49013 11.069 6.331C11.069 6.17187 11.1322 6.01926 11.2447 5.90674C11.3573 5.79421 11.5099 5.731 11.669 5.731Z" fill="white"/>
    </svg>'
];

// Social media icons for Layout 3 (outline icons with dynamic color)
$social_icons_layout3 = [
    'facebook' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path
            d="M10.54 12.984V19.971H13.665V12.983H15.995L16.438 10.093H13.665V8.217C13.665 7.426 14.052 6.655 15.295 6.655H16.555V4.195C16.555 4.195 15.411 4 14.317 4C12.032 4 10.54 5.384 10.54 7.89V10.093H8V12.984H10.54Z"
            stroke="' . esc_attr($icon_color) . '" stroke-width="1.5" stroke-linecap="round"/>
    </svg>',
    'twitter' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path fill-rule="evenodd" clip-rule="evenodd" d="M20.7071 4.70711C21.0976 4.31658 21.0976 3.68342 20.7071 3.29289C20.3166 2.90237 19.6834 2.90237 19.2929 3.29289L13.348 9.23779L9.07342 3.40865C8.88504 3.15177 8.58555 3 8.267 3H4C3.62317 3 3.27833 3.21184 3.108 3.54798C2.93766 3.88412 2.97075 4.28747 3.19359 4.59135L9.45538 13.1304L3.29289 19.2929C2.90237 19.6834 2.90237 20.3166 3.29289 20.7071C3.68342 21.0976 4.31658 21.0976 4.70711 20.7071L10.652 14.7622L14.9266 20.5914C15.115 20.8482 15.4145 21 15.733 21H20C20.3768 21 20.7217 20.7882 20.892 20.452C21.0623 20.1159 21.0293 19.7125 20.8064 19.4086L14.5446 10.8696L20.7071 4.70711ZM12.3703 11.2865C12.4012 11.338 12.4371 11.3872 12.4781 11.4336L18.0266 19H16.2398L5.97338 5H7.76026L12.3703 11.2865Z" fill="' . esc_attr($icon_color) . '"/>
    </svg>',
    'linkedin' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path
            d="M20 13.1958V19H16.5699V13.5847C16.5699 12.225 16.0747 11.2965 14.8328 11.2965C13.885 11.2965 13.322 11.9216 13.0734 12.5269C12.9831 12.7433 12.9598 13.0437 12.9598 13.3471V18.9998H9.52945C9.52945 18.9998 9.57549 9.82809 9.52945 8.8787H12.96V10.313C12.9531 10.3242 12.9434 10.3353 12.9373 10.3461H12.96V10.313C13.4159 9.62481 14.2288 8.64104 16.0514 8.64104C18.3081 8.64104 20 10.0876 20 13.1958ZM5.94104 4C4.76767 4 4 4.75565 4 5.74845C4 6.72018 4.74541 7.49766 5.89602 7.49766H5.91827C7.11466 7.49766 7.85854 6.72018 7.85854 5.74845C7.83577 4.75565 7.11466 4 5.94104 4ZM4.20388 19H7.63292V8.8787H4.20388V19Z"
            fill="' . esc_attr($icon_color) . '"/>
    </svg>',
    'instagram' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path fill-rule="evenodd" clip-rule="evenodd"
              d="M12.315 2C14.745 2 15.099 2.013 16.123 2.06C17.187 2.109 17.914 2.278 18.55 2.525C19.2175 2.77587 19.8222 3.16936 20.322 3.678C20.8306 4.17777 21.2241 4.7825 21.475 5.45C21.722 6.086 21.891 6.813 21.94 7.877C21.988 8.944 22 9.284 22 12V12.08C22 14.723 21.988 15.067 21.94 16.123C21.891 17.187 21.722 17.914 21.475 18.55C21.2241 19.2175 20.8306 19.8222 20.322 20.322C19.8222 20.8306 19.2175 21.2241 18.55 21.475C17.914 21.722 17.187 21.891 16.123 21.94C15.056 21.988 14.716 22 12 22H11.92C9.277 22 8.933 21.988 7.877 21.94C6.813 21.891 6.086 21.722 5.45 21.475C4.7825 21.2241 4.17777 20.8306 3.678 20.322C3.16936 19.8222 2.77587 19.2175 2.525 18.55C2.278 17.914 2.109 17.187 2.06 16.123C2.013 15.099 2 14.744 2 12.315V11.685C2 9.255 2.013 8.901 2.06 7.877C2.109 6.813 2.278 6.086 2.525 5.45C2.77587 4.7825 3.16936 4.17777 3.678 3.678C4.17777 3.16936 4.7825 2.77587 5.45 2.525C6.086 2.278 6.813 2.109 7.877 2.06C8.901 2.013 9.256 2 11.685 2H12.315ZM12.234 3.802H11.766C9.31 3.802 8.982 3.813 7.959 3.86C6.984 3.905 6.455 4.067 6.102 4.204C5.635 4.386 5.302 4.602 4.952 4.952C4.602 5.302 4.386 5.635 4.204 6.102C4.067 6.455 3.904 6.984 3.86 7.959C3.813 8.982 3.802 9.31 3.802 11.766V12.234C3.802 14.69 3.813 15.018 3.86 16.041C3.905 17.016 4.067 17.545 4.204 17.898C4.386 18.364 4.603 18.698 4.952 19.048C5.302 19.398 5.635 19.614 6.102 19.796C6.455 19.933 6.984 20.096 7.959 20.14C9.013 20.188 9.329 20.198 12 20.198H12.08C14.677 20.198 14.997 20.188 16.04 20.14C17.016 20.095 17.545 19.933 17.898 19.796C18.364 19.614 18.698 19.398 19.048 19.048C19.398 18.698 19.614 18.365 19.796 17.898C19.933 17.545 20.096 17.016 20.14 16.041C20.188 14.986 20.198 14.671 20.198 12V11.92C20.198 9.323 20.188 9.003 20.14 7.96C20.095 6.984 19.933 6.455 19.796 6.102C19.6358 5.66757 19.3802 5.2746 19.048 4.952C18.7254 4.61986 18.3324 4.36426 17.898 4.204C17.545 4.067 17.016 3.904 16.041 3.86C15.018 3.813 14.69 3.802 12.234 3.802ZM12 6.865C12.6743 6.865 13.3421 6.99782 13.9651 7.25588C14.5881 7.51394 15.1542 7.89218 15.631 8.36901C16.1078 8.84584 16.4861 9.41191 16.7441 10.0349C17.0022 10.6579 17.135 11.3257 17.135 12C17.135 12.6743 17.0022 13.3421 16.7441 13.9651C16.4861 14.5881 16.1078 15.1542 15.631 15.631C15.1542 16.1078 14.5881 16.4861 13.9651 16.7441C13.3421 17.0022 12.6743 17.135 12 17.135C10.6381 17.135 9.33201 16.594 8.36901 15.631C7.40601 14.668 6.865 13.3619 6.865 12C6.865 10.6381 7.40601 9.33201 8.36901 8.36901C9.33201 7.40601 10.6381 6.865 12 6.865ZM12 8.667C11.116 8.667 10.2683 9.01815 9.64321 9.64321C9.01815 10.2683 8.667 11.116 8.667 12C8.667 12.884 9.01815 13.7317 9.64321 14.3568C10.2683 14.9818 11.116 15.333 12 15.333C12.884 15.333 13.7317 14.9818 14.3568 14.3568C14.9818 13.7317 15.333 12.884 15.333 12C15.333 11.116 14.9818 10.2683 14.3568 9.64321C13.7317 9.01815 12.884 8.667 12 8.667ZM17.338 5.462C17.6563 5.462 17.9615 5.58843 18.1865 5.81347C18.4116 6.03852 18.538 6.34374 18.538 6.662C18.538 6.98026 18.4116 7.28548 18.1865 7.51053C17.9615 7.73557 17.6563 7.862 17.338 7.862C17.0197 7.862 16.7145 7.73557 16.4895 7.51053C16.2644 7.28548 16.138 6.98026 16.138 6.662C16.138 6.34374 16.2644 6.03852 16.4895 5.81347C16.7145 5.58843 17.0197 5.462 17.338 5.462Z"
              fill="' . esc_attr($icon_color) . '"/>
    </svg>'
];

// Get the appropriate icons based on layout
$social_icons = $layout_style === 'layout-2' ? $social_icons_layout2 : ( $layout_style === 'layout-3' ? $social_icons_layout3 : $social_icons_layout1 );

// Get alignment class
$align_class = '';
if ( isset( $attributes['align'] ) && $attributes['align'] ) {
    $align_class = ' has-text-align-' . esc_attr( $attributes['align'] );
}

// Get block wrapper attributes (includes styling from block editor)
$wrapper_attributes = wpuf_get_block_wrapper_attributes( $attributes, 'social-fields', [
    'class' => 'wpuf-user-social-fields' . $align_class,
    'data-user-id' => $user->ID,
] );

?>
<div <?php echo $wrapper_attributes; ?>>
    <?php if ( $layout_style === 'layout-2' ) : ?>
        <div class="wpuf-social-layout-2">
            <div class="wpuf-bg-white wpuf-rounded-lg wpuf-border wpuf-border-gray-200 wpuf-divide-y wpuf-divide-gray-200">
                <?php if ( $show_facebook && ! empty( $facebook_username ) ) : ?>
                    <a
                        href="https://facebook.com/<?php echo esc_attr( $facebook_username ); ?>"
                        target="_blank"
                        rel="noopener noreferrer"
                        class="wpuf-flex wpuf-items-center wpuf-justify-between wpuf-px-4 wpuf-py-2 wpuf-text-sm wpuf-text-gray-700 hover:wpuf-bg-gray-50 wpuf-no-underline wpuf-transition-colors wpuf-group wpuf-icon-<?php echo esc_attr( $icon_size ); ?>"
                        title="<?php echo esc_attr( sprintf( __( 'Follow %s on Facebook', 'wpuf-pro' ), $user->display_name ) ); ?>"
                    >
                        <div class="wpuf-flex wpuf-items-center wpuf-gap-2">
                            <?php echo $social_icons['facebook']; ?>
                            <span class="wpuf-font-normal">Facebook</span>
                        </div>
                        <svg width="15" height="17" viewBox="0 0 15 17" fill="none" xmlns="http://www.w3.org/2000/svg"
                             class="wpuf-opacity-60 group-hover:wpuf-opacity-100 wpuf-transition-opacity">
                            <path
                                d="M6.64384 4.14385C7.13608 3.65157 7.80379 3.375 8.50003 3.375H14C14.3452 3.375 14.625 3.65483 14.625 4C14.625 4.34518 14.3452 4.625 14 4.625H8.50003C8.13535 4.625 7.78561 4.76987 7.52776 5.02773C7.26982 5.28559 7.12501 5.63533 7.12501 6V8.50001C7.12501 8.84518 6.84517 9.12501 6.5 9.12501C6.15482 9.12501 5.875 8.84518 5.875 8.50001V6C5.875 5.30381 6.15156 4.63613 6.64384 4.14385Z"
                                fill="#9CA3AF"></path>
                            <path
                                d="M10.5581 0.558059C10.8021 0.31398 11.1979 0.31398 11.442 0.558059L14.4419 3.55806C14.686 3.80214 14.686 4.19787 14.4419 4.44195L11.442 7.44194C11.1979 7.68602 10.8021 7.68602 10.5581 7.44194C10.314 7.19786 10.314 6.80213 10.5581 6.55806L13.1161 4L10.5581 1.44194C10.314 1.19787 10.314 0.802136 10.5581 0.558059Z"
                                fill="#9CA3AF"></path>
                            <path
                                d="M1.99999 6.625C1.90054 6.625 1.80515 6.6645 1.73483 6.73483C1.6645 6.80516 1.62499 6.90055 1.62499 6.99999V15C1.62499 15.0995 1.6645 15.1949 1.73483 15.2652C1.80515 15.3355 1.90054 15.375 1.99999 15.375H11C11.0994 15.375 11.1948 15.3355 11.2652 15.2652C11.3355 15.1949 11.375 15.0995 11.375 15V9.99998C11.375 9.65483 11.6548 9.37502 12 9.37502C12.3452 9.37502 12.625 9.65483 12.625 9.99998V15C12.625 15.4309 12.4538 15.8443 12.1491 16.1491C11.8443 16.4538 11.431 16.625 11 16.625H1.99999C1.56902 16.625 1.1557 16.4538 0.850947 16.1491C0.546198 15.8443 0.375 15.4309 0.375 15V6.99999C0.375 6.56902 0.546198 6.1557 0.850947 5.85095C1.1557 5.5462 1.56902 5.375 1.99999 5.375H3.5C3.84518 5.375 4.12499 5.65482 4.12499 6C4.12499 6.34517 3.84518 6.625 3.5 6.625H1.99999Z"
                                fill="#9CA3AF"></path>
                        </svg>
                    </a>
                <?php endif; ?>

                <?php if ( $show_twitter && ! empty( $twitter_username ) ) : ?>
                    <a
                        href="https://x.com/<?php echo esc_attr( $twitter_username ); ?>"
                        target="_blank"
                        rel="noopener noreferrer"
                        class="wpuf-flex wpuf-items-center wpuf-justify-between wpuf-px-4 wpuf-py-2 wpuf-text-sm wpuf-text-gray-700 hover:wpuf-bg-gray-50 wpuf-no-underline wpuf-transition-colors wpuf-group wpuf-icon-<?php echo esc_attr( $icon_size ); ?>"
                        title="<?php echo esc_attr( sprintf( __( 'Follow %s on X', 'wpuf-pro' ), $user->display_name ) ); ?>"
                    >
                        <div class="wpuf-flex wpuf-items-center wpuf-gap-2">
                            <?php echo $social_icons['twitter']; ?>
                            <span class="wpuf-font-normal">X (Twitter)</span>
                        </div>
                        <svg width="15" height="17" viewBox="0 0 15 17" fill="none" xmlns="http://www.w3.org/2000/svg"
                             class="wpuf-opacity-60 group-hover:wpuf-opacity-100 wpuf-transition-opacity">
                            <path
                                d="M6.64384 4.14385C7.13608 3.65157 7.80379 3.375 8.50003 3.375H14C14.3452 3.375 14.625 3.65483 14.625 4C14.625 4.34518 14.3452 4.625 14 4.625H8.50003C8.13535 4.625 7.78561 4.76987 7.52776 5.02773C7.26982 5.28559 7.12501 5.63533 7.12501 6V8.50001C7.12501 8.84518 6.84517 9.12501 6.5 9.12501C6.15482 9.12501 5.875 8.84518 5.875 8.50001V6C5.875 5.30381 6.15156 4.63613 6.64384 4.14385Z"
                                fill="#9CA3AF"></path>
                            <path
                                d="M10.5581 0.558059C10.8021 0.31398 11.1979 0.31398 11.442 0.558059L14.4419 3.55806C14.686 3.80214 14.686 4.19787 14.4419 4.44195L11.442 7.44194C11.1979 7.68602 10.8021 7.68602 10.5581 7.44194C10.314 7.19786 10.314 6.80213 10.5581 6.55806L13.1161 4L10.5581 1.44194C10.314 1.19787 10.314 0.802136 10.5581 0.558059Z"
                                fill="#9CA3AF"></path>
                            <path
                                d="M1.99999 6.625C1.90054 6.625 1.80515 6.6645 1.73483 6.73483C1.6645 6.80516 1.62499 6.90055 1.62499 6.99999V15C1.62499 15.0995 1.6645 15.1949 1.73483 15.2652C1.80515 15.3355 1.90054 15.375 1.99999 15.375H11C11.0994 15.375 11.1948 15.3355 11.2652 15.2652C11.3355 15.1949 11.375 15.0995 11.375 15V9.99998C11.375 9.65483 11.6548 9.37502 12 9.37502C12.3452 9.37502 12.625 9.65483 12.625 9.99998V15C12.625 15.4309 12.4538 15.8443 12.1491 16.1491C11.8443 16.4538 11.431 16.625 11 16.625H1.99999C1.56902 16.625 1.1557 16.4538 0.850947 16.1491C0.546198 15.8443 0.375 15.4309 0.375 15V6.99999C0.375 6.56902 0.546198 6.1557 0.850947 5.85095C1.1557 5.5462 1.56902 5.375 1.99999 5.375H3.5C3.84518 5.375 4.12499 5.65482 4.12499 6C4.12499 6.34517 3.84518 6.625 3.5 6.625H1.99999Z"
                                fill="#9CA3AF"></path>
                        </svg>
                    </a>
                <?php endif; ?>

                <?php if ( $show_linkedin && ! empty( $linkedin_username ) ) : ?>
                    <a
                        href="https://www.linkedin.com/in/<?php echo esc_attr( $linkedin_username ); ?>"
                        target="_blank"
                        rel="noopener noreferrer"
                        class="wpuf-flex wpuf-items-center wpuf-justify-between wpuf-px-4 wpuf-py-2 wpuf-text-sm wpuf-text-gray-700 hover:wpuf-bg-gray-50 wpuf-no-underline wpuf-transition-colors wpuf-group wpuf-icon-<?php echo esc_attr( $icon_size ); ?>"
                        title="<?php echo esc_attr( sprintf( __( 'Connect with %s on LinkedIn', 'wpuf-pro' ), $user->display_name ) ); ?>"
                    >
                        <div class="wpuf-flex wpuf-items-center wpuf-gap-2">
                            <?php echo $social_icons['linkedin']; ?>
                            <span class="wpuf-normal">LinkedIn</span>
                        </div>
                        <svg width="15" height="17" viewBox="0 0 15 17" fill="none" xmlns="http://www.w3.org/2000/svg"
                             class="wpuf-opacity-60 group-hover:wpuf-opacity-100 wpuf-transition-opacity">
                            <path
                                d="M6.64384 4.14385C7.13608 3.65157 7.80379 3.375 8.50003 3.375H14C14.3452 3.375 14.625 3.65483 14.625 4C14.625 4.34518 14.3452 4.625 14 4.625H8.50003C8.13535 4.625 7.78561 4.76987 7.52776 5.02773C7.26982 5.28559 7.12501 5.63533 7.12501 6V8.50001C7.12501 8.84518 6.84517 9.12501 6.5 9.12501C6.15482 9.12501 5.875 8.84518 5.875 8.50001V6C5.875 5.30381 6.15156 4.63613 6.64384 4.14385Z"
                                fill="#9CA3AF"></path>
                            <path
                                d="M10.5581 0.558059C10.8021 0.31398 11.1979 0.31398 11.442 0.558059L14.4419 3.55806C14.686 3.80214 14.686 4.19787 14.4419 4.44195L11.442 7.44194C11.1979 7.68602 10.8021 7.68602 10.5581 7.44194C10.314 7.19786 10.314 6.80213 10.5581 6.55806L13.1161 4L10.5581 1.44194C10.314 1.19787 10.314 0.802136 10.5581 0.558059Z"
                                fill="#9CA3AF"></path>
                            <path
                                d="M1.99999 6.625C1.90054 6.625 1.80515 6.6645 1.73483 6.73483C1.6645 6.80516 1.62499 6.90055 1.62499 6.99999V15C1.62499 15.0995 1.6645 15.1949 1.73483 15.2652C1.80515 15.3355 1.90054 15.375 1.99999 15.375H11C11.0994 15.375 11.1948 15.3355 11.2652 15.2652C11.3355 15.1949 11.375 15.0995 11.375 15V9.99998C11.375 9.65483 11.6548 9.37502 12 9.37502C12.3452 9.37502 12.625 9.65483 12.625 9.99998V15C12.625 15.4309 12.4538 15.8443 12.1491 16.1491C11.8443 16.4538 11.431 16.625 11 16.625H1.99999C1.56902 16.625 1.1557 16.4538 0.850947 16.1491C0.546198 15.8443 0.375 15.4309 0.375 15V6.99999C0.375 6.56902 0.546198 6.1557 0.850947 5.85095C1.1557 5.5462 1.56902 5.375 1.99999 5.375H3.5C3.84518 5.375 4.12499 5.65482 4.12499 6C4.12499 6.34517 3.84518 6.625 3.5 6.625H1.99999Z"
                                fill="#9CA3AF"></path>
                        </svg>
                    </a>
                <?php endif; ?>

                <?php if ( $show_instagram && ! empty( $instagram_username ) ) : ?>
                    <a
                        href="https://instagram.com/<?php echo esc_attr( $instagram_username ); ?>"
                        target="_blank"
                        rel="noopener noreferrer"
                        class="wpuf-flex wpuf-items-center wpuf-justify-between wpuf-px-4 wpuf-py-2 wpuf-text-sm wpuf-text-gray-700 hover:wpuf-bg-gray-50 wpuf-no-underline wpuf-transition-colors wpuf-group wpuf-icon-<?php echo esc_attr( $icon_size ); ?>"
                        title="<?php echo esc_attr( sprintf( __( 'Follow %s on Instagram', 'wpuf-pro' ), $user->display_name ) ); ?>"
                    >
                        <div class="wpuf-flex wpuf-items-center wpuf-gap-2">
                            <?php echo $social_icons['instagram']; ?>
                            <span class="wpuf-font-normal">Instagram</span>
                        </div>
                        <svg width="15" height="17" viewBox="0 0 15 17" fill="none" xmlns="http://www.w3.org/2000/svg"
                             class="wpuf-opacity-60 group-hover:wpuf-opacity-100 wpuf-transition-opacity">
                            <path
                                d="M6.64384 4.14385C7.13608 3.65157 7.80379 3.375 8.50003 3.375H14C14.3452 3.375 14.625 3.65483 14.625 4C14.625 4.34518 14.3452 4.625 14 4.625H8.50003C8.13535 4.625 7.78561 4.76987 7.52776 5.02773C7.26982 5.28559 7.12501 5.63533 7.12501 6V8.50001C7.12501 8.84518 6.84517 9.12501 6.5 9.12501C6.15482 9.12501 5.875 8.84518 5.875 8.50001V6C5.875 5.30381 6.15156 4.63613 6.64384 4.14385Z"
                                fill="#9CA3AF"></path>
                            <path
                                d="M10.5581 0.558059C10.8021 0.31398 11.1979 0.31398 11.442 0.558059L14.4419 3.55806C14.686 3.80214 14.686 4.19787 14.4419 4.44195L11.442 7.44194C11.1979 7.68602 10.8021 7.68602 10.5581 7.44194C10.314 7.19786 10.314 6.80213 10.5581 6.55806L13.1161 4L10.5581 1.44194C10.314 1.19787 10.314 0.802136 10.5581 0.558059Z"
                                fill="#9CA3AF"></path>
                            <path
                                d="M1.99999 6.625C1.90054 6.625 1.80515 6.6645 1.73483 6.73483C1.6645 6.80516 1.62499 6.90055 1.62499 6.99999V15C1.62499 15.0995 1.6645 15.1949 1.73483 15.2652C1.80515 15.3355 1.90054 15.375 1.99999 15.375H11C11.0994 15.375 11.1948 15.3355 11.2652 15.2652C11.3355 15.1949 11.375 15.0995 11.375 15V9.99998C11.375 9.65483 11.6548 9.37502 12 9.37502C12.3452 9.37502 12.625 9.65483 12.625 9.99998V15C12.625 15.4309 12.4538 15.8443 12.1491 16.1491C11.8443 16.4538 11.431 16.625 11 16.625H1.99999C1.56902 16.625 1.1557 16.4538 0.850947 16.1491C0.546198 15.8443 0.375 15.4309 0.375 15V6.99999C0.375 6.56902 0.546198 6.1557 0.850947 5.85095C1.1557 5.5462 1.56902 5.375 1.99999 5.375H3.5C3.84518 5.375 4.12499 5.65482 4.12499 6C4.12499 6.34517 3.84518 6.625 3.5 6.625H1.99999Z"
                                fill="#9CA3AF"></path>
                        </svg>
                    </a>
                <?php endif; ?>
            </div>
        </div>
    <?php else : ?>
        <div class="wpuf-flex wpuf-gap-3 wpuf-items-center">
            <?php if ( $show_facebook && ! empty( $facebook_username ) ) : ?>
                <a
                    href="https://facebook.com/<?php echo esc_attr( $facebook_username ); ?>"
                    target="_blank"
                    rel="noopener noreferrer"
                    class="wpuf-social-link wpuf-social-facebook wpuf-icon-<?php echo esc_attr( $icon_size ); ?>"
                    title="<?php echo esc_attr( sprintf( __( 'Follow %s on Facebook', 'wpuf-pro' ), $user->display_name ) ); ?>"
                >
                    <?php echo $social_icons['facebook']; ?>
                </a>
            <?php endif; ?>

            <?php if ( $show_twitter && ! empty( $twitter_username ) ) : ?>
                <a
                    href="https://x.com/<?php echo esc_attr( $twitter_username ); ?>"
                    target="_blank"
                    rel="noopener noreferrer"
                    class="wpuf-social-link wpuf-social-twitter wpuf-icon-<?php echo esc_attr( $icon_size ); ?>"
                    title="<?php echo esc_attr( sprintf( __( 'Follow %s on X', 'wpuf-pro' ), $user->display_name ) ); ?>"
                >
                    <?php echo $social_icons['twitter']; ?>
                </a>
            <?php endif; ?>

            <?php if ( $show_linkedin && ! empty( $linkedin_username ) ) : ?>
                <a
                    href="https://www.linkedin.com/in/<?php echo esc_attr( $linkedin_username ); ?>"
                    target="_blank"
                    rel="noopener noreferrer"
                    class="wpuf-social-link wpuf-social-linkedin wpuf-icon-<?php echo esc_attr( $icon_size ); ?>"
                    title="<?php echo esc_attr( sprintf( __( 'Connect with %s on LinkedIn', 'wpuf-pro' ), $user->display_name ) ); ?>"
                >
                    <?php echo $social_icons['linkedin']; ?>
                </a>
            <?php endif; ?>

            <?php if ( $show_instagram && ! empty( $instagram_username ) ) : ?>
                <a
                    href="https://instagram.com/<?php echo esc_attr( $instagram_username ); ?>"
                    target="_blank"
                    rel="noopener noreferrer"
                    class="wpuf-social-link wpuf-social-instagram wpuf-icon-<?php echo esc_attr( $icon_size ); ?>"
                    title="<?php echo esc_attr( sprintf( __( 'Follow %s on Instagram', 'wpuf-pro' ), $user->display_name ) ); ?>"
                >
                    <?php echo $social_icons['instagram']; ?>
                </a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>
