<?php

use WeDevs\Wpuf\Pro\Admin\Forms\Profile_Form_Template;

/**
 * Determine form is vendor reg or not
 *
 * @since 3.4.7
 *
 * @param $form_id
 *
 * @return bool
 */
function wpuf_is_vendor_reg( $form_id ) {
    $vendor_reg = false;

    if ( isset( $form_id ) ) {
        $form_settings     = wpuf_get_form_settings( $form_id );
        $profile_templates = ( new Profile_Form_Template() )->wpuf_get_profile_form_templates();

        $profile_templates = array_map(
            function ( $template_name, $profile_form ) {
                return $template_name;
            }, array_keys( $profile_templates ), $profile_templates
        );

        $role = [ 'shop_manager', 'seller' ];

        if ( ( ! empty( $form_settings['form_template'] ) && in_array( $form_settings['form_template'], $profile_templates, true ) ) && in_array( $form_settings['role'], $role, true ) ) {
            $vendor_reg = true;
        }
    }

    return $vendor_reg;
}

/**
 * Get setup wizard or WC account page url
 *
 * @since 3.4.7
 *
 * @return url
 */
function wpuf_get_dokan_redirect_url() {
    $wc_acc_page  = get_option( 'woocommerce_myaccount_page_id' );
    $redirect_url = wpuf_get_option( 'account_page', 'wpuf_my_account', 0 );

    if ( function_exists( 'dokan_get_option' ) ) {
        $redirect_url = dokan_get_option( 'disable_welcome_wizard', 'dokan_general', 'off' ) !== 'on' ? get_site_url() . '/?page=dokan-seller-setup' : '';
    }

    if ( $wc_acc_page ) {
        $redirect_url = $redirect_url === '' ? get_permalink( $wc_acc_page ) : $redirect_url;
    }

    return $redirect_url;
}

/**
 * Include a template file
 *
 * If the third arguments is not passed,
 * looks up the file on the child theme directory, then on parent theme directory,
 * if not found loads from pro plugin folder
 * We can use this function to include a template file from module directory
 *
 * @since 3.1.11
 * @since 3.4.11 param $wpuf_pro_dir added
 *
 * @param string $file file name
 * @param string $wpuf_pro_dir the file path
 *
 * @return void
 */
if ( ! function_exists( 'wpuf_load_pro_template' ) ) {
    function wpuf_load_pro_template( $file, $args = [], $wpuf_pro_dir = '' ) {
        if ( ! $wpuf_pro_dir ) {
            $wpuf_pro_dir = WPUF_PRO_INCLUDES . '/templates/';
        }

        if ( ! wpuf()->is_pro() ) {
            return;
        }

        if ( $args && is_array( $args ) ) {
            extract( $args ); //phpcs:ignore
        }

        $child_theme_dir  = get_stylesheet_directory() . '/wpuf/';
        $parent_theme_dir = get_template_directory() . '/wpuf/';

        if ( file_exists( $child_theme_dir . $file ) ) {
            include $child_theme_dir . $file;
        } elseif ( file_exists( $parent_theme_dir . $file ) ) {
            include $parent_theme_dir . $file;
        } elseif ( file_exists( $wpuf_pro_dir . $file ) ) {
            include $wpuf_pro_dir . $file;
        }
    }
}

/**
 * Get the registration form builder setting menu titles. The titles will show on Registration forms > Settings > left side menu
 *
 * @since 4.1.0
 *
 * @return mixed|null
 */
function wpuf_get_registration_form_builder_setting_menu_titles() {
    $registration_settings_fields = apply_filters(
        'wpuf_registration_form_builder_setting_menu_titles', [
            'general'      => [
                'label' => __( 'General', 'wpuf-pro' ),
                'icon'  => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" class="custom-stroke">
                                <path d="M8.75 5L16.875 5M8.75 5C8.75 5.69036 8.19036 6.25 7.5 6.25C6.80964 6.25 6.25 5.69036 6.25 5M8.75 5C8.75 4.30964 8.19036 3.75 7.5 3.75C6.80964 3.75 6.25 4.30964 6.25 5M3.125 5H6.25M8.75 15H16.875M8.75 15C8.75 15.6904 8.19036 16.25 7.5 16.25C6.80964 16.25 6.25 15.6904 6.25 15M8.75 15C8.75 14.3096 8.19036 13.75 7.5 13.75C6.80964 13.75 6.25 14.3096 6.25 15M3.125 15L6.25 15M13.75 10L16.875 10M13.75 10C13.75 10.6904 13.1904 11.25 12.5 11.25C11.8096 11.25 11.25 10.6904 11.25 10M13.75 10C13.75 9.30964 13.1904 8.75 12.5 8.75C11.8096 8.75 11.25 9.30964 11.25 10M3.125 10H11.25" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>',
            ],
            'notification' => [
                'label' => __( 'Notification Settings', 'wpuf-pro' ),
                'icon'  => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg" class="custom-stroke">
                    <path d="M14.8559 17.0817C16.7504 16.857 18.5773 16.4116 20.3102 15.7719C18.8734 14.177 17.9988 12.0656 17.9988 9.75V9.04919C17.999 9.03281 17.999 9.01641 17.999 9C17.999 5.68629 15.3127 3 11.999 3C8.68531 3 5.99902 5.68629 5.99902 9L5.99883 9.75C5.99883 12.0656 5.1243 14.177 3.6875 15.7719C5.42043 16.4116 7.24746 16.857 9.14216 17.0818M14.8559 17.0817C13.919 17.1928 12.9656 17.25 11.9988 17.25C11.0322 17.25 10.0789 17.1929 9.14216 17.0818M14.8559 17.0817C14.9488 17.3711 14.999 17.6797 14.999 18C14.999 19.6569 13.6559 21 11.999 21C10.3422 21 8.99902 19.6569 8.99902 18C8.99902 17.6797 9.04921 17.3712 9.14216 17.0818" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>',
            ],
            'display'      => [
                'label' => __( 'Display Settings', 'wpuf-pro' ),
                'icon'  => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="custom-stroke">
                            <path d="M9 17.25V18.2574C9 19.053 8.68393 19.8161 8.12132 20.3787L7.5 21H16.5L15.8787 20.3787C15.3161 19.8161 15 19.053 15 18.2574V17.25M21 5.25V15C21 16.2426 19.9926 17.25 18.75 17.25H5.25C4.00736 17.25 3 16.2426 3 15V5.25M21 5.25C21 4.00736 19.9926 3 18.75 3H5.25C4.00736 3 3 4.00736 3 5.25M21 5.25V12C21 13.2426 19.9926 14.25 18.75 14.25H5.25C4.00736 14.25 3 13.2426 3 12V5.25" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>',
            ],
            'advanced'     => [
                'label' => __( 'Advanced Settings', 'wpuf-pro' ),
                'icon'  => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="custom-stroke">
                                <path d="M4.5 11.9993C4.5 16.1414 7.85786 19.4993 12 19.4993C16.1421 19.4993 19.5 16.1414 19.5 11.9993M4.5 11.9993C4.5 7.85712 7.85786 4.49925 12 4.49925C16.1421 4.49926 19.5 7.85712 19.5 11.9993M4.5 11.9993L3 11.9993M19.5 11.9993L21 11.9993M19.5 11.9993L12 11.9993M3.54256 15.0774L4.9521 14.5644M19.0475 9.43411L20.457 8.92108M5.10547 17.785L6.25454 16.8208M17.7452 7.17897L18.8943 6.21479M7.4999 19.7943L8.2499 18.4952M15.7499 5.50484L16.4999 4.2058M10.4371 20.8633L10.6975 19.386M13.3023 4.61393L13.5627 3.13672M13.5627 20.8633L13.3023 19.3861M10.6976 4.61397L10.4371 3.13676M16.4999 19.7941L15.7499 18.4951M7.49995 4.20565L12 11.9993M18.8944 17.7843L17.7454 16.8202M6.25469 7.17835L5.10562 6.21417M20.4573 15.0776L19.0477 14.5646M4.95235 9.43426L3.54281 8.92123M12 11.9993L8.25 18.4944"  stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                </svg>',
            ],
        ]
    );

    return apply_filters(
        'wpuf_registration_builder_post_settings_menu_items', [
            'form_settings' => [
                'label'     => __( 'Form Settings', 'wpuf-pro' ),
                'icon'      => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M8.34332 1.94005C8.43373 1.39759 8.90307 1 9.45302 1H10.547C11.0969 1 11.5663 1.39759 11.6567 1.94005L11.8056 2.83386C11.8764 3.25813 12.1893 3.59838 12.5865 3.76332C12.9839 3.92832 13.4403 3.90629 13.7904 3.65617L14.528 3.12933C14.9755 2.80969 15.5885 2.86042 15.9774 3.24929L16.751 4.02284C17.1398 4.41171 17.1906 5.02472 16.8709 5.47223L16.3439 6.21007C16.0938 6.56012 16.0718 7.01633 16.2367 7.41363C16.4016 7.81078 16.7418 8.12363 17.166 8.19433L18.0599 8.34332C18.6024 8.43373 19 8.90307 19 9.45302V10.547C19 11.0969 18.6024 11.5663 18.0599 11.6567L17.1661 11.8056C16.7419 11.8764 16.4016 12.1893 16.2367 12.5865C16.0717 12.9839 16.0937 13.4403 16.3438 13.7904L16.8706 14.5278C17.1902 14.9753 17.1395 15.5884 16.7506 15.9772L15.9771 16.7508C15.5882 17.1396 14.9752 17.1904 14.5277 16.8707L13.7901 16.3439C13.44 16.0938 12.9837 16.0718 12.5864 16.2367C12.1892 16.4016 11.8764 16.7418 11.8057 17.166L11.6567 18.0599C11.5663 18.6024 11.0969 19 10.547 19H9.45302C8.90307 19 8.43373 18.6024 8.34332 18.0599L8.19435 17.1661C8.12364 16.7419 7.81072 16.4016 7.41349 16.2367C7.01608 16.0717 6.55975 16.0937 6.2096 16.3438L5.47198 16.8707C5.02447 17.1904 4.41146 17.1396 4.02259 16.7507L3.24904 15.9772C2.86017 15.5883 2.80944 14.9753 3.12909 14.5278L3.65612 13.79C3.90616 13.4399 3.92821 12.9837 3.76326 12.5864C3.59837 12.1892 3.25819 11.8764 2.83402 11.8057L1.94005 11.6567C1.39759 11.5663 1 11.0969 1 10.547V9.45302C1 8.90307 1.39759 8.43373 1.94005 8.34332L2.83386 8.19436C3.25813 8.12364 3.59838 7.81071 3.76332 7.41347C3.92833 7.01605 3.90629 6.5597 3.65618 6.20954L3.12948 5.47216C2.80983 5.02465 2.86057 4.41164 3.24943 4.02277L4.02298 3.24922C4.41185 2.86036 5.02486 2.80962 5.47237 3.12927L6.20997 3.65613C6.56004 3.90618 7.01628 3.92822 7.4136 3.76326C7.81077 3.59837 8.12364 3.25819 8.19433 2.834L8.34332 1.94005Z" stroke="#9CA3AF" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                            <path d="M13.0007 10C13.0007 11.6569 11.6576 13 10.0007 13C8.34384 13 7.0007 11.6569 7.0007 10C7.0007 8.34317 8.34384 7.00002 10.0007 7.00002C11.6576 7.00002 13.0007 8.34317 13.0007 10Z" stroke="#9CA3AF" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>',
                'sub_items' => $registration_settings_fields,
            ],
            'modules'       => [
                'label' => __( 'Modules', 'wp-user-frontend' ),
                'icon'  => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M6.42857 9.75L2.25 12L6.42857 14.25M6.42857 9.75L12 12.75L17.5714 9.75M6.42857 9.75L2.25 7.5L12 2.25L21.75 7.5L17.5714 9.75M17.5714 9.75L21.75 12L17.5714 14.25M17.5714 14.25L21.75 16.5L12 21.75L2.25 16.5L6.42857 14.25M17.5714 14.25L12 17.25L6.42857 14.25" stroke="#9CA3AF" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>',
                'sub_items' => apply_filters( 'wpuf_registration_form_module_settings_menu_sub_items', [] ),
            ],
        ]
    );
}

/**
 * Get the registration form builder setting menu contents
 *
 * @since 4.1.0
 *
 * @return array
 */
function wpuf_get_registration_form_builder_setting_menu_contents() {
    $pages    = wpuf_get_pages();
    $registry = ( new Profile_Form_Template() )->wpuf_get_profile_form_templates();

    $template_options = [
        ''                            => __( '-- Select Template --', 'wpuf-pro' ),
        'dokan_vendor_reg_template'   => __( 'Dokan Vendor Registration Form', 'wpuf-pro' ),
        'wc_vendor_reg_template'      => __( 'WC Vendors Registration Form', 'wpuf-pro' ),
        'wc_marketplace_reg_template' => __( 'WCFM Membership Registration Form', 'wpuf-pro' ),
    ];

    foreach ( $registry as $key => $template ) {
        if ( ! $template->is_enabled() ) {
            // remove the template if it's not enabled from $template_options
            unset( $template_options[ $key ] );
        }
    }

    $general = apply_filters(
        'wpuf_registration_form_builder_settings_general', [
            'section' => [
                'before_sign_up_settings' => [
                    'label'  => __( 'Before Sign Up Settings', 'wpuf-pro' ),
                    'desc'   => __(
                        'Set up your preferred user role assignments, manage registration approvals, and customize the registration form setup to ensure a smooth signup experience',
                        'wpuf-pro'
                    ),
                    'fields' => [
                        'role'             => [
                            'label'     => __( 'New User Role', 'wpuf-pro' ),
                            'type'      => 'select',
                            'help_text' => __(
                                'Assign the default role for users upon completing registration.', 'wpuf-pro'
                            ),
                            'default'   => 'subscriber',
                            'options'   => wpuf_get_user_roles(),
                        ],
                        'user_status' => [
                            'label' => __( 'Required Approval After Registration', 'wpuf-pro' ),
                            'type'  => 'toggle',
                            'help_text'  => __(
                                'When enabled, newly registered users must be approved by admin before they can log in', 'wpuf-pro'
                            ),
                        ],
                        'hidden' => [
                            'name' => 'wpuf_settings[wpuf_user_status]',
                            'id'   => 'wpuf_user_status',
                        ],
                        'form_template'    => [
                            'label'     => __( 'Choose Form Template', 'wpuf-pro' ),
                            'type'      => 'select',
                            'help_text' => __(
                                'Select a registration form template to automatically apply its layout and settings during user signup',
                                'wpuf-pro'
                            ),
                            'options'   => $template_options,
                        ],
                    ],
                ],
                'after_sign_up_settings'  => [
                    'label'  => __( 'After Sign Up Settings', 'wpuf-pro' ),
                    'desc'   => __(
                        'Customize post-signup and profile update experiences by adjusting redirection, success messages, and button text for a seamless user interaction',
                        'wpuf-pro'
                    ),
                    'fields' => [
                        'reg_redirect_to'     => [
                            'label'   => __( 'After Registration Successful Redirection', 'wpuf-pro' ),
                            'type'    => 'select',
                            'help_text'    => __(
                                'Select the page where users are redirected after a successful registration', 'wpuf-pro'
                            ),
                            'options' => [
                                'same' => __( 'Same page', 'wpuf-pro' ),
                                'page' => __( 'To a page', 'wpuf-pro' ),
                                'url'  => __( 'To a custom URL', 'wpuf-pro' ),
                            ],
                        ],
                        'message'             => [
                            'label'   => __( 'Registration Success Message', 'wpuf-pro' ),
                            'type'    => 'textarea',
                            'help_text'    => __(
                                'Customize the message displayed to users upon successful registration', 'wpuf-pro'
                            ),
                            'default' => __( 'Registration successful', 'wpuf-pro' ),
                        ],
                        'reg_page_id'         => [
                            'label'   => __( 'Page', 'wpuf-pro' ),
                            'type'    => 'select',
                            'options' => $pages,
                        ],
                        'registration_url'    => [
                            'label' => __( 'Custom URL', 'wpuf-pro' ),
                            'type'  => 'text',
                        ],
                        'submit_text'         => [
                            'label'   => __( 'Submit Button Text', 'wpuf-pro' ),
                            'type'    => 'text',
                            'default' => 'Register',
                            'help_text'    => __(
                                'Set the text that appears on the submit button during registration', 'wpuf-pro'
                            ),
                        ],
                        'profile_redirect_to' => [
                            'label'   => __( 'After Profile Update Successful Redirection', 'wpuf-pro' ),
                            'type'    => 'select',
                            'help_text'    => __(
                                'Select the page where users are redirected after updating their profile', 'wpuf-pro'
                            ),
                            'options' => [
                                'same' => __( 'Same page', 'wpuf-pro' ),
                                'page' => __( 'To a page', 'wpuf-pro' ),
                                'url'  => __( 'To a custom URL', 'wpuf-pro' ),
                            ],
                        ],
                        'update_message'      => [
                            'label'   => __( 'Update Profile Message', 'wpuf-pro' ),
                            'type'    => 'textarea',
                            'default' => __( 'Profile updated successfully', 'wpuf-pro' ),
                            'help_text' => __( 'Customize the message shown to users after a successful profile update', 'wpuf-pro' ),
                        ],
                        'profile_page_id'     => [
                            'label'   => __( 'Page', 'wpuf-pro' ),
                            'type'    => 'select',
                            'options' => $pages,
                        ],
                        'profile_url'         => [
                            'label' => __( 'Custom URL', 'wpuf-pro' ),
                            'type'  => 'text',
                        ],
                        'update_text'         => [
                            'label'   => __( 'Update Button Text', 'wpuf-pro' ),
                            'type'    => 'text',
                            'default' => 'Update Profile',
                            'help_text'   => __( 'Set the text displayed on the button for updating user profiles', 'wpuf-pro' ),
                        ],
                    ],
                ],
            ],
        ]
    );

    $notification = apply_filters(
        'wpuf_registration_form_builder_settings_notification', [
            'section' => [
                'user_notification'  => [
                    'label'  => __( 'User Notification', 'wpuf-pro' ),
                    'desc'   => __(
                        'Customize email verification and welcome messages for new users, including subject lines and body content, to ensure a smooth and engaging onboarding process',
                        'wpuf-pro'
                    ),
                    'fields' => [
                        'user_notification'     => [
                            'label' => __( 'Enable User Notification', 'wpuf-pro' ),
                            'type'  => 'toggle',
                            'help_text' => __( 'Enable to send email notifications to users upon completing registration', 'wpuf-pro' ),
                        ],
                        'notification_type'     => [
                            'label'   => __( 'User Notification Type', 'wpuf-pro' ),
                            'type'    => 'card-radio',
                            'help_text'    => __( 'An email will be sent to the user after registration', 'wpuf-pro' ),
                            'options' => [
                                'email_verification' => [
                                    'label'     => __( 'Email-Verification', 'wpuf-pro' ),
                                    'sub_label' => __(
                                        'Ensure account security by sending a verification email, prompting users to confirm their email addresses', 'wpuf-pro'
                                    ),
                                ],
                                'welcome_email'      => [
                                    'label'     => __( 'Welcome Email', 'wpuf-pro' ),
                                    'sub_label' => __(
                                        'Send a friendly welcome message to new users, introducing them to your platform and providing key information', 'wpuf-pro'
                                    ),
                                ],
                            ],
                        ],
                        'verification_subject'  => [
                            'label' => __( 'Confirmation Email Subject', 'wpuf-pro' ),
                            'type'  => 'text',
                            'name'  => 'wpuf_settings[notification][verification_subject]',
                        ],
                        'verification_body'     => [
                            'label'     => __( 'Confirmation Email Body', 'wpuf-pro' ),
                            'type'      => 'rich-text',
                            'name'      => 'wpuf_settings[notification][verification_body]',
                            'long_help' => '<p class="wpuf-m-0"><span class="wpuf-text-sm wpuf-text-gray-700 wpuf-font-medium">You may use:</span>
                                         <span class="wpuf-leading-8 wpuf-ml-3">
                                         <span data-clipboard-text="{username}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{username}</span>
                                         <span data-clipboard-text="{blogname}" class="wpuf-post-content wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{blogname}</span>
                                         <span data-clipboard-text="{activation_link}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{activation_link}</span></p>
                                         <p class="wpuf-mt-1 wpuf-mb-0 wpuf-my-0"><span class="wpuf-text-sm wpuf-text-gray-700 wpuf-font-medium">You also may use meta key:</span>
                                         <span class="wpuf-leading-8 wpuf-ml-3">
                                         <span data-clipboard-text="{your_custom_hook_name}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{your_custom_hook_name}</span>
                                         <span data-clipboard-text="{terms_and_conditions}" class="wpuf-post-content wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{terms_and_conditions}</span></span>',
                        ],
                        'welcome_email_subject' => [
                            'label' => __( 'Welcome Email Subject', 'wpuf-pro' ),
                            'type'  => 'text',
                            'name'  => 'wpuf_settings[notification][welcome_email_subject]',
                        ],
                        'welcome_email_body'    => [
                            'label'     => __( 'Welcome Email Body', 'wpuf-pro' ),
                            'type'      => 'rich-text',
                            'long_help' => '<p class="wpuf-m-0"><span class="wpuf-text-sm wpuf-text-gray-700 wpuf-font-medium">You may use:</span>
                                         <span class="wpuf-leading-8 wpuf-ml-3">
                                         <span data-clipboard-text="{username}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{username}</span>
                                         <span data-clipboard-text="{blogname}" class="wpuf-post-content wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{blogname}</span>
                                         <span data-clipboard-text="{activation_link}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{activation_link}</span></p>
                                         <p class="wpuf-mt-1 wpuf-mb-0 wpuf-my-0"><span class="wpuf-text-sm wpuf-text-gray-700 wpuf-font-medium">You also may use meta key:</span>
                                         <span class="wpuf-leading-8 wpuf-ml-3">
                                         <span data-clipboard-text="{your_custom_hook_name}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{your_custom_hook_name}</span>
                                         <span data-clipboard-text="{terms_and_conditions}" class="wpuf-post-content wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{terms_and_conditions}</span></span>',
                        ],
                    ],
                ],
                'admin_notification' => [
                    'label'  => __( 'Admin Notification', 'wpuf-pro' ),
                    'desc'   => __(
                        'Set up admin notifications to receive alerts about user activity. Enable notifications, and customize the subject and message content for effective communication',
                        'wpuf-pro'
                    ),
                    'fields' => [
                        'admin_notification'  => [
                            'label' => __( 'Enable Admin Notification', 'wpuf-pro' ),
                            'type'  => 'toggle',
                            'help_text' => __( 'Enable to receive email notifications for new user registrations and activity', 'wpuf-pro' ),
                        ],
                        'admin_email_subject' => [
                            'label' => __( 'Subject', 'wpuf-pro' ),
                            'type'  => 'text',
                        ],
                        'admin_email_body'    => [
                            'label'     => __( 'Message', 'wpuf-pro' ),
                            'type'      => 'textarea',
                            'long_help' => '<p class="wpuf-m-0"><span class="wpuf-text-sm wpuf-text-gray-700 wpuf-font-medium">You may use:</span>
                                         <span class="wpuf-leading-8 wpuf-ml-3">
                                         <span data-clipboard-text="{username}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{username}</span>
                                         <span data-clipboard-text="{user_email}" class="wpuf-post-content wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{user_email}</span>
                                         <span data-clipboard-text="{display_name}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{display_name}</span>
                                         <span data-clipboard-text="{user_status}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{user_status}</span>
                                         <span data-clipboard-text="{pending_users}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{pending_users}</span>
                                         <span data-clipboard-text="{approved_users}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{approved_users}</span>
                                         <span data-clipboard-text="{denied_users}" class="wpuf-pill-green hover:wpuf-cursor-pointer wpuf-template-text">{denied_users}</span></p>',
                        ],
                    ],
                ],
            ],
        ]
    );

    $display = apply_filters(
        'wpuf_registration_form_builder_settings_display', [
            'section' => [
                'form_style'  => [
                    'label'  => __( 'Choose Form Style', 'wpuf-pro' ),
                    'desc'   => __(
                        'Select a form style to match your website\'s design, enhancing visual appeal and user experience while maintaining brand consistency across all form submissions',
                        'wpuf-pro'
                    ),
                    'fields' => [
                        'profile_form_layout' => [
                            'label'   => __( 'Choose Form Template', 'wpuf-pro' ),
                            'type'    => 'pic-radio',
                            'options' => [
                                'layout1' => [
                                    'label' => __( 'Profile Form', 'wpuf-pro' ),
                                    'image' => WPUF_ASSET_URI . '/images/forms/layout1.png',
                                ],
                                'layout2' => [
                                    'label' => __( 'Profile Form', 'wpuf-pro' ),
                                    'image' => WPUF_ASSET_URI . '/images/forms/layout2.png',
                                ],
                                'layout3' => [
                                    'label' => __( 'Profile Form', 'wpuf-pro' ),
                                    'image' => WPUF_ASSET_URI . '/images/forms/layout3.png',
                                ],
                                'layout4' => [
                                    'label' => __( 'Profile Form', 'wpuf-pro' ),
                                    'image' => WPUF_ASSET_URI . '/images/forms/layout4.png',
                                ],
                                'layout5' => [
                                    'label' => __( 'Profile Form', 'wpuf-pro' ),
                                    'image' => WPUF_ASSET_URI . '/images/forms/layout5.png',
                                ],
                            ],
                        ],
                        'label_position'      => [
                            'label'     => __( 'Label Position', 'wpuf-pro' ),
                            'help_text' => __( 'Customize the position of form labels for improved user navigation and clarity', 'wpuf-pro' ),
                            'type'      => 'select',
                            'options'   => [
                                'above'  => __( 'Above Element', 'wpuf-pro' ),
                                'left'   => __( 'Left of Element', 'wpuf-pro' ),
                                'right'  => __( 'Right of Element', 'wpuf-pro' ),
                                'hidden' => __( 'Hidden', 'wpuf-pro' ),
                            ],
                            'under_field_text' =>  __( 'Select where the form labels should appear for better readability', 'wpuf-pro' ),
                        ],
                        'use_theme_css'       => [
                            'label'     => __( 'Use Theme CSS', 'wpuf-pro' ),
                            'help_text' => __(
                                'Apply your site\'s theme CSS for consistent styling and appearance', 'wpuf-pro'
                            ),
                            'type'      => 'select',
                            'options'   => [
                                'wpuf-style'       => __( 'No', 'wpuf-pro' ),
                                'wpuf-theme-style' => __( 'Yes', 'wpuf-pro' ),
                            ],
                        ],
                    ],
                ],
            ],
        ]
    );

    $advanced = apply_filters(
        'wpuf_registration_form_builder_setting_advanced', [
            'section' => [
                'multistep_form' => [
                    'label'  => __( 'Multistep Form', 'wpuf-pro' ),
                    'desc'   => 'Divide your form into multiple steps with customizable progress indicators, enhancing user experience through personalized text and background colors for active & inactive steps',
                    'fields' => [
                        'enable_multistep'           => [
                            'label'     => __( 'Enable MultiStep', 'wp-user-frontend' ),
                            'type'      => 'toggle',
                            'help_text' => __(
                                'Split the form into multiple steps for a better user experience', 'wp-user-frontend'
                            ),
                            'link'      => esc_url_raw(
                                'https://wedevs.com/docs/wp-user-frontend-pro/registration-profile-forms/creating-a-multistep-registration-form/'
                            ),
                        ],
                        'multistep_progressbar_type' => [
                            'label'     => __( 'Multistep Progressbar Type', 'wp-user-frontend' ),
                            'type'      => 'select',
                            'help_text' => __(
                                'Select how to display progress in a multi-step form', 'wp-user-frontend'
                            ),
                            'options'   => [
                                'progressive'  => __( 'Progressbar', 'wp-user-frontend' ),
                                'step_by_step' => __( 'Step by Step', 'wp-user-frontend' ),
                            ],
                        ],
                        'ms_ac_txt_color'            => [
                            'label'     => __( 'Active Text Color', 'wp-user-frontend' ),
                            'type'      => 'color-picker',
                            'help_text' => __(
                                'Set the color of the text for the active step in the progress bar', 'wp-user-frontend'
                            ),
                            'default'   => '#fff',
                        ],
                        'ms_active_bgcolor'          => [
                            'label'     => __( 'Active Background Color', 'wp-user-frontend' ),
                            'type'      => 'color-picker',
                            'help_text' => __(
                                'Set the background color for the active step in the progress bar', 'wp-user-frontend'
                            ),
                            'default'   => '#00a0d2',
                        ],
                        'ms_bgcolor'                 => [
                            'label'     => __( 'Background Color', 'wp-user-frontend' ),
                            'type'      => 'color-picker',
                            'help_text' => __(
                                'Set the background color for inactive steps in the progress bar', 'wp-user-frontend'
                            ),
                            'default'   => '#E4E4E4',
                        ],
                    ],
                ],
            ],
        ]
    );

    $form_settings['form_settings'] = apply_filters(
        'wpuf_form_builder_post_settings', [
            'general'      => $general,
            'notification' => $notification,
            'display'      => $display,
            'advanced'     => $advanced,
        ]
    );

    return apply_filters(
        'wpuf_registration_form_builder_setting_menu_contents', $form_settings
    );
}

/**
 * Get extension to MIME type mapping for profile photos
 *
 * @since WPUF_PRO
 *
 * @return array Array with extensions as keys and mime types as values
 */
function wpuf_get_profile_photo_extension_mime_map() {
    $extension_mime_map = [
        'jpg'  => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png'  => 'image/png',
        'gif'  => 'image/gif',
        'webp' => 'image/webp',
    ];

    /**
     * Filter profile photo extension to MIME type mapping
     *
     * @since WPUF_PRO
     *
     * @param array $extension_mime_map Extension => MIME type mapping
     */
    return apply_filters( 'wpuf_profile_photo_extension_mime_map', $extension_mime_map );
}

/**
 * Validate profile photo file by extension and MIME type
 *
 * @since WPUF_PRO
 *
 * @param string $file_path Full path to the file
 * @param string $original_filename Original filename with extension
 *
 * @return array|false Array with validation info or false on failure
 */
function wpuf_validate_profile_photo_file( $file_path, $original_filename ) {
    // the wpuf free is not updated
    if ( ! function_exists( 'wpuf_field_profile_photo_allowed_extensions' ) ) {
        return false;
    }

    // Get allowed extensions and mimes
    $allowed_extensions = array_keys( wpuf_field_profile_photo_allowed_extensions() );
    $allowed_mimes = wpuf_field_profile_photo_allowed_mimes();
    $extension_mime_map = wpuf_get_profile_photo_extension_mime_map();

    // Get file extension
    $file_extension = strtolower( pathinfo( $original_filename, PATHINFO_EXTENSION ) );

    // Check if extension is allowed
    if ( ! in_array( $file_extension, $allowed_extensions, true ) ) {
        return false;
    }

    // Check file exists and is readable
    if ( ! file_exists( $file_path ) || ! is_readable( $file_path ) ) {
        return false;
    }

    // Get detected MIME type
    $file_info = wp_check_filetype_and_ext( $file_path, $original_filename );
    $detected_mime = $file_info['type'];

    // Check if MIME type is allowed
    if ( ! in_array( $detected_mime, $allowed_mimes, true ) ) {
        return false;
    }

    // Check if extension matches expected MIME type
    $expected_mime = isset( $extension_mime_map[ $file_extension ] ) ? $extension_mime_map[ $file_extension ] : null;
    if ( $expected_mime && $detected_mime !== $expected_mime ) {
        return false;
    }

    return [
        'extension'     => $file_extension,
        'detected_mime' => $detected_mime,
        'expected_mime' => $expected_mime,
        'file_size'     => filesize( $file_path ),
        'is_valid'      => true,
    ];
}

// Add WebP MIME type support for WordPress uploads if not already supported
add_filter( 'upload_mimes', function( $mimes ) {
    // Only add WebP support if it's not already present
    if ( ! isset( $mimes['webp'] ) ) {
        $mimes['webp'] = 'image/webp';
    }
    return $mimes;
} );

/**
 * Check if Post View Categories feature is available
 * This provides backward compatibility for users who updated WPUF Pro but not WPUF Free
 *
 * @since 4.2.2
 *
 * @return bool
 */
function wpuf_is_post_view_categories_available() {
    // Check if WPUF Free version supports this feature (4.1.14+)
    if ( defined( 'WPUF_VERSION' ) ) {
        if ( version_compare( WPUF_VERSION, '4.1.14', '>=' ) ) {
            return true;
        }
    }

    return false;
}

/**
 * Validate and process profile photo attachment
 * Extracted from Profile_Form.php for reuse across the plugin
 *
 * @since WPUF_PRO
 *
 * @param int $attachment_id Attachment ID
 * @param int $user_id User ID
 * @param string $meta_key User meta key to store the attachment ID
 * @param int $max_size Maximum file size in bytes (default: 2MB)
 * @return array Result with 'success' boolean and 'message' string
 */
function wpuf_process_profile_photo_attachment( $attachment_id, $user_id, $meta_key = 'wpuf_profile_photo', $max_size = null ) {
    // Set default max size if not provided
    if ( $max_size === null ) {
        $max_size = apply_filters( 'wpuf_profile_photo_max_size', 2 * 1024 * 1024 ); // 2MB default
    }
    

    // Verify the attachment exists and is valid
    $attachment = get_post( $attachment_id );
    if ( ! $attachment || 'attachment' !== $attachment->post_type ) {
        return [
            'success' => false,
            'message' => __( 'Invalid attachment.', 'wpuf-pro' )
        ];
    }
    

    $file_path = get_attached_file( $attachment_id );
    if ( ! $file_path || ! file_exists( $file_path ) ) {
        return [
            'success' => false,
            'message' => __( 'Attachment file not found.', 'wpuf-pro' )
        ];
    }
    
    // Security: Remove any potential directory traversal
    $file_path = wp_normalize_path( realpath( $file_path ) );
    $upload_base = wp_normalize_path( realpath( wp_upload_dir()['basedir'] ) );
    

    // Verify the file is within uploads directory
    if ( ! $file_path || 0 !== strpos( $file_path, $upload_base ) ) {
        return [
            'success' => false,
            'message' => __( 'File is outside allowed directory.', 'wpuf-pro' )
        ];
    }

    // Use the existing validation function if available
    if ( function_exists( 'wpuf_validate_profile_photo_file' ) ) {
        $validation_result = wpuf_validate_profile_photo_file( $file_path, basename( $file_path ) );

        if ( ! $validation_result ) {
            return [
                'success' => false,
                'message' => __( 'File validation failed.', 'wpuf-pro' )
            ];
        }
        // Check if MIME type matches
        if ( $validation_result['detected_mime'] !== $attachment->post_mime_type ) {
            return [
                'success' => false,
                'message' => __( 'File type mismatch detected.', 'wpuf-pro' )
            ];
        }
        // Additional security checks for malicious content
        $file_content = file_get_contents( $file_path );
        if ( preg_match( '/<\?(php|=|\s)/i', $file_content ) ||
             preg_match( '/^(\x7FELF|\x4D\x5A)/', $file_content ) ||
             preg_match( '/^PK\x03\x04/', $file_content ) ) {
            return [
                'success' => false,
                'message' => __( 'File contains potentially malicious content.', 'wpuf-pro' )
            ];
        }
        // Check file size
        if ( $validation_result['file_size'] > $max_size ) {
            return [
                'success' => false,
                'message' => sprintf( __( 'File size must be less than %s.', 'wpuf-pro' ), size_format( $max_size ) )
            ];
        }

        // All validations passed, update user meta
        update_user_meta( $user_id, $meta_key, $attachment_id );

        return [
            'success' => true,
            'message' => __( 'Profile photo updated successfully.', 'wpuf-pro' ),
            'attachment_id' => $attachment_id,
            'file_size' => $validation_result['file_size']
        ];
    }
    // Fallback validation if wpuf_validate_profile_photo_file doesn't exist
    $image_info = getimagesize( $file_path );
    if ( ! $image_info ) {
        return [
            'success' => false,
            'message' => __( 'Invalid image file.', 'wpuf-pro' )
        ];
    }
    // Check file size
    $file_size = filesize( $file_path );
    if ( $file_size > $max_size ) {
        return [
            'success' => false,
            'message' => sprintf( __( 'File size must be less than %s.', 'wpuf-pro' ), size_format( $max_size ) )
        ];
    }
    // Basic security check for malicious content
    $file_content = file_get_contents( $file_path );
    if ( preg_match( '/<\?(php|=|\s)/i', $file_content ) ||
         preg_match( '/^(\x7FELF|\x4D\x5A)/', $file_content ) ||
         preg_match( '/^PK\x03\x04/', $file_content ) ) {
        return [
            'success' => false,
            'message' => __( 'File contains potentially malicious content.', 'wpuf-pro' )
        ];
    }

    // All validations passed, update user meta
    update_user_meta( $user_id, $meta_key, $attachment_id );

    return [
        'success' => true,
        'message' => __( 'Profile photo updated successfully.', 'wpuf-pro' ),
        'attachment_id' => $attachment_id,
        'file_size' => $file_size
    ];
}

/**
 * Validate uploaded file for profile photo (for direct file uploads)
 * Extracted from Profile_Form.php logic for reuse
 *
 * @since WPUF_PRO
 *
 * @param array $file Uploaded file array
 * @param int $max_size Maximum file size in bytes (default: 5MB)
 * @return array Result with 'valid' boolean and 'message' string
 */
function wpuf_validate_uploaded_profile_photo( $file, $max_size = null ) {
    if ( $max_size === null ) {
        $max_size = 5 * 1024 * 1024; // 5MB default
    }
    // Check if file array is valid
    if ( ! isset( $file['tmp_name'] ) || ! isset( $file['name'] ) || ! isset( $file['size'] ) ) {
        return [
            'valid' => false,
            'message' => __( 'Invalid file data.', 'wpuf-pro' )
        ];
    }
    // Check file size
    if ( $file['size'] > $max_size ) {
        return [
            'valid' => false,
            'message' => sprintf( __( 'File size must be less than %s.', 'wpuf-pro' ), size_format( $max_size ) )
        ];
    }

    // Check file type by extension using the existing function
    $file_type = wp_check_filetype( $file['name'] );

    // Use the existing allowed extensions function if available
    if ( function_exists( 'wpuf_field_profile_photo_allowed_extensions' ) ) {
        $allowed_extensions = wpuf_field_profile_photo_allowed_extensions();
        $file_extension = strtolower( pathinfo( $file['name'], PATHINFO_EXTENSION ) );
        if ( ! in_array( $file_extension, array_keys( $allowed_extensions ), true ) ) {
            return [
                'valid' => false,
                'message' => __( 'File type not allowed.', 'wpuf-pro' )
            ];
        }
    } else {
        // Fallback to basic validation
        $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        if ( ! in_array( $file_type['type'], $allowed_types, true ) ) {
            return [
                'valid' => false,
                'message' => __( 'Only JPG, PNG, GIF, and WebP images are allowed.', 'wpuf-pro' )
            ];
        }
    }
    // Additional security check using getimagesize
    $image_info = getimagesize( $file['tmp_name'] );
    if ( ! $image_info ) {
        return [
            'valid' => false,
            'message' => __( 'Invalid image file.', 'wpuf-pro' )
        ];
    }
    // Check for malicious content
    $file_content = file_get_contents( $file['tmp_name'] );
    if ( preg_match( '/<\?(php|=|\s)/i', $file_content ) ||
         preg_match( '/^(\x7FELF|\x4D\x5A)/', $file_content ) ||
         preg_match( '/^PK\x03\x04/', $file_content ) ) {
        return [
            'valid' => false,
            'message' => __( 'File contains potentially malicious content.', 'wpuf-pro' )
        ];
    }
    // Check image dimensions
    $min_width = 32;
    $min_height = 32;
    $max_width = 2048;
    $max_height = 2048;
    if ( $image_info[0] < $min_width || $image_info[1] < $min_height ) {
        return [
            'valid' => false,
            'message' => sprintf( __( 'Image must be at least %dx%d pixels.', 'wpuf-pro' ), $min_width, $min_height )
        ];
    }
    if ( $image_info[0] > $max_width || $image_info[1] > $max_height ) {
        return [
            'valid' => false,
            'message' => sprintf( __( 'Image must be no larger than %dx%d pixels.', 'wpuf-pro' ), $max_width, $max_height )
        ];
    }
    return [
        'valid' => true,
        'message' => __( 'File validation passed.', 'wpuf-pro' ),
        'file_info' => [
            'width' => $image_info[0],
            'height' => $image_info[1],
            'type' => $image_info[2],
            'mime' => $image_info['mime'],
            'size' => $file['size']
        ]
    ];
}
