<?php

namespace WeDevs\Wpuf\Pro\Fields;

use WeDevs\Wpuf\Ajax\Upload_Ajax;

/**
 * Profile Photo Field Class
 *
 * @since 3.1.0
 **/
class Field_Profile_Photo extends Profile_Field {

    public function __construct() {
        $this->name       = __( 'Profile Photo', 'wpuf-pro' );
        $this->input_type = 'profile_photo';
        $this->icon       = 'user-circle';
    }

    /**
     * Render the Profile Photo field
     *
     * @param array   $field_settings
     *
     * @param integer $form_id
     *
     * @param string  $type
     *
     * @param integer $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        if ( method_exists( 'WeDevs\Wpuf\Fields\Field_Contract', 'enqueue_file_upload_scripts' ) ) {
            $this->enqueue_file_upload_scripts();
        }

        // Get profile photo attachment ID
        $attachment_data = $this->get_user_meta_attachment( $post_id, 'wpuf_profile_photo' );
        $has_photo       = $attachment_data && $attachment_data['has_value'];
        $attachment_id   = $attachment_data ? $attachment_data['attachment_id'] : 0;

        $unique_id = sprintf( '%s-%d', $field_settings['name'], $form_id );
        ?>
        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <?php $this->print_label( $field_settings, $form_id ); ?>

            <div class="wpuf-fields">
                <div id="wpuf-<?php echo $unique_id; ?>-upload-container">
                    <div class="wpuf-attachment-upload-filelist" data-type="file"
                         data-required="<?php echo $field_settings['required']; ?>">
                        <a id="wpuf-<?php echo $unique_id; ?>-pickfiles" data-form_id="<?php echo $form_id; ?>"
                           class="button file-selector <?php echo ' wpuf_' . $field_settings['name'] . '_' . $form_id; ?>"
                           href="#"><?php echo $field_settings['button_label']; ?></a>

                        <ul class="wpuf-attachment-list thumbnail">
                            <?php
                            if ( $has_photo && $attachment_id ) {
                                // Use the same pattern as other file upload fields
                                echo Upload_Ajax::attach_html( $attachment_id, $field_settings['name'] );
                            }
                            ?>
                        </ul>
                    </div>
                </div><!-- .container -->

                <?php $this->help_text( $field_settings ); ?>

            </div> <!-- .wpuf-fields -->

            <script type="text/javascript">
                ;( function ( $ ) {
                    $( document ).ready( function () {
                        var uploader = new WPUF_Uploader( 'wpuf-<?php echo $unique_id; ?>-pickfiles', 'wpuf-<?php echo $unique_id; ?>-upload-container', 1, '<?php echo $field_settings['name']; ?>', 'jpg,jpeg,gif,png,bmp', <?php echo $field_settings['max_size'] ?>);
                        wpuf_plupload_items.push( uploader );
                    } );
                } )( jQuery );
            </script>
        </li>

        <?php
    }

    /**
     * Get field options setting
     *
     * @return array
     **/
    public function get_options_settings() {
        $default_options = $this->get_default_option_settings( false, [ 'dynamic', 'count', 'width' ] );
        $settings = [
            $this->get_meta_key_setting( 'wpuf_profile_photo' ),
            [
                'name'      => 'button_label',
                'title'     => __( 'Button Label', 'wpuf-pro' ),
                'type'      => 'text',
                'default'   => __( 'Select Profile Photo', 'wpuf-pro' ),
                'section'   => 'basic',
                'priority'  => 20,
                'help_text' => __( 'Enter a label for the Profile Photo button', 'wpuf-pro' ),
            ],
            [
                'name'      => 'max_size',
                'title'     => __( 'Max. file size', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 20,
                'help_text' => __( 'Enter maximum upload size limit in KB', 'wpuf-pro' ),
            ],
        ];

        if ( function_exists( 'wpuf_field_profile_photo_allowed_extensions' ) ) {
            $settings[] = [
                'name'        => 'extension',
                'title'       => __( 'Allowed file types', 'wpuf-pro' ),
                'title_class' => 'label-hr',
                'type'        => 'checkbox',
                'options'     => wpuf_field_profile_photo_allowed_extensions(),
                'section'     => 'advanced',
                'priority'    => 21,
                'help_text'   => __( 'Enter The allowed file types', 'wpuf-pro' ),
            ];
        }

        return array_merge( $default_options, $settings );
    }

    /**
     * Get the field props
     *
     * @return array
     **/
    public function get_field_props() {
        $defaults = $this->default_attributes();
        $props = [
            'input_type'    => 'profile_photo',
            'required'      => 'yes',
            'is_meta'       => 'no', // we will handle this manually
            'button_label'  => __( 'Select Profile Photo', 'wpuf-pro' ),
            'name'          => 'wpuf_profile_photo',
            'max_size'      => '2048', // default 2MB
            'extension'     => [ 'jpg', 'jpeg', 'png' ],
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Prepare entry
     *
     * @param $field
     *
     * @return void
     **/
    public function prepare_entry( $field ) {
        return isset( $_POST['wpuf_files'][ $field['name'] ] ) ? $_POST['wpuf_files'][ $field['name'] ] : [];
    }

}
