<?php

namespace WeDevs\Wpuf\Pro\Fields;

/**
 * Abstract base class for profile-related fields
 *
 * Provides common utilities for fields that work with user profile data
 *
 * @since 4.2.3
 */
abstract class Profile_Field extends Pro_Field_Contract {

    /**
     * Get user meta value from profile context
     *
     * In profile forms, $post_id is actually the user ID
     *
     * @since 4.2.3
     *
     * @param int|null $user_id User ID (may be passed as $post_id in profile context)
     * @param string   $meta_key Meta key to retrieve
     * @param mixed    $default Default value if meta doesn't exist
     *
     * @return mixed
     */
    protected function get_user_meta_value( $user_id, $meta_key, $default = null ) {
        if ( ! $user_id || ! $user_id > 0 ) {
            return $default;
        }

        $value = get_user_meta( $user_id, $meta_key, true );

        return $value ? $value : $default;
    }

    /**
     * Get profile field value with common pattern handling
     *
     * Handles the common pattern: check if post_id exists, if field is meta, get value, otherwise use default.
     * For profile fields, post_id is actually the user ID and type is always 'user'.
     *
     * @since 4.2.3
     *
     * @param array    $field_settings Field settings array
     * @param int|null $post_id User ID (may be passed as $post_id in profile context)
     * @param mixed    $default Default value if no value found
     *
     * @return mixed
     */
    protected function get_profile_field_value( $field_settings, $post_id = null, $default = '' ) {
        if ( ! empty( $post_id ) && $this->is_meta( $field_settings ) ) {
            return $this->get_meta( $post_id, $field_settings['name'], 'user' );
        }

        return isset( $field_settings['default'] ) ? $field_settings['default'] : $default;
    }

    /**
     * Check if user meta attachment exists and is valid image
     *
     * @since 4.2.3
     *
     * @param int|null $user_id User ID
     * @param string   $meta_key Meta key containing attachment ID
     *
     * @return array|false Returns array with 'has_value' and 'attachment_id', or false
     */
    protected function get_user_meta_attachment( $user_id, $meta_key ) {
        if ( ! $user_id || ! $user_id > 0 ) {
            return false;
        }

        $attachment_id = get_user_meta( $user_id, $meta_key, true );

        if ( $attachment_id && wp_attachment_is_image( $attachment_id ) ) {
            return [
                'has_value'     => true,
                'attachment_id' => $attachment_id,
            ];
        }

        return [
            'has_value'     => false,
            'attachment_id' => 0,
        ];
    }

    /**
     * Print list attributes for profile fields
     *
     * @since 4.2.3
     *
     * @param array $field Field settings
     *
     * @return void
     */
    public function print_list_attributes( $field ) {
        $label      = isset( $field['label'] ) ? $field['label'] : '';
        $el_name    = ! empty( $field['name'] ) ? $field['name'] : '';
        $el_name    = 'wpuf_' . $el_name;
        $class_name = ! empty( $field['css'] ) ? ' ' . $field['css'] : '';
        $field_size = ! empty( $field['width'] ) ? ' field-size-' . $field['width'] : '';
        printf( 'class="wpuf-el %s%s%s" data-label="%s"', $el_name, $class_name, $field_size, $label );
    }

    /**
     * Get common meta key field setting
     *
     * Provides a standardized read-only meta key field for profile fields
     *
     * @since 4.2.3
     *
     * @param string $meta_key Default meta key value
     * @param int    $priority Field priority in settings
     *
     * @return array
     */
    protected function get_meta_key_setting( $meta_key, $priority = 11 ) {
        return [
            'name'         => 'name',
            'title'        => __( 'Meta Key', 'wp-user-frontend' ),
            'type'         => 'text-meta',
            'is_read_only' => true,
            'section'      => 'basic',
            'priority'     => $priority,
            'help_text'    => __( 'Name of the meta key this field will save to', 'wp-user-frontend' ),
        ];
    }
}
