<?php
/**
 * Block Configuration Constants
 *
 * Centralized configuration for all block names, contexts, and related constants.
 * This ensures consistency across the codebase and makes namespace changes easier.
 *
 * @since 4.2.0
 * @package WPUF_Pro
 */

namespace WPUF\UserDirectory;

/**
 * Block Configuration Class
 */
class BlockConfig {

    /**
     * Block Namespace
     */
    const NAMESPACE = 'wpuf-ud';

    /**
     * Block Category
     */
    const CATEGORY = 'wpuf-ud';

    /**
     * Block Names (without namespace)
     */
    const BLOCKS = [
        // Container blocks
        'DIRECTORY'      => 'directory',
        'DIRECTORY_ITEM' => 'directory-item',
        'PROFILE'        => 'profile',

        // Individual blocks
        'AVATAR'         => 'avatar',
        'COVER_PHOTO'    => 'cover-photo',
        'NAME'           => 'name',
        'BIO'            => 'bio',
        'CONTACT'        => 'contact',
        'CUSTOM_FIELD'   => 'custom-field',
        'POSTS'          => 'posts',
        'COMMENTS'       => 'comments',
        'FILES'          => 'files',
        'SOCIAL'         => 'social',
        'TABS'           => 'tabs',
        'BUTTON'         => 'button',
        'PROFILE_BUTTON' => 'profile-button',
        'MESSAGE_BUTTON' => 'message-button',
    ];

    /**
     * Context Keys
     */
    const CONTEXTS = [
        'USER_ID'           => 'wpuf-ud/userId',
        'USER_OBJECT'       => 'wpuf-ud/userObject',
        'IS_DIRECTORY_MODE' => 'wpuf-ud/isDirectoryMode',
        'CAN_EDIT'          => 'wpuf-ud/canEdit',
        'CONTAINER_LAYOUT'  => 'wpuf-ud/containerLayout',
    ];

    /**
     * Get full block name with namespace
     *
     * @param string $block_name Block name without namespace
     * @return string Full block name with namespace
     */
    public static function get_block_name( $block_name ) {
        return self::NAMESPACE . '/' . $block_name;
    }

    /**
     * Get all block names with namespace
     *
     * @return array Array of block names with namespace
     */
    public static function get_all_block_names() {
        $block_names = [];
        foreach ( self::BLOCKS as $key => $block_name ) {
            $block_names[ $key ] = self::get_block_name( $block_name );
        }
        return $block_names;
    }

    /**
     * Get context key
     *
     * @param string $context_key Context key constant
     * @return string Context key value
     */
    public static function get_context( $context_key ) {
        return self::CONTEXTS[ $context_key ] ?? '';
    }

    /**
     * Check if a block name matches our namespace
     *
     * @param string $block_name Full block name to check
     * @return bool True if block belongs to our namespace
     */
    public static function is_our_block( $block_name ) {
        return strpos( $block_name, self::NAMESPACE . '/' ) === 0;
    }

    /**
     * Get block name without namespace
     *
     * @param string $full_block_name Full block name with namespace
     * @return string Block name without namespace
     */
    public static function get_block_name_without_namespace( $full_block_name ) {
        if ( self::is_our_block( $full_block_name ) ) {
            return str_replace( self::NAMESPACE . '/', '', $full_block_name );
        }
        return $full_block_name;
    }

    /**
     * Get directory paths for blocks
     *
     * @return array Array of block directory paths
     */
    public static function get_block_paths() {
        $paths = [];
        foreach ( self::BLOCKS as $key => $block_name ) {
            $paths[ $block_name ] = WPUF_UD_ROOT . '/src/js/blocks/' . $block_name . '/block.json';
        }
        return $paths;
    }
}
