<?php
/**
 * Profile Data Helper Functions
 * Centralized functions for managing dynamic data across all profile layouts
 *
 * @since 4.2.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Get comprehensive user profile data for layouts
 *
 * @param WP_User $user User object
 * @param array $template_data Template configuration data
 * @param string $layout_style Layout style (layout-1, layout-2, layout-3)
 * @return array Complete profile data array
 */
function wpuf_ud_get_profile_data( $user, $template_data, $layout_style = 'layout-1' ) {
    if ( ! $user || ! is_a( $user, 'WP_User' ) ) {
        return [];
    }

    $profile_data = [
        'user' => $user,
        'template_config' => wpuf_ud_get_template_config( $template_data ),
        'user_meta' => wpuf_ud_get_user_meta_data( $user ),
        'avatar' => wpuf_ud_get_avatar_data( $user, $template_data ),
        'social_media' => wpuf_ud_get_social_media_data( $user, $template_data ),
        'contact_info' => wpuf_ud_get_contact_info_data( $user, $layout_style ),
        'navigation' => wpuf_ud_get_navigation_data( $user ),
        'tab_config' => wpuf_ud_get_tab_configuration( $template_data )
    ];

    return apply_filters( 'wpuf_ud_profile_data', $profile_data, $user, $template_data );
}

/**
 * Get template configuration data
 *
 * @param array $template_data Raw template data
 * @return array Processed template configuration
 */
function wpuf_ud_get_template_config( $template_data ) {
    return [
        'show_avatar' => ! empty( $template_data['show_avatar'] ),
        'enable_tabs' => ! empty( $template_data['enable_tabs'] ),
        'default_tabs' => isset( $template_data['default_tabs'] ) ? $template_data['default_tabs'] : [],
        'default_active_tab' => ! empty( $template_data['default_active_tab'] ) ? $template_data['default_active_tab'] : 'about',
        'avatar_size' => ! empty( $template_data['avatar_size'] ) ? $template_data['avatar_size'] : 'thumbnail',
        'custom_tab_labels' => ! empty( $template_data['custom_tab_labels'] ) ? $template_data['custom_tab_labels'] : []
    ];
}

/**
 * Get user meta data
 *
 * @param WP_User $user User object
 * @return array User meta data
 */
function wpuf_ud_get_user_meta_data( $user ) {
    $first_name = get_user_meta( $user->ID, 'first_name', true );
    $last_name = get_user_meta( $user->ID, 'last_name', true );
    $full_name = trim( $first_name . ' ' . $last_name );

    return [
        'first_name' => $first_name,
        'last_name' => $last_name,
        'full_name' => $full_name,
        'display_name' => $full_name ?: $user->display_name ?: $user->user_login,
        'phone' => get_user_meta( $user->ID, 'wpuf_profile_phone', true ),
        'website' => get_user_meta( $user->ID, 'user_url', true ) ?: $user->user_url,
        'bio' => get_user_meta( $user->ID, 'description', true ),
        'address' => get_user_meta( $user->ID, 'address', true )
    ];
}

/**
 * Get avatar configuration and data
 *
 * @param WP_User $user User object
 * @param array $template_data Template data
 * @return array Avatar data
 */
function wpuf_ud_get_avatar_data( $user, $template_data ) {
    $avatar_size = ! empty( $template_data['avatar_size'] ) ? $template_data['avatar_size'] : 'thumbnail';

    $avatar_sizes = [
        'thumbnail' => 150,
        'medium' => 300,
        'large' => 600
    ];

    $size = $avatar_sizes[ $avatar_size ] ?? 150;

    return [
        'size_key' => $avatar_size,
        'size_px' => $size,
        'sizes' => $avatar_sizes,
        'html_128' => wpuf_ud_get_user_avatar_html( $user, 128, '!wpuf-w-full !wpuf-h-full !wpuf-object-cover !wpuf-rounded-full' ),
        'html_80' => wpuf_ud_get_user_avatar_html( $user, 80, '!wpuf-w-full !wpuf-h-full !wpuf-object-cover !wpuf-rounded-full' )
    ];
}

/**
 * Get social media data and configuration
 *
 * @param WP_User $user User object
 * @param array $template_data Template data containing allowed social fields
 * @return array Social media data
 */
function wpuf_ud_get_social_media_data( $user, $template_data = [] ) {
    $social_fields = [
        'wpuf_social_facebook' => 'Facebook',
        'wpuf_social_twitter' => 'X',
        'wpuf_social_instagram' => 'Instagram',
        'wpuf_social_dribbble' => 'Dribbble',
        'wpuf_social_linkedin' => 'LinkedIn'
    ];

    // Get allowed social fields from template data (same logic as social-icons.php)
    $allowed_social_fields = [];
    $has_social_config = isset( $template_data['social_fields'] );
    if ( $has_social_config ) {
        $allowed_social_fields = $template_data['social_fields'];
    }

    $social_icons = wpuf_ud_get_social_media_icons();

    $social_data = [];
    foreach ( $social_fields as $field => $label ) {
        // Convert field name to match directory settings format (remove wpuf_social_ prefix)
        $field_key = str_replace( 'wpuf_social_', '', $field );

        // Only include this social field if it's in the allowed list
        // If social_fields is explicitly set (even as empty array), respect that setting
        // If social_fields is not set at all, show all fields (backward compatibility)
        if ( ! $has_social_config || in_array( $field_key, $allowed_social_fields ) ) {
            $value = get_user_meta( $user->ID, $field, true );
            if ( $value ) {
                $social_data[ $field ] = [
                    'label' => $label,
                    'url' => $value,
                    'icon_html' => isset( $social_icons[ $field ] ) ? $social_icons[ $field ] : wpuf_ud_get_default_social_icon()
                ];
            }
        }
    }

    return [
        'fields' => $social_fields,
        'icons' => $social_icons,
        'user_social' => $social_data,
        'has_social' => ! empty( $social_data )
    ];
}

/**
 * Get social media icons HTML
 *
 * @return array Social media icons
 */
function wpuf_ud_get_social_media_icons() {
    return [
        'wpuf_social_facebook' => '<svg class="!wpuf-w-5 !wpuf-h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M20 10c0-5.523-4.477-10-10-10S0 4.477 0 10c0 4.991 3.657 9.128 8.438 9.878v-6.987h-2.54V10h2.54V7.797c0-2.506 1.492-3.89 3.777-3.89 1.094 0 2.238.195 2.238.195v2.46h-1.26c-1.243 0-1.63.771-1.63 1.562V10h2.773l-.443 2.89h-2.33v6.988C16.343 19.128 20 14.991 20 10z" clip-rule="evenodd"></path></svg>',
        'wpuf_social_instagram' => '<svg class="!wpuf-w-5 !wpuf-h-5" viewBox="0 0 20 20" fill="currentColor"><path fill-rule="evenodd" d="M10 0C7.284 0 6.944.012 5.877.06c-1.064.049-1.791.218-2.427.465a4.902 4.902 0 00-1.772 1.153A4.902 4.902 0 00.525 3.45C.278 4.086.109 4.813.06 5.877.012 6.944 0 7.284 0 10s.012 3.056.06 4.123c.049 1.064.218 1.791.465 2.427a4.902 4.902 0 001.153 1.772 4.902 4.902 0 001.772 1.153c.636.247 1.363.416 2.427.465C6.944 19.988 7.284 20 10 20s3.056-.012 4.123-.06c1.064-.049 1.791-.218 2.427-.465a4.902 4.902 0 001.772-1.153 4.902 4.902 0 001.153-1.772c.247-.636.416-1.363.465-2.427.048-1.067.06-1.407.06-4.123s-.012-3.056-.06-4.123c-.049-1.064-.218-1.791-.465-2.427a4.902 4.902 0 00-1.153-1.772A4.902 4.902 0 0016.55.525C15.914.278 15.187.109 14.123.06 13.056.012 12.716 0 10 0zm0 1.802c2.67 0 2.986.01 4.04.058.976.045 1.505.207 1.858.344.466.182.8.399 1.15.748.35.35.566.684.748 1.15.137.353.3.882.344 1.857.048 1.055.058 1.37.058 4.041 0 2.67-.01 2.986-.058 4.04-.045.976-.207 1.505-.344 1.858a3.097 3.097 0 01-.748 1.15 3.098 3.098 0 01-1.15.748c-.353.137-.882.3-1.857.344-1.054.048-1.37.058-4.041.058-2.67 0-2.987-.01-4.04-.058-.976-.045-1.505-.207-1.858-.344a3.098 3.098 0 01-1.15-.748 3.098 3.098 0 01-.748-1.15c-.137-.353-.3-.882-.344-1.857-.048-1.055-.058-1.37-.058-4.041 0-2.67.01-2.986.058-4.04.045-.976.207-1.505.344-1.858.182-.466.399-.8.748-1.15.35-.35.684-.566 1.15-.748.353-.137.882-.3 1.857-.344 1.055-.048 1.37-.058 4.041-.058z" clip-rule="evenodd"></path><path fill-rule="evenodd" d="M10 13.333a3.333 3.333 0 110-6.666 3.333 3.333 0 010 6.666zm0-8.468a5.135 5.135 0 100 10.27 5.135 5.135 0 000-10.27z" clip-rule="evenodd"></path><circle cx="15.338" cy="4.662" r="1.2"></circle></svg>',
        'wpuf_social_twitter' => '<svg class="!wpuf-w-5 !wpuf-h-5" viewBox="0 0 20 20" fill="currentColor"><path d="M11.905 8.505L19.34 0h-1.764l-6.453 7.395L5.829 0H0l7.802 11.345L0 20h1.764l6.815-7.807L14.171 20H20L11.905 8.505zm-2.412 2.764l-.79-1.13L2.4 1.3h2.706l5.082 7.268.79 1.13 6.597 9.436h-2.706l-5.376-7.686v-.179z"/></svg>',
        'wpuf_social_dribbble' => '<svg class="!wpuf-w-5 !wpuf-h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 0C4.477 0 0 4.484 0 10.017c0 5.533 4.477 10.017 10 10.017s10-4.484 10-10.017C20 4.484 15.523 0 10 0zm6.238 4.62c1.199 1.467 1.926 3.337 1.949 5.373a11.926 11.926 0 00-3.648-.555c-1.434 0-2.814.226-4.092.638a47.325 47.325 0 00-.95-2.259c2.435-1.028 4.508-2.598 5.741-4.197zm-1.442-1.2C13.755 4.65 11.946 5.951 9.76 6.835a42.24 42.24 0 00-3.77-5.84A8.394 8.394 0 0110 0c1.773 0 3.42.547 4.796 1.42zm-10.34.358a50.61 50.61 0 013.693 5.734C3.618 10.457 1.618 10.545.55 10.55c-.133 0-.264-.002-.392-.005A8.41 8.41 0 014.456 1.778zm-2.86 7.285c.16.003.323.005.49.005 1.436 0 3.866-.121 6.71-1.334.263.514.514 1.035.75 1.562-2.885.825-5.424 2.614-6.958 4.868a8.363 8.363 0 01-1.992-5.1zm3.343 6.539c1.334-2.098 3.667-3.728 6.355-4.444.914 2.382 1.346 4.372 1.513 5.593a8.426 8.426 0 01-4.595 1.364 8.364 8.364 0 01-3.273-.513zm9.212-.473c-.198-1.175-.607-2.988-1.447-5.182a10.417 10.417 0 013.25.498 8.375 8.375 0 01-1.803 4.684z" clip-rule="evenodd"></path></svg>',
        'wpuf_social_linkedin' => '<svg class="!wpuf-w-5 !wpuf-h-5" fill="currentColor" viewBox="0 0 16 15"><path d="M16 9.19575V15H12.5699V9.58474C12.5699 8.22503 12.0747 7.29647 10.8328 7.29647C9.88502 7.29647 9.32199 7.92162 9.07335 8.52694C8.98305 8.74327 8.95978 9.04367 8.95978 9.34708V14.9998H5.52945C5.52945 14.9998 5.57549 5.82809 5.52945 4.8787H8.96003V6.31295C8.95312 6.32424 8.9434 6.33529 8.93726 6.34608H8.96003V6.31295C9.41587 5.62481 10.2288 4.64104 12.0514 4.64104C14.3081 4.64104 16 6.08759 16 9.19575ZM1.94104 0C0.767666 0 0 0.755647 0 1.74845C0 2.72018 0.745411 3.49766 1.89602 3.49766H1.91827C3.11466 3.49766 3.85854 2.72018 3.85854 1.74845C3.83577 0.755647 3.11466 0 1.94104 0ZM0.203875 15H3.63292V4.8787H0.203875V15Z"/></svg>'
    ];
}

/**
 * Get default social media icon
 *
 * @return string Default icon HTML
 */
function wpuf_ud_get_default_social_icon() {
    return '<svg class="!wpuf-w-4 !wpuf-h-4" fill="currentColor" viewBox="0 0 20 20"><circle cx="10" cy="10" r="8"></circle></svg>';
}

/**
 * Get contact information data
 *
 * @param WP_User $user User object
 * @param string $layout_style Layout style (layout-1, layout-2, layout-3)
 * @return array Contact information
 */
function wpuf_ud_get_contact_info_data( $user, $layout_style = 'layout-1' ) {
    $user_meta = wpuf_ud_get_user_meta_data( $user );

    $contact_items = [];

    if ( $user->user_email ) {
        $contact_items['email'] = [
            'type' => 'email',
            'label' => __( 'Email', 'wpuf-pro' ),
            'value' => $user->user_email,
            'display_value' => $user->user_email,
            'icon' => wpuf_ud_get_contact_icon( 'email', $layout_style )
        ];
    }

    // Phone contact is only available in layout-2 and layout-3, not in layout-1
    if ( $user_meta['phone'] && $layout_style !== 'layout-1' ) {
        $contact_items['phone'] = [
            'type' => 'phone',
            'label' => __( 'Phone', 'wpuf-pro' ),
            'value' => $user_meta['phone'],
            'display_value' => $user_meta['phone'],
            'icon' => wpuf_ud_get_contact_icon( 'phone', $layout_style )
        ];
    }

    if ( $user_meta['website'] ) {
        $contact_items['website'] = [
            'type' => 'website',
            'label' => __( 'Website', 'wpuf-pro' ),
            'value' => $user_meta['website'],
            'display_value' => parse_url( $user_meta['website'], PHP_URL_HOST ),
            'icon' => wpuf_ud_get_contact_icon( 'website', $layout_style )
        ];
    }

    if ( $user_meta['address'] ) {
        $contact_items['address'] = [
            'type' => 'address',
            'label' => __( 'Address', 'wpuf-pro' ),
            'value' => $user_meta['address'],
            'display_value' => $user_meta['address'],
            'icon' => wpuf_ud_get_contact_icon( 'address', $layout_style )
        ];
    }

    return $contact_items;
}

/**
 * Get contact icons based on layout style
 *
 * @param string $type Contact type
 * @param string $layout_style Layout style (layout-1, layout-2, layout-3)
 * @return string Icon HTML
 */
function wpuf_ud_get_contact_icon( $type, $layout_style = 'layout-1' ) {
    $icons_layout_1 = [
        'email' => '<svg class="!wpuf-w-4 !wpuf-h-4 !wpuf-mr-4 !wpuf-text-gray-400" viewBox="0 0 14 15" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M12.3948 13.9452H1.60522C0.72015 13.9452 0 13.2252 0 12.34V5.92456C0 5.03937 0.72015 4.31934 1.60522 4.31934H12.3948C13.2798 4.31934 14 5.03937 14 5.92456V12.3399C14 13.2251 13.2798 13.9451 12.3948 13.9451V13.9452ZM1.60522 5.19772C1.20449 5.19772 0.878382 5.52382 0.878382 5.92456V12.3399C0.878382 12.7406 1.20437 13.0667 1.60522 13.0667H12.3948C12.7955 13.0667 13.1216 12.7408 13.1216 12.3399V5.92456C13.1216 5.52382 12.7956 5.19772 12.3948 5.19772H1.60522Z" fill="#707070"/><path d="M9.62423 5.19806C9.38172 5.19806 9.18504 5.00138 9.18504 4.75887C9.18504 3.55405 8.2048 2.57369 6.99997 2.57369C5.79514 2.57369 4.8149 3.55393 4.8149 4.75887C4.8149 5.00138 4.61834 5.19806 4.37571 5.19806C4.13309 5.19806 3.93652 5.00138 3.93652 4.75887C3.93652 3.94061 4.25514 3.17123 4.83373 2.59264C5.41233 2.01404 6.18159 1.69531 6.99986 1.69531C7.81812 1.69531 8.5875 2.01393 9.16609 2.59264C9.74469 3.17123 10.0634 3.94061 10.0634 4.75887C10.0634 5.00138 9.86673 5.19806 9.62423 5.19806Z" fill="#707070"/><path d="M5.68782 8.69674H0.439191C0.196683 8.69674 0 8.50006 0 8.25755C0 8.01504 0.19657 7.81836 0.439191 7.81836H5.68782C5.93032 7.81836 6.12701 8.01504 6.12701 8.25755C6.12701 8.50006 5.93044 8.69674 5.68782 8.69674Z" fill="#707070"/><path d="M13.5609 8.69674H8.31224C8.06962 8.69674 7.87305 8.50006 7.87305 8.25755C7.87305 8.01504 8.06962 7.81836 8.31224 7.81836H13.5609C13.8034 7.81836 14.0001 8.01504 14.0001 8.25755C14.0001 8.50006 13.8034 8.69674 13.5609 8.69674Z" fill="#707070"/><path d="M6.99985 10.5917C6.53242 10.5917 6.09289 10.4098 5.76213 10.0794C5.43115 9.74876 5.24888 9.309 5.24854 8.84122V8.14138C5.24899 7.17645 6.03425 6.39119 6.99906 6.39062H6.99997C7.96535 6.39062 8.75083 7.17566 8.75129 8.14115V8.841C8.75083 9.80592 7.96557 10.5912 7.00076 10.5916H6.99985V10.5917ZM6.12692 8.84077C6.12703 9.07398 6.218 9.29312 6.38292 9.45793C6.54773 9.62262 6.76687 9.71325 6.99985 9.71325H7.00031C7.48113 9.71303 7.87256 9.32159 7.8729 8.84077V8.14138C7.87268 7.66022 7.48079 7.26878 6.99951 7.26912C6.51869 7.26935 6.12726 7.66078 6.12703 8.1416V8.84077H6.12692Z" fill="#707070"/></svg>',
        'phone' => '<svg class="!wpuf-w-4 !wpuf-h-4 !wpuf-mr-4 !wpuf-text-gray-400" viewBox="0 0 14 15" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M8.3125 10.8828H7.875C7.52702 10.8824 7.19339 10.744 6.94733 10.498C6.70127 10.2519 6.56287 9.9183 6.5625 9.57031V7.38281H6.125C6.00897 7.38281 5.89769 7.33672 5.81564 7.25467C5.73359 7.17262 5.6875 7.06134 5.6875 6.94531C5.6875 6.82928 5.73359 6.718 5.81564 6.63595C5.89769 6.55391 6.00897 6.50781 6.125 6.50781H6.5625C6.79451 6.508 7.01696 6.60024 7.18101 6.7643C7.34507 6.92835 7.43731 7.1508 7.4375 7.38281V9.57031C7.43769 9.68629 7.48384 9.79746 7.56585 9.87947C7.64785 9.96147 7.75902 10.0076 7.875 10.0078H8.3125C8.42853 10.0078 8.53981 10.0539 8.62186 10.136C8.70391 10.218 8.75 10.3293 8.75 10.4453C8.75 10.5613 8.70391 10.6726 8.62186 10.7547C8.53981 10.8367 8.42853 10.8828 8.3125 10.8828Z" fill="#707070"/></svg>',
        'website' => '<svg class="!wpuf-w-4 !wpuf-h-4 !wpuf-mr-4 !wpuf-text-gray-400" viewBox="0 0 12 13" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M0.399902 6.8207C0.399902 7.92828 0.728337 9.01098 1.34367 9.9319C1.95901 10.8528 2.83361 11.5706 3.85687 11.9944C4.88014 12.4183 6.00611 12.5292 7.09241 12.3131C8.1787 12.097 9.17653 11.5637 9.9597 10.7805C10.7429 9.99733 11.2762 8.9995 11.4923 7.91321C11.7084 6.82692 11.5975 5.70094 11.1736 4.67768C10.7498 3.65441 10.032 2.77981 9.11109 2.16447C8.19018 1.54914 7.10748 1.2207 5.9999 1.2207C4.51469 1.2207 3.09031 1.8107 2.0401 2.86091C0.989901 3.91111 0.399902 5.33549 0.399902 6.8207Z" stroke="#707070" stroke-width="0.805" stroke-linecap="round" stroke-linejoin="round"/><path d="M4.94422 12.3217C4.16302 11.2276 3.63452 9.171 3.63452 6.8463C3.63452 4.5216 4.16302 2.416 4.94422 1.3457" stroke="#707070" stroke-width="0.805" stroke-linecap="round" stroke-linejoin="round"/><path d="M7.05542 12.3217C7.83662 11.2269 8.36442 9.1717 8.36442 6.8463C8.36442 4.5209 7.83662 2.416 7.05542 1.3457" stroke="#707070" stroke-width="0.805" stroke-linecap="round" stroke-linejoin="round"/><path d="M0.399902 6.82422H11.5999" stroke="#707070" stroke-width="0.805" stroke-linecap="round" stroke-linejoin="round"/><path d="M1.04663 9.43359H10.9523" stroke="#707070" stroke-width="0.805" stroke-linecap="round" stroke-linejoin="round"/><path d="M1.04663 4.20801H10.9523" stroke="#707070" stroke-width="0.805" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'address' => '<svg class="!wpuf-w-4 !wpuf-h-4 !wpuf-mr-4 !wpuf-text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>'
    ];

    $icons_layout_2 = [
        'phone' => '<svg width="44" height="44" viewBox="0 0 44 44" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="44" height="44" rx="22" fill="#D1FAE5"/><path d="M13.875 17.125C13.875 24.0286 19.4714 29.625 26.375 29.625H28.25C29.2855 29.625 30.125 28.7855 30.125 27.75V26.607C30.125 26.1768 29.8322 25.8018 29.4149 25.6975L25.7289 24.776C25.363 24.6845 24.9778 24.8212 24.7515 25.123L23.943 26.201C23.7083 26.5139 23.3025 26.6522 22.9353 26.5177C20.1795 25.5082 17.9918 23.3205 16.9823 20.5647C16.8478 20.1975 16.9861 19.7917 17.299 19.557L18.377 18.7485C18.6788 18.5222 18.8155 18.137 18.724 17.7711L17.8025 14.0851C17.6982 13.6678 17.3232 13.375 16.893 13.375H15.75C14.7145 13.375 13.875 14.2145 13.875 15.25V17.125Z" stroke="#059669" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'email' => '<svg width="44" height="44" viewBox="0 0 44 44" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="44" height="44" rx="22" fill="#D1FAE5"/><path d="M30.125 17.125V25.875C30.125 26.9105 29.2855 27.75 28.25 27.75H15.75C14.7145 27.75 13.875 26.9105 13.875 25.875V17.125M30.125 17.125C30.125 16.0895 29.2855 15.25 28.25 15.25H15.75C14.7145 15.25 13.875 16.0895 13.875 17.125M30.125 17.125V17.3273C30.125 17.9784 29.7872 18.5829 29.2327 18.9241L22.9827 22.7703C22.38 23.1411 21.62 23.1411 21.0173 22.7703L14.7673 18.9241C14.2128 18.5829 13.875 17.9784 13.875 17.3273V17.125" stroke="#059669" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'website' => '<svg width="44" height="44" viewBox="0 0 44 44" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="44" height="44" rx="22" fill="#D1FAE5"/><path d="M22 29C25.4938 29 28.4296 26.611 29.2631 23.3775M22 29C18.5062 29 15.5703 26.611 14.7369 23.3775M22 29C24.0711 29 25.75 25.6421 25.75 21.5C25.75 17.3579 24.0711 14 22 14M22 29C19.9289 29 18.25 25.6421 18.25 21.5C18.25 17.3579 19.9289 14 22 14M22 14C24.8043 14 27.2492 15.5391 28.5359 17.8187M22 14C19.1957 14 16.7508 15.5391 15.4641 17.8187M28.5359 17.8187C26.7831 19.3337 24.4986 20.25 22 20.25C19.5014 20.25 17.2169 19.3337 15.4641 17.8187M28.5359 17.8187C29.1497 18.9062 29.5 20.1622 29.5 21.5C29.5 22.1483 29.4177 22.7774 29.2631 23.3775M29.2631 23.3775C27.1111 24.5706 24.6349 25.25 22 25.25C19.3651 25.25 16.8889 24.5706 14.7369 23.3775M14.7369 23.3775C14.5823 22.7774 14.5 22.1483 14.5 21.5C14.5 20.1622 14.8503 18.9062 15.4641 17.8187" stroke="#059669" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'address' => '<svg width="44" height="44" viewBox="0 0 44 44" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="44" height="44" rx="22" fill="#D1FAE5"/><path d="M22 29C25.4938 29 28.4296 26.611 29.2631 23.3775" stroke="#059669" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>'
    ];

    $icons_layout_3 = $icons_layout_1; // Layout 3 uses same icons as layout 1 for now

    switch ( $layout_style ) {
        case 'layout-2':
            $icons = $icons_layout_2;
            break;
        case 'layout-3':
            $icons = $icons_layout_3;
            break;
        default:
            $icons = $icons_layout_1;
            break;
    }

    return isset( $icons[ $type ] ) ? $icons[ $type ] : '';
}

/**
 * Get navigation links and data
 *
 * @param WP_User $user User object
 * @return array Navigation data
 */
function wpuf_ud_get_navigation_data( $user ) {
    $current_user_id = get_current_user_id();
    $account_page_id = wpuf_get_option( 'account_page', 'wpuf_my_account', false );

    $navigation_data = [
        'current_user_id' => $current_user_id,
        'account_page_id' => $account_page_id,
        'is_own_profile' => $current_user_id && $current_user_id === $user->ID,
        'can_message' => defined( 'WPUF_PM_DIR' ) && $current_user_id && $current_user_id !== $user->ID,
        'edit_profile_link' => '',
        'message_link' => ''
    ];

    if ( $account_page_id ) {
        $account_page_link = get_page_link( $account_page_id );
        $navigation_data['edit_profile_link'] = $account_page_link . '?section=edit-profile';
        $navigation_data['message_link'] = $account_page_link . '?section=message#/user/' . $user->ID;
    }

    return $navigation_data;
}

/**
 * Get tab configuration data
 *
 * @param array $template_data Template data
 * @return array Tab configuration
 */
function wpuf_ud_get_tab_configuration( $template_data ) {
    $default_tabs = isset( $template_data['default_tabs'] ) ? $template_data['default_tabs'] : [];
    $default_active_tab = ! empty( $template_data['default_active_tab'] ) ? $template_data['default_active_tab'] : 'about';
    $custom_tab_labels = ! empty( $template_data['custom_tab_labels'] ) ? $template_data['custom_tab_labels'] : [];

    $tab_labels = [
        'about' => __( 'About', 'wpuf-pro' ),
        'posts' => __( 'Posts', 'wpuf-pro' ),
        'comments' => __( 'Comments', 'wpuf-pro' ),
        'file' => __( 'Files', 'wpuf-pro' ),
        'message' => __( 'Message', 'wpuf-pro' ),
        'activity' => __( 'Activity', 'wpuf-pro' )
    ];

    // Merge with custom labels
    $final_labels = array_merge( $tab_labels, $custom_tab_labels );

    return [
        'default_tabs' => $default_tabs,
        'default_active_tab' => $default_active_tab,
        'custom_tab_labels' => $custom_tab_labels,
        'tab_labels' => $final_labels,
        'available_tabs' => [ 'about', 'posts', 'comments', 'file', 'message', 'activity' ]
    ];
}

/**
 * Get tab label for a specific tab
 *
 * @param string $tab_key Tab key
 * @param array $profile_data Profile data containing tab configuration
 * @return string Tab label
 */
function wpuf_ud_get_tab_label( $tab_key, $profile_data ) {
    return isset( $profile_data['tab_config']['tab_labels'][ $tab_key ] )
        ? $profile_data['tab_config']['tab_labels'][ $tab_key ]
        : ucfirst( $tab_key );
}

/**
 * Check if a tab should be displayed
 *
 * @param string $tab_key Tab key
 * @param array $profile_data Profile data
 * @return bool Whether tab should be displayed
 */
function wpuf_ud_should_display_tab( $tab_key, $profile_data ) {
    // Skip activity tab if User Activity module is not active
    if ( $tab_key === 'activity' && ! class_exists( 'WPUF_User_Activity' ) ) {
        return false;
    }

    return in_array( $tab_key, $profile_data['tab_config']['available_tabs'] );
}
