/**
 * WordPress dependencies
 */
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies
 */
import settingsSchema from '../settings-schema.json';

/**
 * Get all fields from the settings schema for a specific block
 * 
 * @param {string} blockName - The block name to filter fields for
 * @returns {Object} Object containing all fields for the block
 */
export function getBlockFields(blockName) {
	const fields = {};
	
	Object.values(settingsSchema.panels).forEach(panel => {
		if (panel.fields) {
			Object.entries(panel.fields).forEach(([fieldName, field]) => {
				// Check if this field targets our block
				if (!field.target_blocks || field.target_blocks.includes(blockName)) {
					fields[fieldName] = field;
				}
			});
		}
	});
	
	return fields;
}

/**
 * Convert schema fields to block attributes
 * 
 * @param {string} blockName - The block name
 * @returns {Object} Block attributes object
 */
export function getBlockAttributes(blockName) {
	const fields = getBlockFields(blockName);
	const attributes = {};
	
	Object.entries(fields).forEach(([fieldName, field]) => {
		// Convert schema field to block attribute
		const attribute = {
			type: getAttributeType(field.type),
		};
		
		// Add default value if present
		if (field.default !== undefined) {
			attribute.default = field.default;
		}
		
		attributes[fieldName] = attribute;
	});
	
	return attributes;
}

/**
 * Convert schema field type to block attribute type
 * 
 * @param {string} schemaType - The schema field type
 * @returns {string} Block attribute type
 */
function getAttributeType(schemaType) {
	switch (schemaType) {
		case 'number':
			return 'number';
		case 'boolean':
			return 'boolean';
		case 'array':
			return 'array';
		case 'object':
			return 'object';
		default:
			return 'string';
	}
}

/**
 * Get default values for a block from the schema
 * 
 * @param {string} blockName - The block name
 * @returns {Object} Default values object
 */
export function getBlockDefaults(blockName) {
	const fields = getBlockFields(blockName);
	const defaults = {};
	
	Object.entries(fields).forEach(([fieldName, field]) => {
		if (field.default !== undefined) {
			defaults[fieldName] = field.default;
		}
	});
	
	return defaults;
}

/**
 * Get field options for select/multiselect controls
 * 
 * @param {Object} field - The schema field
 * @returns {Array} Options array
 */
export function getFieldOptions(field) {
	// If field has inline options, use them
	if (field.options) {
		return field.options;
	}
	
	// If field has options_source, we'll need to get them from a hook
	// This will be implemented when we add the options sources
	if (field.options_source) {
		return [];
	}
	
	return [];
}

/**
 * Get field label with translation
 * 
 * @param {Object} field - The schema field
 * @returns {string} Translated label
 */
export function getFieldLabel(field) {
	return field.label ? __(field.label, 'wpuf-pro') : '';
}

/**
 * Get field description with translation
 * 
 * @param {Object} field - The schema field
 * @returns {string} Translated description
 */
export function getFieldDescription(field) {
	return field.description ? __(field.description, 'wpuf-pro') : '';
}

/**
 * Get panels for a block from the schema
 * 
 * @param {string} blockName - The block name
 * @returns {Object} Panels object
 */
export function getBlockPanels(blockName) {
	const panels = {};
	
	Object.entries(settingsSchema.panels).forEach(([panelName, panel]) => {
		// Check if this panel has fields that target our block
		const hasBlockFields = panel.fields && Object.values(panel.fields).some(field => 
			!field.target_blocks || field.target_blocks.includes(blockName)
		);
		
		if (hasBlockFields) {
			panels[panelName] = {
				...panel,
				fields: Object.fromEntries(
					Object.entries(panel.fields).filter(([fieldName, field]) => 
						!field.target_blocks || field.target_blocks.includes(blockName)
					)
				)
			};
		}
	});
	
	return panels;
} 