import React from 'react';
import { __ } from '@wordpress/i18n';
import MultiSelect from '../MultiSelect';
import Tooltip from '../Tooltip';
import { getOptimalAvatarSize } from '../../utils/avatarSizeHelper';

const AdvancedControl = ({ formData, setFormData }) => {
    // Layout display names from DirectoryLayout
    const layoutDisplayNames = {
        'layout-1': 'Classic Table',
        'layout-2': 'Rounded Sidecard (2x)',
        'layout-3': 'Round Grid (3x)',
        'layout-4': 'Square Grid (3x)',
        'layout-5': 'Round Grid (4x)',
        'layout-6': 'Square Grid (4x)'
    };
    
    // Extract custom meta fields from about_tabs_content
    const getCustomMetaFields = () => {
        const metaFields = [];
        if (formData.about_tabs_content && Array.isArray(formData.about_tabs_content)) {
            formData.about_tabs_content.forEach(item => {
                if ((item.type === 'meta_field' || item.type === 'meta') && item.meta_key) {
                    // Add meta field with its label or meta_key as fallback
                    metaFields.push({
                        value: item.meta_key,
                        label: item.label || item.meta_key
                    });
                }
            });
        }
        return metaFields;
    };
    
    // Initialize UI fields from schema fields when formData changes
    React.useEffect(() => {
        let updates = {};
        
        // Initialize default_sort_by from orderby if not set
        if (formData.default_sort_by === undefined && formData.orderby !== undefined) {
            updates.default_sort_by = formData.orderby;
        }
        
        // Initialize default_sort_order from order if not set
        if (formData.default_sort_order === undefined && formData.order !== undefined) {
            updates.default_sort_order = formData.order;
        }
        
        // Initialize users_per_page from max_item_per_page if not set
        if (formData.users_per_page === undefined && formData.max_item_per_page !== undefined) {
            updates.users_per_page = formData.max_item_per_page;
        }
        
        // Initialize user_limit_type based on max_item if not set
        if (formData.user_limit_type === undefined) {
            if (formData.max_item === -1 || formData.max_item === undefined) {
                updates.user_limit_type = 'all';
            } else if (formData.max_item > 0) {
                updates.user_limit_type = 'specific';
                updates.max_users = formData.max_item;
            }
        }
        
        // Apply updates if needed
        if (Object.keys(updates).length > 0) {
            setFormData(prev => ({ ...prev, ...updates }), { skipDirtyCheck: true });
        }
    }, [formData.orderby, formData.order, formData.max_item_per_page, formData.max_item]); // Re-run when these backend fields change


    const handleChange = (e) => {
        const { name, value, type, checked } = e.target;
        
        // Handle special cases where we need to update multiple fields
        if (name === 'users_per_page') {
            setFormData(prev => ({
                ...prev,
                users_per_page: value,
                max_item_per_page: value
            }));
        } else if (name === 'max_item') {
            // Convert empty string to null, handle -1 specially, and ensure numeric values
            let numericValue;
            if (value === '' || value === null || value === undefined) {
                numericValue = null;
            } else {
                numericValue = parseInt(value);
                // Handle NaN case
                if (isNaN(numericValue)) {
                    numericValue = null;
                }
            }
            setFormData(prev => ({
                ...prev,
                max_item: numericValue
            }));
        } else if (name === 'default_sort_by') {
            setFormData(prev => ({
                ...prev,
                default_sort_by: value,
                orderby: value
            }));
        } else if (name === 'default_sort_order') {
            setFormData(prev => ({
                ...prev,
                default_sort_order: value,
                order: value
            }));
        } else if (name === 'directory_layout') {
            const optimalSize = getOptimalAvatarSize(value);
            setFormData(prev => ({
                ...prev,
                [name]: value,
                avatar_size: optimalSize
            }));
        } else {
            setFormData(prev => ({
                ...prev,
                [name]: type === 'checkbox' ? checked : value
            }));
        }
    };

    // Set initial avatar size based on current layout if avatar_size is not set
    React.useEffect(() => {
        if (formData.directory_layout && !formData.avatar_size) {
            const optimalSize = getOptimalAvatarSize(formData.directory_layout);
            setFormData(prev => ({
                ...prev,
                avatar_size: optimalSize
            }), { skipDirtyCheck: true });
        }
    }, [formData.directory_layout, formData.avatar_size, setFormData]);

    return (
            <div className="wpuf-w-full wpuf-bg-white wpuf-max-w-3xl wpuf-mx-auto">
                <h2 className="wpuf-text-center"
                    style={{
                        
                        fontWeight: 400,
                        fontSize: '20px',
                        lineHeight: '36px',
                        letterSpacing: '0%',
                        textAlign: 'center',
                        color: '#000000'
                    }}
                >
                    {__('Advanced Control', 'wpuf-pro')}
                </h2>
                <p className="wpuf-text-center wpuf-mt-2"
                    style={{
                        
                        fontWeight: 400,
                        fontSize: '14px',
                        lineHeight: '20px',
                        letterSpacing: '0%',
                        textAlign: 'center',
                        color: '#64748B'
                    }}
                >{__('Manage how your user directory looks and behaves, from permalinks, socials and avatars display to sorting, for a clean, intuitive directory experience', 'wpuf-pro')}</p>
                
                <div className="wpuf-mt-12">
                    {/* Users Per Page */}
                    <div className="wpuf-mt-[25px]">
                        <label htmlFor="users_per_page" className="wpuf-flex wpuf-text-left wpuf-text-sm wpuf-font-medium wpuf-text-gray-700 wpuf-mb-3">
                            {__('Users Per Page', 'wpuf-pro')}
                            <Tooltip className="wpuf-ml-2" content={__('Set how many users will be displayed per page in this directory', 'wpuf-pro')}>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.375 9.375L9.40957 9.35771C9.88717 9.11891 10.4249 9.55029 10.2954 10.0683L9.70458 12.4317C9.57507 12.9497 10.1128 13.3811 10.5904 13.1423L10.625 13.125M17.5 10C17.5 14.1421 14.1421 17.5 10 17.5C5.85786 17.5 2.5 14.1421 2.5 10C2.5 5.85786 5.85786 2.5 10 2.5C14.1421 2.5 17.5 5.85786 17.5 10ZM10 6.875H10.0063V6.88125H10V6.875Z" stroke="#9CA3AF" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"></path></svg>
                            </Tooltip>
                        </label>
                        <input
                            type="number"
                            id="users_per_page"
                            name="users_per_page"
                            className="wpuf-block wpuf-min-w-full wpuf-m-0 wpuf-leading-none wpuf-text-gray-700 placeholder:wpuf-text-gray-400 wpuf-max-w-full focus:wpuf-ring-transparent"
                            style={{
                                maxWidth: '760px',
                                width: '100%',
                                height: '42px',
                                borderRadius: '6px',
                                borderWidth: '1px',
                                paddingTop: '9px',
                                paddingRight: '13px',
                                paddingBottom: '9px',
                                paddingLeft: '13px',
                                backgroundColor: '#FFFFFF',
                                borderColor: '#CBD5E1',
                                borderStyle: 'solid',
                                opacity: 1,
                                boxSizing: 'border-box',
                                fontSize: '16px'
                            }}
                            value={formData.users_per_page || formData.max_item_per_page || 10}
                            onChange={handleChange}
                            min="1"
                            max="100"
                            placeholder={__('Enter number of users per page', 'wpuf-pro')}
                        />
                    </div>

                    {/* Number of Users in Directory */}
                    <div className="wpuf-mt-[25px]">
                        <label htmlFor="max_item" className="wpuf-flex wpuf-text-left wpuf-text-sm wpuf-font-medium wpuf-text-gray-700 wpuf-mb-3">
                            {__('Number of Users in Directory', 'wpuf-pro')}
                            <Tooltip className="wpuf-ml-2" content={__('Set the maximum number of users to display in this directory. Enter -1 or leave empty to show all users.', 'wpuf-pro')}>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.375 9.375L9.40957 9.35771C9.88717 9.11891 10.4249 9.55029 10.2954 10.0683L9.70458 12.4317C9.57507 12.9497 10.1128 13.3811 10.5904 13.1423L10.625 13.125M17.5 10C17.5 14.1421 14.1421 17.5 10 17.5C5.85786 17.5 2.5 14.1421 2.5 10C2.5 5.85786 5.85786 2.5 10 2.5C14.1421 2.5 17.5 5.85786 17.5 10ZM10 6.875H10.0063V6.88125H10V6.875Z" stroke="#9CA3AF" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"></path></svg>
                            </Tooltip>
                        </label>
                        <input
                            type="number"
                            id="max_item"
                            name="max_item"
                            className="wpuf-block wpuf-min-w-full wpuf-m-0 wpuf-leading-none wpuf-text-gray-700 placeholder:wpuf-text-gray-400 wpuf-max-w-full focus:wpuf-ring-transparent"
                            style={{
                                maxWidth: '760px',
                                width: '100%',
                                height: '42px',
                                borderRadius: '6px',
                                borderWidth: '1px',
                                paddingTop: '9px',
                                paddingRight: '13px',
                                paddingBottom: '9px',
                                paddingLeft: '13px',
                                backgroundColor: '#FFFFFF',
                                borderColor: '#CBD5E1',
                                borderStyle: 'solid',
                                opacity: 1,
                                boxSizing: 'border-box',
                                fontSize: '16px'
                            }}
                            value={formData.max_item || ''}
                            onChange={handleChange}
                            placeholder={__('Enter -1 for all users', 'wpuf-pro')}
                        />
                    </div>

                    {/* Default Sort - Second */}
                    <div className="wpuf-mt-[25px]">
                        <label htmlFor="default_sort" className="wpuf-flex wpuf-text-left wpuf-text-sm wpuf-font-medium wpuf-text-gray-700 wpuf-mb-3">
                            {__('Default Sort', 'wpuf-pro')}
                            <Tooltip className="wpuf-ml-2" content={__('Select the field by which users will be sorted by default. Supports standard WordPress fields and custom meta', 'wpuf-pro')}>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.375 9.375L9.40957 9.35771C9.88717 9.11891 10.4249 9.55029 10.2954 10.0683L9.70458 12.4317C9.57507 12.9497 10.1128 13.3811 10.5904 13.1423L10.625 13.125M17.5 10C17.5 14.1421 14.1421 17.5 10 17.5C5.85786 17.5 2.5 14.1421 2.5 10C2.5 5.85786 5.85786 2.5 10 2.5C14.1421 2.5 17.5 5.85786 17.5 10ZM10 6.875H10.0063V6.88125H10V6.875Z" stroke="#9CA3AF" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"></path></svg>
                            </Tooltip>
                        </label>
                        <div className="wpuf-grid wpuf-grid-cols-2 wpuf-gap-3">
                            {/* Sort By */}
                            <select
                                id="default_sort_by"
                                name="default_sort_by"
                                className="wpuf-flex-1 wpuf-text-gray-700 wpuf-font-normal focus:wpuf-ring-transparent wpuf-border"
                                style={{
                                    maxWidth: '793px',
                                    width: '100%',
                                    height: '42px',
                                    borderRadius: '6px',
                                    borderWidth: '1px',
                                    paddingTop: '9px',
                                    paddingRight: '37px',
                                    paddingBottom: '9px',
                                    paddingLeft: '13px',
                                    backgroundColor: '#FFFFFF',
                                    borderColor: '#CBD5E1',
                                    borderStyle: 'solid',
                                    opacity: 1,
                                    boxSizing: 'border-box',
                                    fontSize: '16px',
                                    lineHeight: '1',
                                    display: 'flex',
                                    alignItems: 'center',
                                    backgroundImage: 'url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' fill=\'none\' stroke=\'%23374151\' viewBox=\'0 0 24 24\'%3E%3Cpath stroke-linecap=\'round\' stroke-linejoin=\'round\' stroke-width=\'2\' d=\'M19 9l-7 7-7-7\'/%3E%3C/svg%3E")',
                                    backgroundRepeat: 'no-repeat',
                                    backgroundPosition: 'right 13px center',
                                    backgroundSize: '16px',
                                    appearance: 'none',
                                    WebkitAppearance: 'none',
                                    MozAppearance: 'none'
                                }}
                                value={formData.default_sort_by || formData.orderby || 'id'}
                                onChange={handleChange}
                            >
                                <optgroup label={__('Standard Fields', 'wpuf-pro')}>
                                    <option value="id">{__('User ID', 'wpuf-pro')}</option>
                                    <option value="user_login">{__('Username', 'wpuf-pro')}</option>
                                    <option value="user_email">{__('Email', 'wpuf-pro')}</option>
                                    <option value="display_name">{__('Display Name', 'wpuf-pro')}</option>
                                    <option value="registered">{__('Registration Date', 'wpuf-pro')}</option>
                                </optgroup>
                                {getCustomMetaFields().length > 0 && (
                                    <optgroup label={__('Custom Meta Fields', 'wpuf-pro')}>
                                        {getCustomMetaFields().map(field => (
                                            <option key={field.value} value={field.value}>
                                                {field.label}
                                            </option>
                                        ))}
                                    </optgroup>
                                )}
                            </select>
                            
                            {/* Sort Order */}
                            <select
                                id="default_sort_order"
                                name="default_sort_order"
                                className="wpuf-flex-1 wpuf-text-gray-700 wpuf-font-normal focus:wpuf-ring-transparent wpuf-border"
                                style={{
                                    maxWidth: '793px',
                                    width: '100%',
                                    height: '42px',
                                    borderRadius: '6px',
                                    borderWidth: '1px',
                                    paddingTop: '9px',
                                    paddingRight: '37px',
                                    paddingBottom: '9px',
                                    paddingLeft: '13px',
                                    backgroundColor: '#FFFFFF',
                                    borderColor: '#CBD5E1',
                                    borderStyle: 'solid',
                                    opacity: 1,
                                    boxSizing: 'border-box',
                                    fontSize: '16px',
                                    lineHeight: '1',
                                    display: 'flex',
                                    alignItems: 'center',
                                    backgroundImage: 'url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' fill=\'none\' stroke=\'%23374151\' viewBox=\'0 0 24 24\'%3E%3Cpath stroke-linecap=\'round\' stroke-linejoin=\'round\' stroke-width=\'2\' d=\'M19 9l-7 7-7-7\'/%3E%3C/svg%3E")',
                                    backgroundRepeat: 'no-repeat',
                                    backgroundPosition: 'right 13px center',
                                    backgroundSize: '16px',
                                    appearance: 'none',
                                    WebkitAppearance: 'none',
                                    MozAppearance: 'none'
                                }}
                                value={formData.default_sort_order || formData.order || 'desc'}
                                onChange={handleChange}
                            >
                                <option value="asc">{__('ASC', 'wpuf-pro')}</option>
                                <option value="desc">{__('DESC', 'wpuf-pro')}</option>
                            </select>
                        </div>
                    </div>

                    {/* Single Profile Permalink - Third */}
                    <div className="wpuf-mt-[25px]">
                        <label htmlFor="profile_base" className="wpuf-flex wpuf-text-left wpuf-text-sm wpuf-font-medium wpuf-text-gray-700 wpuf-mb-3">
                            {__('Profile Permalink Structure', 'wpuf-pro')}
                            <Tooltip className="wpuf-ml-2" content={__('Set how profile links are generated: use username (e.g., john-doe) for readable URLs or user ID (e.g., 123) for system-based links', 'wpuf-pro')}>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.375 9.375L9.40957 9.35771C9.88717 9.11891 10.4249 9.55029 10.2954 10.0683L9.70458 12.4317C9.57507 12.9497 10.1128 13.3811 10.5904 13.1423L10.625 13.125M17.5 10C17.5 14.1421 14.1421 17.5 10 17.5C5.85786 17.5 2.5 14.1421 2.5 10C2.5 5.85786 5.85786 2.5 10 2.5C14.1421 2.5 17.5 5.85786 17.5 10ZM10 6.875H10.0063V6.88125H10V6.875Z" stroke="#9CA3AF" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"></path></svg>
                            </Tooltip>
                        </label>
                        <select
                            id="profile_base"
                            name="profile_base"
                            className="wpuf-block wpuf-min-w-full wpuf-text-gray-700 wpuf-font-normal focus:wpuf-ring-transparent wpuf-border"
                            style={{
                                maxWidth: '793px',
                                width: '100%',
                                height: '42px',
                                borderRadius: '6px',
                                borderWidth: '1px',
                                paddingTop: '9px',
                                paddingRight: '37px',
                                paddingBottom: '9px',
                                paddingLeft: '13px',
                                backgroundColor: '#FFFFFF',
                                borderColor: '#CBD5E1',
                                borderStyle: 'solid',
                                opacity: 1,
                                boxSizing: 'border-box',
                                fontSize: '16px',
                                lineHeight: '1',
                                display: 'flex',
                                alignItems: 'center',
                                backgroundImage: 'url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' fill=\'none\' stroke=\'%23374151\' viewBox=\'0 0 24 24\'%3E%3Cpath stroke-linecap=\'round\' stroke-linejoin=\'round\' stroke-width=\'2\' d=\'M19 9l-7 7-7-7\'/%3E%3C/svg%3E")',
                                backgroundRepeat: 'no-repeat',
                                backgroundPosition: 'right 13px center',
                                backgroundSize: '16px',
                                appearance: 'none',
                                WebkitAppearance: 'none',
                                MozAppearance: 'none'
                            }}
                            value={formData.profile_base || 'username'}
                            onChange={handleChange}
                        >
                            {typeof wpuf_ud !== 'undefined' && wpuf_ud.profile_base && Object.entries(wpuf_ud.profile_base).map(([base, label]) => (
                                <option key={base} value={base}>{label}</option>
                            ))}
                        </select>
                    </div>

                    {/* Profile Gallery Image Size - Fourth */}
                    <div className="wpuf-mt-[25px]">
                        <label htmlFor="profile_size" className="wpuf-flex wpuf-text-left wpuf-text-sm wpuf-font-medium wpuf-text-gray-700 wpuf-mb-3">
                            {__('Profile Gallery Image Size', 'wpuf-pro')}
                            <Tooltip className="wpuf-ml-2" content={__('Select the default size for profile gallery images', 'wpuf-pro')}>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.375 9.375L9.40957 9.35771C9.88717 9.11891 10.4249 9.55029 10.2954 10.0683L9.70458 12.4317C9.57507 12.9497 10.1128 13.3811 10.5904 13.1423L10.625 13.125M17.5 10C17.5 14.1421 14.1421 17.5 10 17.5C5.85786 17.5 2.5 14.1421 2.5 10C2.5 5.85786 5.85786 2.5 10 2.5C14.1421 2.5 17.5 5.85786 17.5 10ZM10 6.875H10.0063V6.88125H10V6.875Z" stroke="#9CA3AF" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"></path></svg>
                            </Tooltip>
                        </label>
                        <select
                            id="profile_size"
                            name="profile_size"
                            className="wpuf-block wpuf-min-w-full wpuf-text-gray-700 wpuf-font-normal focus:wpuf-ring-transparent wpuf-border"
                            style={{
                                maxWidth: '793px',
                                width: '100%',
                                height: '42px',
                                borderRadius: '6px',
                                borderWidth: '1px',
                                paddingTop: '9px',
                                paddingRight: '37px',
                                paddingBottom: '9px',
                                paddingLeft: '13px',
                                backgroundColor: '#FFFFFF',
                                borderColor: '#CBD5E1',
                                borderStyle: 'solid',
                                opacity: 1,
                                boxSizing: 'border-box',
                                fontSize: '16px',
                                lineHeight: '1',
                                display: 'flex',
                                alignItems: 'center',
                                backgroundImage: 'url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' fill=\'none\' stroke=\'%23374151\' viewBox=\'0 0 24 24\'%3E%3Cpath stroke-linecap=\'round\' stroke-linejoin=\'round\' stroke-width=\'2\' d=\'M19 9l-7 7-7-7\'/%3E%3C/svg%3E")',
                                backgroundRepeat: 'no-repeat',
                                backgroundPosition: 'right 13px center',
                                backgroundSize: '16px',
                                appearance: 'none',
                                WebkitAppearance: 'none',
                                MozAppearance: 'none'
                            }}
                            value={formData.profile_size || 'thumbnail'}
                            onChange={handleChange}
                        >
                            {typeof wpuf_ud !== 'undefined' && wpuf_ud.profile_size && Object.entries(wpuf_ud.profile_size).map(([size, label]) => (
                                <option key={size} value={size}>{label}</option>
                            ))}
                        </select>
                    </div>

                    {/* Select Social Fields - Fifth */}
                    <div className="wpuf-mt-[25px]">
                        <label htmlFor="social_fields" className="wpuf-flex wpuf-text-left wpuf-text-sm wpuf-font-medium wpuf-text-gray-700 wpuf-mb-3">
                            {__('Select Social Profiles', 'wpuf-pro')}
                            <Tooltip className="wpuf-ml-2" content={__('Select the social media fields to display and ensure the corresponding WPUF social fields are configured in the WPUF Account', 'wpuf-pro')}>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.375 9.375L9.40957 9.35771C9.88717 9.11891 10.4249 9.55029 10.2954 10.0683L9.70458 12.4317C9.57507 12.9497 10.1128 13.3811 10.5904 13.1423L10.625 13.125M17.5 10C17.5 14.1421 14.1421 17.5 10 17.5C5.85786 17.5 2.5 14.1421 2.5 10C2.5 5.85786 5.85786 2.5 10 2.5C14.1421 2.5 17.5 5.85786 17.5 10ZM10 6.875H10.0063V6.88125H10V6.875Z" stroke="#9CA3AF" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"></path></svg>
                            </Tooltip>
                        </label>
                        <MultiSelect
                            selectedLabel="profiles"
                            options={{
                                facebook: {
                                    label: 'Facebook',
                                    icon: (
                                        <svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg" className="wpuf-mr-2">
                                            <path fillRule="evenodd" clipRule="evenodd" d="M13.6668 6.99992C13.6668 3.31792 10.6822 0.333252 7.00016 0.333252C3.31816 0.333252 0.333496 3.31792 0.333496 6.99992C0.333496 10.3273 2.7715 13.0853 5.95883 13.5853V8.92725H4.2655V6.99992H5.95883V5.53125C5.95883 3.86059 6.9535 2.93792 8.47683 2.93792C9.20616 2.93792 9.96883 3.06792 9.96883 3.06792V4.70792H9.12883C8.30016 4.70792 8.04216 5.22192 8.04216 5.74925V6.99992H9.89083L9.5955 8.92659H8.04216V13.5853C11.2288 13.0853 13.6668 10.3273 13.6668 6.99992Z" fill="#94A3B8"/>
                                        </svg>
                                    )
                                },
                                linkedin: {
                                    label: 'LinkedIn',
                                    icon: (
                                        <svg width="12" height="11" viewBox="0 0 12 11" fill="none" xmlns="http://www.w3.org/2000/svg" className="wpuf-mr-2">
                                            <path d="M11.2324 6.79725V10.6667H8.98893V7.05658C8.98893 6.1501 8.66502 5.53106 7.85274 5.53106C7.23286 5.53106 6.86462 5.94783 6.70199 6.35137C6.64293 6.49559 6.62771 6.69586 6.62771 6.89814V10.6666H4.38411C4.38411 10.6666 4.41422 4.55214 4.38411 3.91922H6.62788V4.87538C6.62336 4.88291 6.617 4.89027 6.61299 4.89747H6.62788V4.87538C6.92602 4.41662 7.45772 3.76078 8.64979 3.76078C10.1258 3.76078 11.2324 4.72514 11.2324 6.79725ZM2.03711 0.666748C1.26967 0.666748 0.767578 1.17051 0.767578 1.83238C0.767578 2.4802 1.25511 2.99852 2.00766 2.99852H2.02222C2.80472 2.99852 3.29125 2.4802 3.29125 1.83238C3.27636 1.17051 2.80472 0.666748 2.03711 0.666748ZM0.900922 10.6667H3.14369V3.91922H0.900922V10.6667Z" fill="#94A3B8"/>
                                        </svg>
                                    )
                                },
                                twitter: {
                                    label: 'X (Formally Twitter)',
                                    icon: (
                                        <svg width="12" height="12" viewBox="0 0 12 12" fill="none" xmlns="http://www.w3.org/2000/svg" className="wpuf-mr-2">
                                            <path d="M0.000114006 0.59065C0.00715826 0.60119 1.02857 1.93813 2.27189 3.55967L4.53309 6.50937L2.38459 8.95838C1.20292 10.3059 0.236097 11.4127 0.236097 11.4179C0.236097 11.4232 0.646424 11.4285 1.15009 11.4285H2.06232L3.72829 9.5311L5.39425 7.63374L5.62143 7.92888C5.7447 8.09051 6.35051 8.87757 6.96512 9.68043C7.58149 10.4815 8.13447 11.2036 8.1961 11.2844L8.30705 11.4285H10.1544C11.1688 11.4285 12 11.4232 12 11.4144C12 11.4074 11.1353 10.2637 10.0804 8.87229C6.99682 4.80702 7.27331 5.1777 7.29972 5.14432C7.31381 5.12676 8.20843 4.10604 9.28796 2.87276C10.3675 1.63947 11.2621 0.618758 11.2762 0.60119C11.2991 0.573081 11.248 0.571324 10.3816 0.571324H9.46407L7.94603 2.30179C7.11129 3.25399 6.42447 4.03226 6.41919 4.0305C6.41391 4.0305 6.38045 3.99361 6.34699 3.94793C6.31177 3.90225 5.72181 3.12398 5.03499 2.21746L3.7864 0.571324H1.88797C0.491451 0.571324 -0.00869131 0.576594 0.000114006 0.59065ZM3.66313 2.18233C3.89559 2.48977 5.03323 3.99009 6.19025 5.51677C8.61347 8.71242 9.64898 10.0775 9.75817 10.2215L9.83565 10.3217L9.31966 10.3182L8.80367 10.3129L8.5219 9.94395C8.36517 9.74016 6.87002 7.79009 5.19701 5.60988C3.524 3.42791 2.15566 1.63947 2.15566 1.6342C2.15566 1.62893 2.40044 1.62542 2.69806 1.62542H3.24047L3.66313 2.18233Z" fill="#94A3B8"/>
                                        </svg>
                                    )
                                },
                                instagram: {
                                    label: 'Instagram',
                                    icon: (
                                        <svg width="14" height="14" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" className="wpuf-mr-2">
                                            <path d="M8.20967 1.33325C9.82967 1.33325 10.0657 1.34192 10.7483 1.37325C11.4577 1.40592 11.9423 1.51859 12.3663 1.68325C12.8113 1.8505 13.2145 2.11282 13.5477 2.45192C13.8868 2.7851 14.1491 3.18825 14.3163 3.63325C14.481 4.05725 14.5937 4.54192 14.6263 5.25125C14.6583 5.96259 14.6663 6.18925 14.6663 7.99992V8.05325C14.6663 9.81525 14.6583 10.0446 14.6263 10.7486C14.5937 11.4579 14.481 11.9426 14.3163 12.3666C14.1491 12.8116 13.8868 13.2147 13.5477 13.5479C13.2145 13.887 12.8113 14.1493 12.3663 14.3166C11.9423 14.4813 11.4577 14.5939 10.7483 14.6266C10.037 14.6586 9.81034 14.6666 7.99967 14.6666H7.94634C6.18434 14.6666 5.95501 14.6586 5.25101 14.6266C4.54167 14.5939 4.05701 14.4813 3.63301 14.3166C3.18801 14.1493 2.78486 13.887 2.45167 13.5479C2.11258 13.2147 1.85026 12.8116 1.68301 12.3666C1.51834 11.9426 1.40567 11.4579 1.37301 10.7486C1.34167 10.0659 1.33301 9.82925 1.33301 8.20992V7.78992C1.33301 6.16992 1.34167 5.93392 1.37301 5.25125C1.40567 4.54192 1.51834 4.05725 1.68301 3.63325C1.85026 3.18825 2.11258 2.7851 2.45167 2.45192C2.78486 2.11282 3.18801 1.8505 3.63301 1.68325C4.05701 1.51859 4.54167 1.40592 5.25101 1.37325C5.93367 1.34192 6.17034 1.33325 7.78967 1.33325H8.20967ZM8.15567 2.53459H7.84367C6.20634 2.53459 5.98767 2.54192 5.30567 2.57325C4.65567 2.60325 4.30301 2.71125 4.06767 2.80259C3.75634 2.92392 3.53434 3.06792 3.30101 3.30125C3.06767 3.53459 2.92367 3.75659 2.80234 4.06792C2.71101 4.30325 2.60234 4.65592 2.57301 5.30592C2.54167 5.98792 2.53434 6.20659 2.53434 7.84392V8.15592C2.53434 9.79325 2.54167 10.0119 2.57301 10.6939C2.60301 11.3439 2.71101 11.6966 2.80234 11.9319C2.92367 12.2426 3.06834 12.4653 3.30101 12.6986C3.53434 12.9319 3.75634 13.0759 4.06767 13.1973C4.30301 13.2886 4.65567 13.3973 5.30567 13.4266C6.00834 13.4586 6.21901 13.4653 7.99967 13.4653H8.05301C9.78434 13.4653 9.99767 13.4586 10.693 13.4266C11.3437 13.3966 11.6963 13.2886 11.9317 13.1973C12.2423 13.0759 12.465 12.9319 12.6983 12.6986C12.9317 12.4653 13.0757 12.2433 13.197 11.9319C13.2883 11.6966 13.397 11.3439 13.4263 10.6939C13.4583 9.99059 13.465 9.78059 13.465 7.99992V7.94659C13.465 6.21525 13.4583 6.00192 13.4263 5.30659C13.3963 4.65592 13.2883 4.30325 13.197 4.06792C13.0902 3.7783 12.9198 3.51632 12.6983 3.30125C12.4833 3.07982 12.2213 2.90943 11.9317 2.80259C11.6963 2.71125 11.3437 2.60259 10.6937 2.57325C10.0117 2.54192 9.79301 2.53459 8.15567 2.53459ZM7.99967 4.57659C8.44923 4.57659 8.89439 4.66513 9.30973 4.83717C9.72507 5.00921 10.1025 5.26137 10.4203 5.57926C10.7382 5.89714 10.9904 6.27453 11.1624 6.68987C11.3345 7.1052 11.423 7.55036 11.423 7.99992C11.423 8.44948 11.3345 8.89463 11.1624 9.30997C10.9904 9.72531 10.7382 10.1027 10.4203 10.4206C10.1025 10.7385 9.72507 10.9906 9.30973 11.1627C8.89439 11.3347 8.44923 11.4233 7.99967 11.4233C7.09175 11.4233 6.22101 11.0626 5.57901 10.4206C4.93701 9.77858 4.57634 8.90784 4.57634 7.99992C4.57634 7.09199 4.93701 6.22126 5.57901 5.57926C6.22101 4.93726 7.09175 4.57659 7.99967 4.57659ZM7.99967 5.77792C7.41036 5.77792 6.84519 6.01202 6.42848 6.42873C6.01178 6.84543 5.77767 7.41061 5.77767 7.99992C5.77767 8.58923 6.01178 9.1544 6.42848 9.57111C6.84519 9.98782 7.41036 10.2219 7.99967 10.2219C8.58899 10.2219 9.15416 9.98782 9.57087 9.57111C9.98757 9.1544 10.2217 8.58923 10.2217 7.99992C10.2217 7.41061 9.98757 6.84543 9.57087 6.42873C9.15416 6.01202 8.58899 5.77792 7.99967 5.77792ZM11.5583 3.64125C11.7705 3.64125 11.974 3.72554 12.124 3.87557C12.2741 4.0256 12.3583 4.22908 12.3583 4.44125C12.3583 4.65343 12.2741 4.85691 12.124 5.00694C11.974 5.15697 11.7705 5.24125 11.5583 5.24125C11.3462 5.24125 11.1427 5.15697 10.9927 5.00694C10.8426 4.85691 10.7583 4.65343 10.7583 4.44125C10.7583 4.22908 10.8426 4.0256 10.9927 3.87557C11.1427 3.72554 11.3462 3.64125 11.5583 3.64125Z" fill="#94A3B8"/>
                                        </svg>
                                    )
                                }
                            }}
                            value={formData.social_fields || []}
                            onChange={social_fields => setFormData(prev => ({ ...prev, social_fields }))}
                            placeholder={__('Select social fields...', 'wpuf-pro')}
                        />
                    </div>

                    {/* Show Avatar in Table Layout - Sixth */}
                    {formData.directory_layout === 'layout-1' && (() => {
                        // Determine the checkbox state more clearly
                        const isChecked = formData.show_avatar_in_table === undefined 
                            ? true  // Default to true if undefined
                            : !!formData.show_avatar_in_table;  // Otherwise use the actual value
                        
                        return (
                            <div className="wpuf-mt-[25px]">
                                <label className="wpuf-flex wpuf-items-center wpuf-text-sm wpuf-text-gray-700 wpuf-cursor-pointer">
                                    <div className="wpuf-relative wpuf-mr-3">
                                        <input
                                            type="checkbox"
                                            name="show_avatar_in_table"
                                            className="wpuf-sr-only"
                                            checked={isChecked}
                                            onChange={handleChange}
                                        />
                                        <div className={`wpuf-w-5 wpuf-h-5 wpuf-border-2 wpuf-rounded wpuf-flex wpuf-items-center wpuf-justify-center wpuf-transition-colors ${
                                            isChecked
                                                ? 'wpuf-bg-emerald-600 wpuf-border-emerald-600' 
                                                : 'wpuf-bg-white wpuf-border-gray-300 hover:wpuf-border-emerald-400'
                                        }`}>
                                            {isChecked && (
                                                <svg className="wpuf-w-3 wpuf-h-3 wpuf-text-white" fill="currentColor" viewBox="0 0 20 20">
                                                    <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                                                </svg>
                                            )}
                                        </div>
                                    </div>
                                    {__('Show avatar in table layout', 'wpuf-pro')}
                                    <Tooltip className="wpuf-ml-2" content={__('Enable or disable avatars for the table layout to control how compact the user listing appears', 'wpuf-pro')}>
                                        <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.375 9.375L9.40957 9.35771C9.88717 9.11891 10.4249 9.55029 10.2954 10.0683L9.70458 12.4317C9.57507 12.9497 10.1128 13.3811 10.5904 13.1423L10.625 13.125M17.5 10C17.5 14.1421 14.1421 17.5 10 17.5C5.85786 17.5 2.5 14.1421 2.5 10C2.5 5.85786 5.85786 2.5 10 2.5C14.1421 2.5 17.5 5.85786 17.5 10ZM10 6.875H10.0063V6.88125H10V6.875Z" stroke="#9CA3AF" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"></path></svg>
                                    </Tooltip>
                                </label>
                            </div>
                        );
                    })()}

                    {/* Avatar Size - Seventh (as radio buttons) */}
                    <div className="wpuf-mt-[25px]">
                        <label className="wpuf-flex wpuf-text-left wpuf-text-sm wpuf-font-medium wpuf-text-gray-700 wpuf-mb-3">
                            {__('Avatar Size', 'wpuf-pro')}
                            <Tooltip className="wpuf-ml-2" content={__('Set the avatar size to display in directory layouts (this does not affect individual user profiles)', 'wpuf-pro')}>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.375 9.375L9.40957 9.35771C9.88717 9.11891 10.4249 9.55029 10.2954 10.0683L9.70458 12.4317C9.57507 12.9497 10.1128 13.3811 10.5904 13.1423L10.625 13.125M17.5 10C17.5 14.1421 14.1421 17.5 10 17.5C5.85786 17.5 2.5 14.1421 2.5 10C2.5 5.85786 5.85786 2.5 10 2.5C14.1421 2.5 17.5 5.85786 17.5 10ZM10 6.875H10.0063V6.88125H10V6.875Z" stroke="#9CA3AF" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"></path></svg>
                            </Tooltip>
                        </label>
                        {formData.directory_layout && (
                            <div 
                                className="wpuf-mb-3 wpuf-flex wpuf-items-center"
                                style={{
                                    width: '793px',
                                    height: '52px',
                                    borderRadius: '6px',
                                    opacity: 1,
                                    gap: '12px',
                                    padding: '16px',
                                    backgroundColor: '#EFF6FF'
                                }}
                            >
                                <svg className="wpuf-flex-shrink-0" width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path fillRule="evenodd" clipRule="evenodd" d="M18 10C18 14.4183 14.4183 18 10 18C5.58172 18 2 14.4183 2 10C2 5.58172 5.58172 2 10 2C14.4183 2 18 5.58172 18 10ZM11 6C11 6.55228 10.5523 7 10 7C9.44772 7 9 6.55228 9 6C9 5.44772 9.44772 5 10 5C10.5523 5 11 5.44772 11 6ZM9 9C8.44772 9 8 9.44772 8 10C8 10.5523 8.44772 11 9 11V14C9 14.5523 9.44772 15 10 15H11C11.5523 15 12 14.5523 12 14C12 13.4477 11.5523 13 11 13V10C11 9.44772 10.5523 9 10 9H9Z" fill="#1E40AF"/>
                                </svg>
                                <span 
                                    style={{
                                        
                                        fontWeight: 500,
                                        fontSize: '14px',
                                        lineHeight: '20px',
                                        letterSpacing: '0%',
                                        color: '#1E40AF'
                                    }}
                                >
                                    {__('Recommended for', 'wpuf-pro')} {layoutDisplayNames[formData.directory_layout] || formData.directory_layout} layout: {getOptimalAvatarSize(formData.directory_layout)}×{getOptimalAvatarSize(formData.directory_layout)}px
                                </span>
                            </div>
                        )}
                        <div className="wpuf-flex wpuf-flex-wrap wpuf-gap-1">
                            {['32', '48', '80', '128', '160', '192', '265'].map(size => {
                                const sizeLabel = `${size}×${size}`;
                                const isRecommended = formData.directory_layout && size === getOptimalAvatarSize(formData.directory_layout);
                                const isSelected = formData.avatar_size === size;
                                
                                return (
                                    <label
                                        key={size}
                                        className={`wpuf-relative wpuf-flex wpuf-flex-col wpuf-items-center wpuf-justify-center wpuf-w-[103px] wpuf-h-[68px] wpuf-border wpuf-rounded-lg wpuf-cursor-pointer wpuf-transition-all wpuf-p-3 wpuf-group ${
                                            isSelected 
                                                ? 'wpuf-border-emerald-500 wpuf-bg-emerald-50 wpuf-text-emerald-600' 
                                                : isRecommended
                                                    ? 'wpuf-border-green-800 wpuf-bg-green-50 wpuf-text-grenn-600 hover:wpuf-border-green-500'
                                                    : 'wpuf-border-gray-300 wpuf-bg-white wpuf-text-gray-500 hover:wpuf-border-gray-400'
                                        }`}
                                    >
                                        <input
                                            type="radio"
                                            name="avatar_size"
                                            value={size}
                                            checked={formData.avatar_size === size}
                                            onChange={handleChange}
                                            className="wpuf-sr-only"
                                        />
                                        {formData.avatar_size === size && (
                                            <span className="wpuf-absolute wpuf-top-2 wpuf-right-2">
                                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <rect width="20" height="20" rx="10" fill="#059669"/>
                                                    <path d="M7.5 9.5L9.5 11.5L13 8" stroke="white" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
                                                </svg>
                                            </span>
                                        )}
                                        {isRecommended && !isSelected && (
                                            <span className="wpuf-absolute wpuf-top-1 wpuf-right-1">
                                                <div className="wpuf-bg-green-500 wpuf-text-white wpuf-text-[10px] wpuf-px-1 wpuf-py-0.5 wpuf-rounded wpuf-font-medium">
                                                    {__('Rec', 'wpuf-pro')}
                                                </div>
                                            </span>
                                        )}
                                        <svg width="17" height="18" viewBox="0 0 17 18" fill="none" xmlns="http://www.w3.org/2000/svg" className="wpuf-mb-1">
                                            <path 
                                                d="M13.5676 14.604C12.4264 13.0979 10.6183 12.125 8.58301 12.125C6.54768 12.125 4.73958 13.0979 3.59839 14.604M13.5676 14.604C15.1109 13.2303 16.083 11.2287 16.083 9C16.083 4.85786 12.7251 1.5 8.58301 1.5C4.44087 1.5 1.08301 4.85786 1.08301 9C1.08301 11.2287 2.05509 13.2303 3.59839 14.604M13.5676 14.604C12.2426 15.7834 10.4965 16.5 8.58301 16.5C6.66954 16.5 4.92343 15.7834 3.59839 14.604M11.083 7.125C11.083 8.50571 9.96372 9.625 8.58301 9.625C7.2023 9.625 6.08301 8.50571 6.08301 7.125C6.08301 5.74429 7.2023 4.625 8.58301 4.625C9.96372 4.625 11.083 5.74429 11.083 7.125Z" 
                                                stroke={isSelected ? "#059669" : isRecommended ? "#2563eb" : "#9CA3AF"} 
                                                strokeWidth="1.5" 
                                                strokeLinecap="round" 
                                                strokeLinejoin="round"
                                                style={{
                                                    transition: 'stroke 0.2s ease-in-out'
                                                }}
                                                className={isSelected ? "" : isRecommended ? "" : "group-hover:green-600"}
                                            />
                                        </svg>
                                        <div className="wpuf-text-xs wpuf-text-center wpuf-leading-tight">
                                            {sizeLabel}
                                            {isRecommended && !isSelected && (
                                                <div className="wpuf-text-[10px] wpuf-text-green-600 wpuf-font-medium wpuf-mt-0.5">
                                                    {__('Recommended', 'wpuf-pro')}
                                                </div>
                                            )}
                                        </div>
                                    </label>
                                );
                            })}
                        </div>
                    </div>
                </div>
            </div>
    );
};

export default AdvancedControl;
